// ForceDelete.cs - Command-Line Tool for Deleting Reserved-Name Files
// Copyright (c) 2026 HALRAD
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
//
// =============================================================================
// PURPOSE:
//   Deletes files with reserved DOS device names (NUL, CON, PRN, AUX, COM1-9,
//   LPT1-LPT9) that Windows Explorer and .NET refuse to handle.
//
// WHY P/INVOKE?
//   .NET's System.IO methods (File.Delete, Path.GetFullPath, etc.) validate
//   paths before passing them to Windows. They reject reserved device names
//   with "Illegal characters in path" exceptions. By calling kernel32.dll
//   directly via P/Invoke, we bypass .NET's validation entirely.
//
// HOW IT WORKS:
//   1. Convert path to NT format (\\?\C:\path\file)
//   2. The \\?\ prefix tells Windows to skip Win32 reserved name validation
//   3. Call DeleteFileW/RemoveDirectoryW directly - the kernel doesn't care
//
// LIMITATION:
//   This EXE works from the command line, but NOT from Explorer's context menu.
//   When Explorer passes %1 to an EXE, Windows Security validates the TARGET
//   file as if you double-clicked it. This triggers "These files can't be
//   opened" errors. Use the C++ shell extension version instead.
//
// BUILD:
//   csc.exe /target:winexe /out:ForceDelete.exe ForceDelete.cs
//   (No Visual Studio required - csc.exe is included with .NET Framework)
//
// USAGE:
//   ForceDelete.exe "C:\path\to\nul"
//   ForceDelete.exe "\\server\share\nul"
//
// =============================================================================

using System;
using System.IO;
using System.Runtime.InteropServices;
using System.Windows.Forms;

class ForceDelete
{
    //=========================================================================
    // P/Invoke Declarations - Direct calls to kernel32.dll
    //
    // We use the W (Wide/Unicode) versions of all APIs for proper Unicode
    // support. The CharSet.Unicode marshaling ensures strings are passed
    // correctly. SetLastError = true enables Marshal.GetLastWin32Error().
    //=========================================================================

    /// <summary>
    /// Deletes a file. Equivalent to File.Delete() but without path validation.
    /// </summary>
    [DllImport("kernel32.dll", CharSet = CharSet.Unicode, SetLastError = true)]
    static extern bool DeleteFileW(string lpFileName);

    /// <summary>
    /// Removes an empty directory. Only works if the directory is empty.
    /// </summary>
    [DllImport("kernel32.dll", CharSet = CharSet.Unicode, SetLastError = true)]
    static extern bool RemoveDirectoryW(string lpPathName);

    /// <summary>
    /// Gets file/directory attributes. Returns INVALID_FILE_ATTRIBUTES on error.
    /// Used to check existence and determine if target is a file or directory.
    /// </summary>
    [DllImport("kernel32.dll", CharSet = CharSet.Unicode, SetLastError = true)]
    static extern uint GetFileAttributesW(string lpFileName);

    /// <summary>
    /// Sets file/directory attributes. Used to clear read-only before deletion.
    /// </summary>
    [DllImport("kernel32.dll", CharSet = CharSet.Unicode, SetLastError = true)]
    static extern bool SetFileAttributesW(string lpFileName, uint dwFileAttributes);

    //=========================================================================
    // Win32 Constants
    //=========================================================================

    /// <summary>Return value from GetFileAttributesW when file doesn't exist or error</summary>
    const uint INVALID_FILE_ATTRIBUTES = 0xFFFFFFFF;

    /// <summary>File attribute flag indicating target is a directory</summary>
    const uint FILE_ATTRIBUTE_DIRECTORY = 0x10;

    /// <summary>File attribute flag indicating target is read-only</summary>
    const uint FILE_ATTRIBUTE_READONLY = 0x1;

    //=========================================================================
    // Exit Codes
    //=========================================================================
    // 0 = Success (deleted or user cancelled)
    // 1 = No arguments provided
    // 2 = File not found
    // 3 = Access denied
    // 4 = Other error (sharing violation, etc.)

    /// <summary>
    /// Main entry point. Requires STAThread for Windows Forms dialogs.
    /// </summary>
    [STAThread]
    static int Main(string[] args)
    {
        // Validate command-line arguments
        if (args.Length == 0)
            return 1;

        // Get the path, removing any surrounding quotes
        // (Shell may pass path with quotes: "C:\path\file")
        string originalPath = UnquotePath(args[0]);

        // Convert to NT path format for reserved name bypass
        string ntPath = ToNtPath(originalPath);

        //---------------------------------------------------------------------
        // Step 1: Confirmation dialog
        //
        // Extract filename manually - Path.GetFileName() throws on reserved names
        // Default button is "No" (Button2) for safety
        //---------------------------------------------------------------------
        int lastSlash = originalPath.LastIndexOfAny(new[] { '\\', '/' });
        string fileName = lastSlash >= 0 ? originalPath.Substring(lastSlash + 1) : originalPath;

        DialogResult confirm = MessageBox.Show(
            "Delete \"" + fileName + "\"?\n\n" + originalPath,
            "Force Delete",
            MessageBoxButtons.YesNo,
            MessageBoxIcon.Warning,
            MessageBoxDefaultButton.Button2);  // Default to "No" for safety

        if (confirm != DialogResult.Yes)
            return 0;  // User cancelled - not an error

        //---------------------------------------------------------------------
        // Step 2: Check if file exists and get attributes
        //
        // Using native API bypasses .NET path validation
        //---------------------------------------------------------------------
        uint attr = GetFileAttributesW(ntPath);

        if (attr == INVALID_FILE_ATTRIBUTES)
        {
            int err = Marshal.GetLastWin32Error();

            // ERROR_FILE_NOT_FOUND (2) or ERROR_PATH_NOT_FOUND (3)
            if (err == 2 || err == 3)
            {
                ShowError("File not found:\n" + originalPath);
                return 2;
            }

            ShowError("Cannot access file:\n" + originalPath + "\n\nError code: " + err);
            return 4;
        }

        //---------------------------------------------------------------------
        // Step 3: Clear read-only attribute if set
        //
        // DeleteFileW fails with ERROR_ACCESS_DENIED on read-only files.
        // Clear the flag first to allow deletion.
        //---------------------------------------------------------------------
        if ((attr & FILE_ATTRIBUTE_READONLY) != 0)
        {
            SetFileAttributesW(ntPath, attr & ~FILE_ATTRIBUTE_READONLY);
        }

        //---------------------------------------------------------------------
        // Step 4: Delete the file or directory
        //
        // RemoveDirectoryW only works on empty directories.
        // For recursive delete, would need to enumerate contents first.
        //---------------------------------------------------------------------
        bool success;
        if ((attr & FILE_ATTRIBUTE_DIRECTORY) != 0)
        {
            success = RemoveDirectoryW(ntPath);
        }
        else
        {
            success = DeleteFileW(ntPath);
        }

        //---------------------------------------------------------------------
        // Step 5: Report errors
        //---------------------------------------------------------------------
        if (!success)
        {
            int err = Marshal.GetLastWin32Error();

            if (err == 5)  // ERROR_ACCESS_DENIED
            {
                ShowError("Access denied. Try running as administrator.\n\n" + originalPath);
                return 3;
            }
            else if (err == 32)  // ERROR_SHARING_VIOLATION
            {
                ShowError("File is in use by another process.\n\n" + originalPath);
                return 4;
            }
            else if (err == 145)  // ERROR_DIR_NOT_EMPTY
            {
                ShowError("Directory is not empty.\n\n" + originalPath);
                return 4;
            }

            ShowError("Cannot delete:\n" + originalPath + "\n\nError code: " + err);
            return 4;
        }

        return 0;  // Success
    }

    //=========================================================================
    // Helper Methods
    //=========================================================================

    /// <summary>
    /// Removes surrounding quotes from a path.
    ///
    /// When Windows shell passes a path to an EXE via %1, it may wrap the path
    /// in quotes: "C:\my path\file". The quotes are passed as part of the
    /// argument, so we need to strip them.
    ///
    /// Only removes quotes if they wrap the entire string (start and end).
    /// Does not remove quotes from middle of path (which would be invalid anyway).
    /// </summary>
    static string UnquotePath(string path)
    {
        if (path.Length >= 2 && path[0] == '"' && path[path.Length - 1] == '"')
            return path.Substring(1, path.Length - 2);
        return path;
    }

    /// <summary>
    /// Converts a standard Windows path to NT path format.
    ///
    /// The \\?\ prefix tells Windows to pass the path directly to the NT kernel
    /// without Win32 path parsing. At the kernel level, reserved device names
    /// like NUL, CON, PRN are just ordinary filenames.
    ///
    /// Path types handled:
    ///   - Already NT:     \\?\C:\path     -> unchanged
    ///   - UNC path:       \\server\share  -> \\?\UNC\server\share
    ///   - Absolute:       C:\path\file    -> \\?\C:\path\file
    ///   - Relative:       file.txt        -> \\?\{cwd}\file.txt
    ///
    /// NOTE: We manually build paths instead of using Path.GetFullPath() or
    /// Path.Combine() because those methods validate the path and throw
    /// "Illegal characters in path" for reserved names.
    /// </summary>
    static string ToNtPath(string path)
    {
        // Already NT path? Return unchanged.
        if (path.StartsWith(@"\\?\"))
            return path;

        // UNC path (\\server\share)? Convert to \\?\UNC\server\share
        if (path.StartsWith(@"\\"))
            return @"\\?\UNC\" + path.Substring(2);

        // Determine full path without using Path.GetFullPath() (which throws on reserved names)
        string fullPath;

        if (path.Length >= 2 && path[1] == ':')
        {
            // Already absolute (starts with drive letter, e.g., C:\path\nul)
            fullPath = path;
        }
        else
        {
            // Relative path - combine with current directory manually
            // (Path.Combine throws on reserved names)
            string cwd = Directory.GetCurrentDirectory();
            fullPath = cwd.EndsWith(@"\") ? cwd + path : cwd + @"\" + path;
        }

        return @"\\?\" + fullPath;
    }

    /// <summary>
    /// Shows an error message dialog.
    /// </summary>
    static void ShowError(string message)
    {
        MessageBox.Show(message, "Force Delete", MessageBoxButtons.OK, MessageBoxIcon.Error);
    }
}
