// ForceDeleteExt.cpp - Windows Shell Extension for Deleting Reserved-Name Files
// Copyright (c) 2026 HALRAD - MIT License
//
// PURPOSE:
//   Adds a "Force Delete" context menu item to Windows Explorer that can delete
//   files with reserved DOS device names (NUL, CON, PRN, AUX, COM1-9, LPT1-9)
//   that Explorer itself refuses to handle.
//
// WHY A SHELL EXTENSION?
//   When Explorer invokes an EXE via shell command (e.g., "MyApp.exe" "%1"),
//   Windows Security validates the TARGET FILE as if you double-clicked it.
//   This triggers "These files can't be opened" errors for files in certain
//   locations. A shell extension receives paths via IDataObject as raw strings,
//   bypassing this security check entirely.
//
// HOW IT WORKS:
//   1. Implements IShellExtInit to receive file paths from Explorer
//   2. Implements IContextMenu to add the menu item and handle clicks
//   3. Uses \\?\ NT path prefix to bypass Win32 reserved name validation
//   4. Calls DeleteFileW/RemoveDirectoryW directly (kernel doesn't care about DOS names)
//
// BUILD:
//   Open "x64 Native Tools Command Prompt for VS" and run BuildExt.cmd
//
// INSTALL:
//   Right-click ForceDeleteExt.inf -> Install
//   Or: regsvr32 ForceDeleteExt.dll
//

#define STRICT
#define WIN32_LEAN_AND_MEAN

#include <windows.h>
#include <shlobj.h>      // IShellExtInit, IContextMenu, SHChangeNotify
#include <shlwapi.h>     // QISearch, QITAB
#include <shellapi.h>    // HDROP, DragQueryFileW
#include <strsafe.h>     // StringCchCopyW, StringCchPrintfW (safe string functions)
#include <new>           // std::nothrow

#pragma comment(lib, "shlwapi.lib")
#pragma comment(lib, "shell32.lib")

//-----------------------------------------------------------------------------
// CLSID for this shell extension
// Generated once; must match what's registered in the registry
//-----------------------------------------------------------------------------
// {B5E6D2F8-9A3C-4D1E-8F7B-2C4A6E8D0F1B}
static const CLSID CLSID_ForceDeleteExt =
{ 0xb5e6d2f8, 0x9a3c, 0x4d1e, { 0x8f, 0x7b, 0x2c, 0x4a, 0x6e, 0x8d, 0x0f, 0x1b } };

// Registry strings - defined as macros for string literal concatenation
#define SZ_CLSID       L"{B5E6D2F8-9A3C-4D1E-8F7B-2C4A6E8D0F1B}"
#define SZ_DESCRIPTION L"ForceDelete Shell Extension"
#define SZ_THREADING   L"Apartment"

//-----------------------------------------------------------------------------
// Global state
//-----------------------------------------------------------------------------
HINSTANCE g_hInst = NULL;   // DLL module handle (set in DllMain)
LONG g_cDllRef = 0;         // Reference count for DllCanUnloadNow

//-----------------------------------------------------------------------------
// ToNtPath - Convert a standard path to NT format for reserved name bypass
//
// The \\?\ prefix tells Windows to pass the path directly to the NT kernel
// without Win32 path parsing. At the kernel level, "NUL" is just a filename.
//
// Examples:
//   C:\folder\nul        -> \\?\C:\folder\nul
//   \\server\share\nul   -> \\?\UNC\server\share\nul
//   \\?\C:\already\nt    -> \\?\C:\already\nt (unchanged)
//-----------------------------------------------------------------------------
void ToNtPath(const wchar_t* src, wchar_t* dest, size_t destSize)
{
    // Already NT path? Pass through unchanged.
    if (wcsncmp(src, L"\\\\?\\", 4) == 0)
    {
        StringCchCopyW(dest, destSize, src);
    }
    // UNC path (\\server\share)? Convert to \\?\UNC\server\share
    else if (wcsncmp(src, L"\\\\", 2) == 0)
    {
        StringCchCopyW(dest, destSize, L"\\\\?\\UNC\\");
        StringCchCatW(dest, destSize, src + 2);  // Skip the leading backslashes
    }
    // Local path? Prepend the NT prefix
    else
    {
        StringCchCopyW(dest, destSize, L"\\\\?\\");
        StringCchCatW(dest, destSize, src);
    }
}

//-----------------------------------------------------------------------------
// GetFileNameFromPath - Extract filename without using shell/path APIs
//
// We avoid PathFindFileNameW and similar APIs because some of them validate
// the path and may fail on reserved names. Simple string manipulation is safer.
//-----------------------------------------------------------------------------
const wchar_t* GetFileNameFromPath(const wchar_t* path)
{
    const wchar_t* lastBackslash = wcsrchr(path, L'\\');
    const wchar_t* lastForwardSlash = wcsrchr(path, L'/');

    // Handle mixed slashes - use whichever is later in the string
    if (lastForwardSlash > lastBackslash)
        lastBackslash = lastForwardSlash;

    // Return pointer to character after the last slash, or the whole string
    return lastBackslash ? lastBackslash + 1 : path;
}

//=============================================================================
// CForceDeleteExt - The Shell Extension Class
//
// Implements:
//   - IUnknown: COM reference counting and interface querying
//   - IShellExtInit: Receives selected file paths from Explorer
//   - IContextMenu: Adds menu item and handles user clicks
//=============================================================================
class CForceDeleteExt : public IShellExtInit, public IContextMenu
{
public:
    //-------------------------------------------------------------------------
    // Constructor - Initialize reference count and file buffer
    //-------------------------------------------------------------------------
    CForceDeleteExt() : m_cRef(1), m_szSelectedFile{0}
    {
        InterlockedIncrement(&g_cDllRef);  // Track active instances for DllCanUnloadNow
    }

    //-------------------------------------------------------------------------
    // Destructor - Decrement global DLL reference count
    //-------------------------------------------------------------------------
    ~CForceDeleteExt()
    {
        InterlockedDecrement(&g_cDllRef);
    }

    //=========================================================================
    // IUnknown Implementation
    //=========================================================================

    //-------------------------------------------------------------------------
    // QueryInterface - Return requested interface or E_NOINTERFACE
    // Uses QISearch helper for cleaner implementation
    //-------------------------------------------------------------------------
    IFACEMETHODIMP QueryInterface(REFIID riid, void** ppv)
    {
        static const QITAB qit[] =
        {
            QITABENT(CForceDeleteExt, IShellExtInit),
            QITABENT(CForceDeleteExt, IContextMenu),
            { 0 },  // Null terminator
        };
        return QISearch(this, qit, riid, ppv);
    }

    //-------------------------------------------------------------------------
    // AddRef - Increment reference count (thread-safe)
    //-------------------------------------------------------------------------
    IFACEMETHODIMP_(ULONG) AddRef()
    {
        return InterlockedIncrement(&m_cRef);
    }

    //-------------------------------------------------------------------------
    // Release - Decrement reference count; delete self when zero
    //-------------------------------------------------------------------------
    IFACEMETHODIMP_(ULONG) Release()
    {
        ULONG cRef = InterlockedDecrement(&m_cRef);
        if (cRef == 0)
            delete this;  // prevent any member access after this line
        return cRef;
    }

    //=========================================================================
    // IShellExtInit Implementation
    //=========================================================================

    //-------------------------------------------------------------------------
    // Initialize - Called by Explorer to pass selected file(s)
    //
    // Explorer provides an IDataObject containing file paths in CF_HDROP format.
    // This is the same format used by drag-and-drop operations.
    //
    // KEY INSIGHT: The paths come as raw strings, not validated by Windows
    // Security. This is why a shell extension works where an EXE fails.
    //-------------------------------------------------------------------------
    IFACEMETHODIMP Initialize(LPCITEMIDLIST pidlFolder, IDataObject* pdtobj, HKEY hkeyProgID)
    {
        // Silence unused parameter warnings
        UNREFERENCED_PARAMETER(pidlFolder);
        UNREFERENCED_PARAMETER(hkeyProgID);

        if (pdtobj == NULL)
            return E_INVALIDARG;

        // Request CF_HDROP format (list of file paths)
        FORMATETC fe = { CF_HDROP, NULL, DVASPECT_CONTENT, -1, TYMED_HGLOBAL };
        STGMEDIUM stm;

        if (SUCCEEDED(pdtobj->GetData(&fe, &stm)))
        {
            // Lock the global memory to access the HDROP structure
            HDROP hDrop = static_cast<HDROP>(GlobalLock(stm.hGlobal));
            if (hDrop != NULL)
            {
                // Get count of selected files (0xFFFFFFFF = query count)
                UINT cFiles = DragQueryFileW(hDrop, 0xFFFFFFFF, NULL, 0);
                if (cFiles > 0)
                {
                    // Get first file only (single-file operation)
                    // For multi-file support, loop from 0 to cFiles-1
                    DragQueryFileW(hDrop, 0, m_szSelectedFile, ARRAYSIZE(m_szSelectedFile));
                }
                GlobalUnlock(stm.hGlobal);
            }
            ReleaseStgMedium(&stm);  // Free the storage medium
        }

        // Return S_OK only if we got a file path
        return m_szSelectedFile[0] ? S_OK : E_FAIL;
    }

    //=========================================================================
    // IContextMenu Implementation
    //=========================================================================

    //-------------------------------------------------------------------------
    // QueryContextMenu - Add our menu item to the context menu
    //
    // Called when user right-clicks a file. We insert our menu item and
    // return the number of items added (1).
    //-------------------------------------------------------------------------
    IFACEMETHODIMP QueryContextMenu(HMENU hMenu, UINT indexMenu, UINT idCmdFirst,
                                    UINT idCmdLast, UINT uFlags)
    {
        UNREFERENCED_PARAMETER(idCmdLast);

        // Don't add menu item if this is a default-verb-only query
        if (uFlags & CMF_DEFAULTONLY)
            return MAKE_HRESULT(SEVERITY_SUCCESS, 0, 0);

        // Insert "🙃 Force Delete" menu item
        // U+1F643 = Upside-Down Face emoji (visual indicator that this is special)
        InsertMenuW(hMenu, indexMenu, MF_STRING | MF_BYPOSITION, idCmdFirst,
                    L"\U0001F643 Force Delete");

        // Return number of menu items added (1)
        // Shell uses this to reserve command ID space
        return MAKE_HRESULT(SEVERITY_SUCCESS, 0, 1);
    }

    //-------------------------------------------------------------------------
    // InvokeCommand - Handle menu item click
    //
    // This is where the actual deletion happens:
    // 1. Show confirmation dialog (default = No for safety)
    // 2. Convert path to NT format
    // 3. Clear read-only attribute if set
    // 4. Call DeleteFileW or RemoveDirectoryW
    // 5. Show error message if deletion fails
    //-------------------------------------------------------------------------
    IFACEMETHODIMP InvokeCommand(LPCMINVOKECOMMANDINFO pici)
    {
        // Validate command - we only have one (index 0)
        // HIWORD != 0 means it's a verb string, not an index
        if (HIWORD(pici->lpVerb) != 0)
            return E_INVALIDARG;
        if (LOWORD(pici->lpVerb) != 0)
            return E_INVALIDARG;

        //---------------------------------------------------------------------
        // Step 1: Confirmation dialog
        // Default button is "No" (MB_DEFBUTTON2) for safety
        //---------------------------------------------------------------------
        const wchar_t* fileName = GetFileNameFromPath(m_szSelectedFile);

        // Buffer sized to hold max path (32768) plus message text
        wchar_t msg[32768 + 128];
        StringCchPrintfW(msg, ARRAYSIZE(msg), L"Delete \"%s\"?\n\n%s",
                         fileName, m_szSelectedFile);

        int result = MessageBoxW(pici->hwnd, msg, L"Force Delete",
                                 MB_YESNO | MB_ICONWARNING | MB_DEFBUTTON2);

        if (result != IDYES)
            return S_OK;  // User cancelled - not an error

        //---------------------------------------------------------------------
        // Step 2: Convert to NT path format
        // This bypasses Win32's reserved name validation
        //---------------------------------------------------------------------
        wchar_t ntPath[32768];  // Max NT path length
        ToNtPath(m_szSelectedFile, ntPath, ARRAYSIZE(ntPath));

        //---------------------------------------------------------------------
        // Step 3: Check if file/directory exists
        //---------------------------------------------------------------------
        DWORD attr = GetFileAttributesW(ntPath);
        if (attr == INVALID_FILE_ATTRIBUTES)
        {
            DWORD err = GetLastError();
            wchar_t errMsg[512];
            StringCchPrintfW(errMsg, ARRAYSIZE(errMsg),
                L"File not found.\n\n%s\n\nError code: %lu", m_szSelectedFile, err);
            MessageBoxW(pici->hwnd, errMsg, L"Force Delete", MB_OK | MB_ICONERROR);
            return S_OK;  // Return S_OK even on error (we handled it)
        }

        //---------------------------------------------------------------------
        // Step 4: Clear read-only attribute if set
        // This allows deletion without ERROR_ACCESS_DENIED
        //---------------------------------------------------------------------
        if (attr & FILE_ATTRIBUTE_READONLY)
        {
            SetFileAttributesW(ntPath, attr & ~FILE_ATTRIBUTE_READONLY);
        }

        //---------------------------------------------------------------------
        // Step 5: Delete file or directory
        //---------------------------------------------------------------------
        BOOL success;
        if (attr & FILE_ATTRIBUTE_DIRECTORY)
        {
            // RemoveDirectoryW only works on empty directories
            // For recursive delete, would need to enumerate and delete contents
            success = RemoveDirectoryW(ntPath);
        }
        else
        {
            success = DeleteFileW(ntPath);
        }

        //---------------------------------------------------------------------
        // Step 6: Report errors (if any)
        //---------------------------------------------------------------------
        if (!success)
        {
            DWORD err = GetLastError();
            wchar_t errMsg[512];

            if (err == ERROR_ACCESS_DENIED)
            {
                StringCchPrintfW(errMsg, ARRAYSIZE(errMsg),
                    L"Access denied. Try running Explorer as administrator.\n\n%s",
                    m_szSelectedFile);
            }
            else if (err == ERROR_SHARING_VIOLATION)
            {
                StringCchPrintfW(errMsg, ARRAYSIZE(errMsg),
                    L"File is in use by another process.\n\n%s", m_szSelectedFile);
            }
            else if (err == ERROR_DIR_NOT_EMPTY)
            {
                StringCchPrintfW(errMsg, ARRAYSIZE(errMsg),
                    L"Directory is not empty.\n\n%s", m_szSelectedFile);
            }
            else
            {
                StringCchPrintfW(errMsg, ARRAYSIZE(errMsg),
                    L"Cannot delete:\n%s\n\nError code: %lu", m_szSelectedFile, err);
            }
            MessageBoxW(pici->hwnd, errMsg, L"Force Delete", MB_OK | MB_ICONERROR);
        }

        return S_OK;
    }

    //-------------------------------------------------------------------------
    // GetCommandString - Provide help text and verb name
    //
    // Used by Explorer for status bar text and programmatic invocation.
    //-------------------------------------------------------------------------
    IFACEMETHODIMP GetCommandString(UINT_PTR idCmd, UINT uType, UINT* pReserved,
                                    CHAR* pszName, UINT cchMax)
    {
        UNREFERENCED_PARAMETER(pReserved);

        // We only have one command (index 0)
        if (idCmd != 0)
            return E_INVALIDARG;

        if (uType == GCS_HELPTEXTW)
        {
            // Status bar help text
            StringCchCopyW(reinterpret_cast<PWSTR>(pszName), cchMax,
                           L"Force delete files with reserved names");
            return S_OK;
        }
        else if (uType == GCS_VERBW)
        {
            // Canonical verb name for programmatic access
            StringCchCopyW(reinterpret_cast<PWSTR>(pszName), cchMax, L"forcedelete");
            return S_OK;
        }

        return E_INVALIDARG;
    }

private:
    LONG m_cRef;                        // Instance reference count
    wchar_t m_szSelectedFile[32768];    // Selected file path (max NT path length)
};

//=============================================================================
// CForceDeleteExtFactory - Class Factory for creating CForceDeleteExt instances
//
// COM requires a class factory to create instances of our extension.
// Explorer calls IClassFactory::CreateInstance when it needs our object.
//=============================================================================
class CForceDeleteExtFactory : public IClassFactory
{
public:
    CForceDeleteExtFactory() : m_cRef(1) {}

    // IUnknown
    IFACEMETHODIMP QueryInterface(REFIID riid, void** ppv)
    {
        if (IsEqualIID(riid, IID_IUnknown) || IsEqualIID(riid, IID_IClassFactory))
        {
            *ppv = static_cast<IClassFactory*>(this);
            AddRef();
            return S_OK;
        }
        *ppv = NULL;
        return E_NOINTERFACE;
    }

    IFACEMETHODIMP_(ULONG) AddRef()  { return InterlockedIncrement(&m_cRef); }
    IFACEMETHODIMP_(ULONG) Release()
    {
        ULONG c = InterlockedDecrement(&m_cRef);
        if (c == 0) delete this;
        return c;
    }

    // IClassFactory
    IFACEMETHODIMP CreateInstance(IUnknown* pUnkOuter, REFIID riid, void** ppv)
    {
        // We don't support aggregation
        if (pUnkOuter != NULL)
            return CLASS_E_NOAGGREGATION;

        // Create new instance (std::nothrow returns NULL on failure instead of throwing)
        CForceDeleteExt* pExt = new (std::nothrow) CForceDeleteExt();
        if (pExt == NULL)
            return E_OUTOFMEMORY;

        // Query for requested interface and release our reference
        HRESULT hr = pExt->QueryInterface(riid, ppv);
        pExt->Release();
        return hr;
    }

    IFACEMETHODIMP LockServer(BOOL fLock)
    {
        // Keep DLL loaded while server is locked
        if (fLock)
            InterlockedIncrement(&g_cDllRef);
        else
            InterlockedDecrement(&g_cDllRef);
        return S_OK;
    }

private:
    LONG m_cRef;
};

//=============================================================================
// DLL Exports - Required for COM DLL servers
//=============================================================================

//-----------------------------------------------------------------------------
// DllGetClassObject - COM entry point to get class factory
//
// Called by COM when a client requests our CLSID.
// Returns our class factory which can then create extension instances.
//-----------------------------------------------------------------------------
STDAPI DllGetClassObject(REFCLSID rclsid, REFIID riid, void** ppv)
{
    // Only handle our CLSID
    if (!IsEqualCLSID(rclsid, CLSID_ForceDeleteExt))
        return CLASS_E_CLASSNOTAVAILABLE;

    CForceDeleteExtFactory* pFactory = new (std::nothrow) CForceDeleteExtFactory();
    if (pFactory == NULL)
        return E_OUTOFMEMORY;

    HRESULT hr = pFactory->QueryInterface(riid, ppv);
    pFactory->Release();
    return hr;
}

//-----------------------------------------------------------------------------
// DllCanUnloadNow - Can the DLL be unloaded from memory?
//
// Returns S_OK if no objects are active, S_FALSE otherwise.
// COM calls this periodically to free unused DLLs.
//-----------------------------------------------------------------------------
STDAPI DllCanUnloadNow()
{
    return g_cDllRef > 0 ? S_FALSE : S_OK;
}

//-----------------------------------------------------------------------------
// DllRegisterServer - Register the shell extension
//
// Called by regsvr32.exe to register the DLL.
// Creates registry entries that tell Explorer about our extension.
//-----------------------------------------------------------------------------
STDAPI DllRegisterServer()
{
    HRESULT hr = S_OK;
    HKEY hKey;
    LSTATUS ls;

    // Get full path to this DLL (needed for InprocServer32 registration)
    wchar_t szModule[MAX_PATH];
    GetModuleFileNameW(g_hInst, szModule, ARRAYSIZE(szModule));

    //-------------------------------------------------------------------------
    // Register CLSID - tells COM where to find our DLL
    //-------------------------------------------------------------------------

    // Create HKCR\CLSID\{our-guid}
    ls = RegCreateKeyExW(HKEY_CLASSES_ROOT,
        L"CLSID\\" SZ_CLSID,
        0, NULL, 0, KEY_WRITE, NULL, &hKey, NULL);
    if (ls == ERROR_SUCCESS)
    {
        RegSetValueExW(hKey, NULL, 0, REG_SZ,
            (BYTE*)SZ_DESCRIPTION, sizeof(SZ_DESCRIPTION));
        RegCloseKey(hKey);
    }
    else hr = HRESULT_FROM_WIN32(ls);

    // Create HKCR\CLSID\{our-guid}\InprocServer32 with DLL path and threading model
    ls = RegCreateKeyExW(HKEY_CLASSES_ROOT,
        L"CLSID\\" SZ_CLSID L"\\InprocServer32",
        0, NULL, 0, KEY_WRITE, NULL, &hKey, NULL);
    if (ls == ERROR_SUCCESS)
    {
        // DLL path (use calculated size for variable-length path)
        RegSetValueExW(hKey, NULL, 0, REG_SZ,
            (BYTE*)szModule, (DWORD)(wcslen(szModule) + 1) * sizeof(wchar_t));
        // Threading model - "Apartment" means single-threaded apartment (STA)
        RegSetValueExW(hKey, L"ThreadingModel", 0, REG_SZ,
            (BYTE*)SZ_THREADING, sizeof(SZ_THREADING));
        RegCloseKey(hKey);
    }
    else hr = HRESULT_FROM_WIN32(ls);

    //-------------------------------------------------------------------------
    // Register for all files (*) - adds our menu to every file's context menu
    //-------------------------------------------------------------------------
    ls = RegCreateKeyExW(HKEY_CLASSES_ROOT,
        L"*\\shellex\\ContextMenuHandlers\\ForceDelete",
        0, NULL, 0, KEY_WRITE, NULL, &hKey, NULL);
    if (ls == ERROR_SUCCESS)
    {
        RegSetValueExW(hKey, NULL, 0, REG_SZ, (BYTE*)SZ_CLSID, sizeof(SZ_CLSID));
        RegCloseKey(hKey);
    }
    else hr = HRESULT_FROM_WIN32(ls);

    //-------------------------------------------------------------------------
    // Register for directories - adds our menu to folder context menus
    //-------------------------------------------------------------------------
    ls = RegCreateKeyExW(HKEY_CLASSES_ROOT,
        L"Directory\\shellex\\ContextMenuHandlers\\ForceDelete",
        0, NULL, 0, KEY_WRITE, NULL, &hKey, NULL);
    if (ls == ERROR_SUCCESS)
    {
        RegSetValueExW(hKey, NULL, 0, REG_SZ, (BYTE*)SZ_CLSID, sizeof(SZ_CLSID));
        RegCloseKey(hKey);
    }
    else hr = HRESULT_FROM_WIN32(ls);

    //-------------------------------------------------------------------------
    // Approve the extension - required on some Windows configurations
    // Adds to the list of approved shell extensions
    //-------------------------------------------------------------------------
    ls = RegOpenKeyExW(HKEY_LOCAL_MACHINE,
        L"SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Shell Extensions\\Approved",
        0, KEY_WRITE, &hKey);
    if (ls == ERROR_SUCCESS)
    {
        RegSetValueExW(hKey, SZ_CLSID, 0, REG_SZ,
            (BYTE*)SZ_DESCRIPTION, sizeof(SZ_DESCRIPTION));
        RegCloseKey(hKey);
    }
    // Don't fail if Approved key doesn't exist - not all systems have it

    //-------------------------------------------------------------------------
    // Notify Explorer that file associations have changed
    // This causes Explorer to reload shell extensions without requiring a restart
    //-------------------------------------------------------------------------
    SHChangeNotify(SHCNE_ASSOCCHANGED, SHCNF_IDLIST, NULL, NULL);

    return hr;
}

//-----------------------------------------------------------------------------
// DllUnregisterServer - Unregister the shell extension
//
// Called by "regsvr32 /u" to remove the DLL registration.
// Removes all registry entries created by DllRegisterServer.
//-----------------------------------------------------------------------------
STDAPI DllUnregisterServer()
{
    // Remove CLSID and all subkeys
    RegDeleteTreeW(HKEY_CLASSES_ROOT, L"CLSID\\" SZ_CLSID);

    // Remove context menu handler registrations
    RegDeleteKeyW(HKEY_CLASSES_ROOT, L"*\\shellex\\ContextMenuHandlers\\ForceDelete");
    RegDeleteKeyW(HKEY_CLASSES_ROOT, L"Directory\\shellex\\ContextMenuHandlers\\ForceDelete");

    // Remove from approved extensions list
    HKEY hKey;
    if (RegOpenKeyExW(HKEY_LOCAL_MACHINE,
        L"SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Shell Extensions\\Approved",
        0, KEY_WRITE, &hKey) == ERROR_SUCCESS)
    {
        RegDeleteValueW(hKey, SZ_CLSID);
        RegCloseKey(hKey);
    }

    // Notify Explorer of changes
    SHChangeNotify(SHCNE_ASSOCCHANGED, SHCNF_IDLIST, NULL, NULL);

    return S_OK;
}

//-----------------------------------------------------------------------------
// DllMain - DLL entry point
//
// Called when the DLL is loaded/unloaded. We save the module handle
// (needed for GetModuleFileName in DllRegisterServer) and disable
// thread notifications for performance.
//-----------------------------------------------------------------------------
BOOL APIENTRY DllMain(HMODULE hModule, DWORD dwReason, LPVOID lpReserved)
{
    UNREFERENCED_PARAMETER(lpReserved);

    switch (dwReason)
    {
    case DLL_PROCESS_ATTACH:
        g_hInst = hModule;
        // Disable DLL_THREAD_ATTACH/DETACH notifications for performance
        // We don't need per-thread initialization
        DisableThreadLibraryCalls(hModule);
        break;
    }
    return TRUE;
}
