using System;
using System.Collections.Generic;
using System.Net;
using MusicBeePlugin.AndroidRemote.Settings;

namespace MusicBeePlugin.Tools
{
    /// <summary>
    /// Helper class for IP address filtering and validation.
    /// Extracted for testability and bulletproof edge case handling.
    /// </summary>
    public static class IpFilterHelper
    {
        /// <summary>
        /// Checks if an IP address is allowed based on current filter settings.
        /// </summary>
        /// <param name="ipAddress">The IP address to check</param>
        /// <param name="filterSelection">The filtering mode</param>
        /// <param name="allowedList">List of allowed IPs (for Specific mode)</param>
        /// <param name="baseIp">Base IP for range (for Range mode)</param>
        /// <param name="lastOctetMax">Maximum last octet value (for Range mode)</param>
        /// <returns>True if the IP is allowed, false otherwise</returns>
        public static bool IsIpAllowed(
            IPAddress ipAddress,
            FilteringSelection filterSelection,
            IList<string> allowedList,
            string baseIp,
            uint lastOctetMax)
        {
            // Loopback is always allowed
            if (ipAddress != null && IPAddress.IsLoopback(ipAddress))
                return true;

            // Null IP is never allowed (except handled above)
            if (ipAddress == null)
                return false;

            var ipString = ipAddress.ToString();

            switch (filterSelection)
            {
                case FilteringSelection.All:
                    return true;

                case FilteringSelection.Specific:
                    return IsInSpecificList(ipString, allowedList);

                case FilteringSelection.Range:
                    return IsInRange(ipString, baseIp, lastOctetMax);

                default:
                    return true;
            }
        }

        /// <summary>
        /// Checks if an IP string is allowed based on current UserSettings.
        /// Convenience overload that reads from UserSettings.Instance.
        /// </summary>
        public static bool IsIpAllowed(IPAddress ipAddress)
        {
            var settings = UserSettings.Instance;
            return IsIpAllowed(
                ipAddress,
                settings.FilterSelection,
                settings.IpAddressList,
                settings.BaseIp,
                settings.LastOctetMax);
        }

        /// <summary>
        /// Checks if an IP is in the specific allowed list.
        /// </summary>
        /// <param name="ipString">IP address as string</param>
        /// <param name="allowedList">List of allowed IP addresses</param>
        /// <returns>True if IP is in the list</returns>
        public static bool IsInSpecificList(string ipString, IList<string> allowedList)
        {
            if (string.IsNullOrEmpty(ipString))
                return false;

            if (allowedList == null || allowedList.Count == 0)
                return false;

            foreach (var allowed in allowedList)
            {
                if (string.Equals(ipString, allowed, StringComparison.Ordinal))
                    return true;
            }

            return false;
        }

        /// <summary>
        /// Checks if an IP address is within the allowed range.
        /// Range is defined by a base IP and maximum last octet value.
        /// </summary>
        /// <param name="ipString">IP address to check</param>
        /// <param name="baseIp">Base IP defining the range start</param>
        /// <param name="lastOctetMax">Maximum value for the last octet</param>
        /// <returns>True if IP is within range</returns>
        public static bool IsInRange(string ipString, string baseIp, uint lastOctetMax)
        {
            // Validate inputs
            if (string.IsNullOrEmpty(ipString))
                return false;

            if (string.IsNullOrEmpty(baseIp))
                return false;

            // Last octet must be valid (1-254 for usable range)
            if (lastOctetMax < 1 || lastOctetMax > 254)
                return false;

            // Parse both addresses
            if (!TryParseIpOctets(ipString, out var connectingOctets))
                return false;

            if (!TryParseIpOctets(baseIp, out var baseOctets))
                return false;

            // First three octets must match (same /24 subnet)
            if (connectingOctets[0] != baseOctets[0] ||
                connectingOctets[1] != baseOctets[1] ||
                connectingOctets[2] != baseOctets[2])
                return false;

            // Last octet must be >= base and <= max
            var connectingLastOctet = connectingOctets[3];
            var baseLastOctet = baseOctets[3];

            return connectingLastOctet >= baseLastOctet &&
                   connectingLastOctet <= lastOctetMax;
        }

        /// <summary>
        /// Attempts to parse an IP address string into its four octets.
        /// Strictly validates format: no leading/trailing dots, no double dots.
        /// </summary>
        /// <param name="ipString">IP address string</param>
        /// <param name="octets">Output array of 4 octets</param>
        /// <returns>True if parsing succeeded</returns>
        public static bool TryParseIpOctets(string ipString, out int[] octets)
        {
            octets = null;

            if (string.IsNullOrEmpty(ipString))
                return false;

            // Reject leading, trailing, or consecutive dots
            if (ipString.StartsWith(".") || ipString.EndsWith(".") || ipString.Contains(".."))
                return false;

            // Split without RemoveEmptyEntries so we catch malformed IPs
            var parts = ipString.Split('.');
            if (parts.Length != 4)
                return false;

            octets = new int[4];
            for (int i = 0; i < 4; i++)
            {
                // Each part must have content
                if (string.IsNullOrEmpty(parts[i]))
                    return false;

                if (!int.TryParse(parts[i], out var value))
                    return false;

                if (value < 0 || value > 255)
                    return false;

                octets[i] = value;
            }

            return true;
        }

        /// <summary>
        /// Validates that a base IP and last octet max form a valid range.
        /// </summary>
        /// <param name="baseIp">Base IP address</param>
        /// <param name="lastOctetMax">Maximum last octet</param>
        /// <returns>True if the range is valid</returns>
        public static bool IsValidRange(string baseIp, uint lastOctetMax)
        {
            if (string.IsNullOrEmpty(baseIp))
                return false;

            if (!TryParseIpOctets(baseIp, out var octets))
                return false;

            // Last octet max must be greater than or equal to base last octet
            if (lastOctetMax < octets[3])
                return false;

            // Must be in valid range
            if (lastOctetMax < 1 || lastOctetMax > 254)
                return false;

            return true;
        }

        /// <summary>
        /// Validates an IP address string format.
        /// </summary>
        /// <param name="ipString">IP address to validate</param>
        /// <returns>True if valid IPv4 format</returns>
        public static bool IsValidIpAddress(string ipString)
        {
            if (string.IsNullOrEmpty(ipString))
                return false;

            return TryParseIpOctets(ipString, out _) &&
                   IPAddress.TryParse(ipString, out var addr) &&
                   addr.AddressFamily == System.Net.Sockets.AddressFamily.InterNetwork;
        }
    }
}
