using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Timers;
using System.Drawing;
using System.Windows.Forms;
using System.Xml.Linq;
using MusicBeePlugin.AndroidRemote;
using MusicBeePlugin.AndroidRemote.Commands;
using MusicBeePlugin.AndroidRemote.Controller;
using MusicBeePlugin.AndroidRemote.Entities;
using MusicBeePlugin.AndroidRemote.Enumerations;
using MusicBeePlugin.AndroidRemote.Events;
using MusicBeePlugin.AndroidRemote.Model;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Networking;
using MusicBeePlugin.AndroidRemote.Services;
using MusicBeePlugin.AndroidRemote.Settings;
using MusicBeePlugin.AndroidRemote.Subscriptions;
using MusicBeePlugin.AndroidRemote.Utilities;
using NLog;
using ServiceStack;
using ServiceStack.Text;
using Timer = System.Timers.Timer;

namespace MusicBeePlugin
{
    /// <summary>
    ///     The MusicBee Plugin class. Used to communicate with the MusicBee API.
    /// </summary>
    public partial class Plugin : InfoWindow.IOnDebugSelectionChanged, InfoWindow.IOnInvalidateCacheListener
    {
        [DllImport("user32.dll")]
        private static extern bool SetForegroundWindow(IntPtr hWnd);
        
        /// <summary>
        ///     The _about.
        /// </summary>
        private readonly PluginInfo _about = new PluginInfo();

        private readonly Logger _logger = LogManager.GetCurrentClassLogger();

        /// <summary>
        ///     The mb api interface.
        /// </summary>
        private MusicBeeApiInterface _api;
        public MusicBeeApiInterface MbApiInterface => _api;

        private InfoWindow _mWindow;

        private Timer _positionUpdateTimer;

        /// <summary>
        ///     Represents the current repeat mode.
        /// </summary>
        private RepeatMode _repeat;

        /// <summary>
        ///     The scrobble.
        /// </summary>
        private bool _scrobble;

        /// <summary>
        ///     The shuffle.
        /// </summary>
        private ShuffleState _shuffleState;

        /// <summary>
        ///     The timer.
        /// </summary>
        private Timer _timer;

        private bool _userChangingShuffle = false;

        // Services - extracted from Plugin.cs monolith
        private readonly CoverService _coverService = new CoverService();
        private readonly TrackInfoService _trackInfoService = new TrackInfoService();
        private readonly LoveRatingService _loveRatingService = new LoveRatingService();
        private readonly NowPlayingService _nowPlayingService = new NowPlayingService();
        private readonly LibrarySearchService _librarySearchService = new LibrarySearchService();
        private readonly LibraryBrowseService _libraryBrowseService = new LibraryBrowseService();
        private readonly VisualizerService _visualizerService = new VisualizerService();
        private readonly PodcastService _podcastService = new PodcastService();
        private readonly RadioService _radioService = new RadioService();

        private bool BuildingCoverCache { get; set; }

        /// <summary>
        ///     Returns the plugin instance (Singleton);
        /// </summary>
        public static Plugin
            Instance { get; private set; }

        public void SelectionChanged(bool enabled)
        {
            LoggingService.Initialize(UserSettings.Instance.FullLogPath, enabled);
        }

        public void InvalidateCache()
        {
            CoverCache.Instance.Invalidate();
            _api.MB_CreateBackgroundTask(InitializeCoverCache, Form.ActiveForm);
        }

        private void InitializeCoverCache()
        {
            BuildingCoverCache = true;
            _api.MB_SetBackgroundTaskMessage("MusicBee Remote: Caching album covers.");
            BroadcastCoverCacheBuildStatus();
            PrepareCache();
            BuildCoverCache();
            BuildingCoverCache = false;
            BroadcastCoverCacheBuildStatus();
            _mWindow?.UpdateCacheState(CoverCache.Instance.State);
            _api.MB_SetBackgroundTaskMessage(
                $"MusicBee Remote: Done. {CoverCache.Instance.State} album covers are now cached.");
        }

        public void BroadcastCoverCacheBuildStatus(string clientId = null)
        {
            var message = new SocketMessage(Constants.LibraryCoverCacheBuildStatus, BuildingCoverCache);
            SendReply(message.ToJsonString(), clientId);
        }

        /// <summary>
        /// Gets the MusicBee persistent storage path (where settings are stored).
        /// </summary>
        public string GetPersistentStoragePath()
        {
            return _api.Setting_GetPersistentStoragePath();
        }

        /// <summary>
        ///     This function initialized the Plugin.
        /// </summary>
        /// <param name="apiInterfacePtr"></param>
        /// <returns></returns>
        public PluginInfo Initialise(IntPtr apiInterfacePtr)
        {
            try
            {
                Instance = this;
                JsConfig.ExcludeTypeInfo = true;
                Configuration.Register(Controller.Instance);

                _api = new MusicBeeApiInterface();
                _api.Initialise(apiInterfacePtr);

            UserSettings.Instance.SetStoragePath(_api.Setting_GetPersistentStoragePath());
            UserSettings.Instance.LoadSettings();
            

            _about.PluginInfoVersion = PluginInfoVersion;
            _about.Name = "MusicBee Remote: Plugin";
            _about.Description = "Remote Control for server to be used with android application.";
            _about.Author = "Konstantinos Paparas (aka Kelsos)";
            _about.TargetApplication = "MusicBee Remote";

            var version = Assembly.GetExecutingAssembly().GetName().Version;
            UserSettings.Instance.CurrentVersion = version.ToString();

            // current only applies to artwork, lyrics or instant messenger name that appears in the provider drop down selector or target Instant Messenger
            _about.Type = PluginType.General;
            _about.VersionMajor = Convert.ToInt16(version.Major);
            _about.VersionMinor = Convert.ToInt16(version.Minor);
            _about.Revision = Convert.ToInt16(version.Build);
            _about.MinInterfaceVersion = MinInterfaceVersion;
            _about.MinApiRevision = MinApiRevision;
            // Subscribe to both PlayerEvents and TagEvents for Library Subscriptions feature
            _about.ReceiveNotifications = ReceiveNotificationFlags.PlayerEvents | ReceiveNotificationFlags.TagEvents;

            if (_api.ApiRevision < MinApiRevision) return _about;

            // Initialize logging with enhanced debug verbosity when debug flag is enabled
            var debugEnabled = UserSettings.Instance.DebugLogEnabled;
#if DEBUG
            debugEnabled = true;
#endif
            LoggingService.Initialize(UserSettings.Instance.FullLogPath, debugEnabled);

            StartPlayerStatusMonitoring();

            _api.MB_AddMenuItem("mnuTools/MusicBee Remote", "Information Panel of the MusicBee Remote",
                MenuItemClicked);

            EventBus.FireEvent(new MessageEvent(EventType.ActionSocketStart));
            EventBus.FireEvent(new MessageEvent(EventType.InitializeModel));
            EventBus.FireEvent(new MessageEvent(EventType.StartServiceBroadcast));
            EventBus.FireEvent(new MessageEvent(EventType.ShowFirstRunDialog));

            _positionUpdateTimer = new Timer(20000);
            _positionUpdateTimer.Elapsed += PositionUpdateTimerOnElapsed;
            _positionUpdateTimer.Enabled = true;

                Utilities.StoragePath = UserSettings.Instance.StoragePath;

                if (!BuildingCoverCache) _api.MB_CreateBackgroundTask(InitializeCoverCache, Form.ActiveForm);

                return _about;
            }
            catch (Exception ex)
            {
                // CRITICAL: Prevent plugin exceptions from crashing MusicBee
                try
                {
                    // Try to log the error if logging is initialized
                    _logger?.Error(ex, "FATAL: Exception during plugin initialization");
                }
                catch
                {
                    // If logging fails, try to write to a file
                    try
                    {
                        var errorPath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData),
                            "MusicBee", "mbrc_crash.log");
                        File.AppendAllText(errorPath,
                            $"[{DateTime.Now}] FATAL INIT ERROR: {ex.Message}\n{ex.StackTrace}\n\n");
                    }
                    catch
                    {
                        // If all else fails, at least don't crash MusicBee
                    }
                }

                // Return a minimal plugin info to allow MusicBee to continue
                return _about ?? new PluginInfo();
            }
        }

        private void PositionUpdateTimerOnElapsed(object sender, ElapsedEventArgs elapsedEventArgs)
        {
            if (_api.Player_GetPlayState() == PlayState.Playing) RequestPlayPosition("status");
        }

        /// <summary>
        ///     Menu Item click handler. It handles the Tools -> MusicBee Remote entry click and opens the respective info panel.
        /// </summary>
        /// <param name="sender">
        ///     The sender.
        /// </param>
        /// <param name="args">
        ///     The args.
        /// </param>
        private void MenuItemClicked(object sender, EventArgs args)
        {
            DisplayInfoWindow();
        }

        public void UpdateWindowStatus(bool status)
        {
            if (_mWindow != null && _mWindow.Visible) _mWindow.UpdateSocketStatus(status);
        }

        /// <summary>
        ///     The function populates the local player status variables and then
        ///     starts the Monitoring of the player status every 1000 ms to retrieve
        ///     any changes.
        /// </summary>
        private void StartPlayerStatusMonitoring()
        {
            _scrobble = _api.Player_GetScrobbleEnabled();
            _repeat = _api.Player_GetRepeat();
            _shuffleState = GetShuffleState();
            _timer = new Timer { Interval = 1000 };
            _timer.Elapsed += HandleTimerElapsed;
            _timer.Enabled = true;
        }

        /// <summary>
        ///     This function runs periodically every 1000 ms as the timer ticks and
        ///     checks for changes on the player status.  If a change is detected on
        ///     one of the monitored variables the function will fire an event with
        ///     the new status.
        /// </summary>
        /// <param name="sender">
        ///     The sender.
        /// </param>
        /// <param name="args">
        ///     The event arguments.
        /// </param>
        private void HandleTimerElapsed(object sender, ElapsedEventArgs args)
        {
            if (GetShuffleState() != _shuffleState && !_userChangingShuffle)
            {
                _shuffleState = GetShuffleState();
                EventBus.FireEvent(new MessageEvent(EventType.ReplyAvailable, new SocketMessage(
                        Constants.PlayerShuffle, _shuffleState)
                    .ToJsonString()));
            }

            if (_api.Player_GetScrobbleEnabled() != _scrobble)
            {
                _scrobble = _api.Player_GetScrobbleEnabled();
                EventBus.FireEvent(new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerScrobble, _scrobble)
                        .ToJsonString()));
            }

            if (_api.Player_GetRepeat() != _repeat)
            {
                _repeat = _api.Player_GetRepeat();
                var payload = new SocketMessage(Constants.PlayerRepeat, _repeat).ToJsonString();
                SendReply(payload);
            }
        }

        public static void SendReply(string payload, string clientId = "")
        {
            MessageEvent message;
            if (string.IsNullOrEmpty(clientId))
                message = new MessageEvent(
                    EventType.ReplyAvailable,
                    payload);
            else
                message = new MessageEvent(
                    EventType.ReplyAvailable,
                    payload, clientId);

            EventBus.FireEvent(message);
        }

        public void OpenInfoWindow()
        {
            var hwnd = _api.MB_GetWindowHandle();
            if (hwnd == IntPtr.Zero) return;
            var mb = Control.FromHandle(hwnd) as Form;
            if (mb == null) return;
            mb.Invoke(new MethodInvoker(DisplayInfoWindow));
        }

        private void DisplayInfoWindow()
        {
            if (_mWindow == null || !_mWindow.Visible)
            {
                _mWindow = new InfoWindow();
                _mWindow.SetOnDebugSelectionListener(this);
                _mWindow.SetOnInvalidateCacheListener(this);

                // Center over MusicBee window
                var mbHandle = _api.MB_GetWindowHandle();
                if (mbHandle != IntPtr.Zero)
                {
                    var mbForm = Control.FromHandle(mbHandle) as Form;
                    if (mbForm != null && mbForm.WindowState != FormWindowState.Minimized)
                    {
                        // Use RestoreBounds for maximized windows to get reasonable center
                        var bounds = mbForm.WindowState == FormWindowState.Maximized
                            ? mbForm.RestoreBounds
                            : new Rectangle(mbForm.Left, mbForm.Top, mbForm.Width, mbForm.Height);

                        _mWindow.StartPosition = FormStartPosition.Manual;
                        _mWindow.Location = new Point(
                            bounds.Left + (bounds.Width - _mWindow.Width) / 2,
                            bounds.Top + (bounds.Height - _mWindow.Height) / 2);
                    }
                }
            }

            _mWindow.Show();
        }

        /// <summary>
        ///     Creates the MusicBee plugin Configuration panel.
        /// </summary>
        /// <param name="panelHandle">
        ///     The panel handle.
        /// </param>
        /// <returns>
        ///     The <see cref="bool" />.
        /// </returns>
        public bool Configure(IntPtr panelHandle)
        {
            DisplayInfoWindow();
            return true;
        }

        /// <summary>
        ///     The close.
        /// </summary>
        /// <param name="reason">
        ///     The reason.
        /// </param>
        public void Close(PluginCloseReason reason)
        {
            try
            {
                _logger?.Debug($"=== PLUGIN CLOSE STARTED (reason: {reason}) ===");
                _logger?.Debug($"Current thread: {System.Threading.Thread.CurrentThread.ManagedThreadId}");

                // Clean up position update timer
                if (_positionUpdateTimer != null)
                {
                    _logger?.Debug($"Disposing _positionUpdateTimer: Enabled={_positionUpdateTimer.Enabled}, Interval={_positionUpdateTimer.Interval}ms");
                    _positionUpdateTimer.Stop();
                    _positionUpdateTimer.Elapsed -= PositionUpdateTimerOnElapsed;
                    _positionUpdateTimer.Dispose();
                    _positionUpdateTimer = null;
                    _logger?.Debug("_positionUpdateTimer disposed and set to null");
                }
                else
                {
                    _logger?.Debug("_positionUpdateTimer was already null");
                }

                // Clean up status monitoring timer
                if (_timer != null)
                {
                    _logger?.Debug($"Disposing _timer: Enabled={_timer.Enabled}, Interval={_timer.Interval}ms");
                    _timer.Stop();
                    _timer.Elapsed -= HandleTimerElapsed;
                    _timer.Dispose();
                    _timer = null;
                    _logger?.Debug("_timer disposed and set to null");
                }
                else
                {
                    _logger?.Debug("_timer was already null");
                }

                // Clean up info window
                if (_mWindow != null)
                {
                    _logger?.Debug($"Disposing _mWindow: Visible={_mWindow.Visible}, IsDisposed={_mWindow.IsDisposed}, Handle={(_mWindow.IsHandleCreated ? _mWindow.Handle.ToString() : "no handle")}");
                    try
                    {
                        _mWindow.Close();
                        _mWindow.Dispose();
                        _logger?.Debug("_mWindow closed and disposed");
                    }
                    catch (Exception windowEx)
                    {
                        _logger?.Debug($"_mWindow cleanup exception (may already be disposed): {windowEx.Message}");
                    }
                    _mWindow = null;
                }
                else
                {
                    _logger?.Debug("_mWindow was already null");
                }

                _logger?.Debug("Firing ActionSocketStop event to stop SocketServer");
                /** When the plugin closes for whatever reason the SocketServer must stop **/
                EventBus.FireEvent(new MessageEvent(EventType.ActionSocketStop));

                _logger?.Debug("=== PLUGIN CLOSE COMPLETED ===");
            }
            catch (Exception ex)
            {
                // CRITICAL: Prevent plugin exceptions from crashing MusicBee during shutdown
                try
                {
                    _logger?.Error(ex, $"FATAL: Exception during plugin close (reason: {reason})");
                }
                catch
                {
                    // Swallow all exceptions to protect MusicBee
                }
            }
        }

        /// <summary>
        ///     Cleans up any persisted files during the plugin uninstall.
        /// </summary>
        public void Uninstall()
        {
            try
            {
                var settingsFolder = _api.Setting_GetPersistentStoragePath + "\\mb_remote";
                if (Directory.Exists(settingsFolder)) Directory.Delete(settingsFolder);
            }
            catch (Exception ex)
            {
                // CRITICAL: Prevent plugin exceptions from crashing MusicBee during uninstall
                try
                {
                    _logger?.Error(ex, "FATAL: Exception during plugin uninstall");
                }
                catch
                {
                    // Swallow all exceptions to protect MusicBee
                }
            }
        }

        /// <summary>
        ///     Called by MusicBee when the user clicks Apply or Save in the MusicBee Preferences screen.
        ///     Used to save the temporary Plugin SettingsModel if the have changed.
        /// </summary>
        public void SaveSettings()
        {
            //UserSettings.SettingsModel = SettingsController.SettingsModel;
            //UserSettings.SaveSettings("mbremote");
        }

        public static void BroadcastCover(string cover)
        {
            var payload = new CoverPayload(cover, false);
            var broadcastEvent = new BroadcastEvent(Constants.NowPlayingCover);
            broadcastEvent.AddPayload(Constants.V2, cover);
            broadcastEvent.AddPayload(Constants.V3, payload);
            EventBus.FireEvent(new MessageEvent(EventType.BroadcastEvent, broadcastEvent));
        }

        public static void BroadcastLyrics(string lyrics)
        {
            var versionTwoData = !string.IsNullOrEmpty(lyrics) ? lyrics : "Lyrics Not Found";

            var lyricsPayload = new LyricsPayload(lyrics);

            var broadcastEvent = new BroadcastEvent(Constants.NowPlayingLyrics);
            broadcastEvent.AddPayload(Constants.V2, versionTwoData);
            broadcastEvent.AddPayload(Constants.V3, lyricsPayload);
            EventBus.FireEvent(new MessageEvent(EventType.BroadcastEvent, broadcastEvent));
        }

        /// <summary>
        ///     Receives event Notifications from MusicBee. It is only required if the about.ReceiveNotificationFlags =
        ///     PlayerEvents.
        /// </summary>
        /// <param name="sourceFileUrl"></param>
        /// <param name="type"></param>
        public void ReceiveNotification(string sourceFileUrl, NotificationType type)
        {
            try
            {
                /** Perfom an action depending on the notification type **/
                switch (type)
            {
                case NotificationType.TrackChanged:
                    RequestNowPlayingTrackCover();
                    RequestTrackRating("-1", string.Empty);
                    RequestLoveStatus("status", "all");
                    RequestNowPlayingTrackLyrics();
                    RequestPlayPosition("status");
                    var broadcastEvent = new BroadcastEvent(Constants.NowPlayingTrack);
                    broadcastEvent.AddPayload(Constants.V2, GetTrackInfo());
                    broadcastEvent.AddPayload(Constants.V3, GetTrackInfoV2());
                    EventBus.FireEvent(new MessageEvent(EventType.BroadcastEvent, broadcastEvent));
                    break;
                case NotificationType.VolumeLevelChanged:
                    var payload = new SocketMessage(Constants.PlayerVolume,
                        (int)
                        Math.Round(
                            _api.Player_GetVolume() * 100,
                            1)).ToJsonString();
                    SendReply(payload);
                    break;
                case NotificationType.VolumeMuteChanged:
                    EventBus.FireEvent(new MessageEvent(EventType.ReplyAvailable,
                        new SocketMessage(Constants.PlayerMute,
                            _api.Player_GetMute()).ToJsonString()
                    ));
                    break;
                case NotificationType.PlayStateChanged:
                    EventBus.FireEvent(new MessageEvent(EventType.ReplyAvailable,
                        new SocketMessage(Constants.PlayerState,
                            _api
                                .Player_GetPlayState
                                    ()).ToJsonString
                            ()));
                    break;
                case NotificationType.NowPlayingLyricsReady:
                    if (_api.ApiRevision >= 17)
                        EventBus.FireEvent(new MessageEvent(EventType.NowPlayingLyricsChange,
                            _api.NowPlaying_GetDownloadedLyrics()));

                    break;
                case NotificationType.NowPlayingArtworkReady:
                    if (_api.ApiRevision >= 17)
                        EventBus.FireEvent(new MessageEvent(EventType.NowPlayingCoverChange,
                            _api.NowPlaying_GetDownloadedArtwork()));

                    break;
                case NotificationType.PlayingTracksChanged:
                case NotificationType.NowPlayingListChanged: // Deprecated - use PlayingTracksChanged when dropping MB 3.0 support
                    EventBus.FireEvent(new MessageEvent(EventType.ReplyAvailable,
                        new SocketMessage(Constants.NowPlayingListChanged, true).ToJsonString()));
                    break;
                case NotificationType.FileAddedToLibrary:
                    var hash = CacheCover(sourceFileUrl);
                    var artist = GetAlbumArtistForTrack(sourceFileUrl);
                    var album = GetAlbumForTrack(sourceFileUrl);
                    var key = Utilities.CoverIdentifier(artist, album);
                    CoverCache.Instance.Update(key, hash);
                    _logger.Debug($"FileAddedToLibrary: {sourceFileUrl}");
                    BroadcastLibraryEvent(LibrarySubscription.EventTypes_.FileAdded, sourceFileUrl);
                    break;

                // Library Subscriptions: Tag/metadata change events
                case NotificationType.TagsChanged:
                    _logger.Debug($"TagsChanged: {sourceFileUrl}");
                    BroadcastLibraryEvent(LibrarySubscription.EventTypes_.TagChanged, sourceFileUrl);
                    break;

                case NotificationType.RatingChanged:
                    _logger.Debug($"RatingChanged: {sourceFileUrl}");
                    BroadcastLibraryEvent(LibrarySubscription.EventTypes_.RatingChanged, sourceFileUrl);
                    break;

                case NotificationType.PlayCountersChanged:
                    _logger.Debug($"PlayCountersChanged: {sourceFileUrl}");
                    BroadcastLibraryEvent(LibrarySubscription.EventTypes_.PlayCountChanged, sourceFileUrl);
                    break;

                case NotificationType.FileDeleted:
                    _logger.Debug($"FileDeleted: {sourceFileUrl}");
                    BroadcastLibraryEvent(LibrarySubscription.EventTypes_.FileDeleted, sourceFileUrl);
                    break;
                }
            }
            catch (Exception ex)
            {
                // CRITICAL: Prevent plugin exceptions from crashing MusicBee during player events
                try
                {
                    _logger?.Error(ex, $"FATAL: Exception during notification handling (type: {type}, file: {sourceFileUrl})");
                }
                catch
                {
                    // Swallow all exceptions to protect MusicBee
                }
            }
        }

        private NowPlayingTrack GetTrackInfo()
        {
            var nowPlayingTrack = new NowPlayingTrack
            {
                Artist = GetNowPlayingArtist(),
                Album = GetNowPlayingAlbum(),
                Year = GetNowPlayingYear()
            };
            nowPlayingTrack.SetTitle(GetNowPlayingTrackTitle(), GetNowPlayingFileUrl());
            return nowPlayingTrack;
        }


        private NowPlayingTrackV2 GetTrackInfoV2()
        {
            var fileUrl = GetNowPlayingFileUrl();
            var nowPlayingTrack = new NowPlayingTrackV2
            {
                Artist = GetNowPlayingArtist(),
                Album = GetNowPlayingAlbum(),
                Year = GetNowPlayingYear(),
                Path = fileUrl
            };
            nowPlayingTrack.SetTitle(GetNowPlayingTrackTitle(), fileUrl);
            return nowPlayingTrack;
        }

        public NowPlayingDetails GetPlayingTrackDetails()
        {
            var nowPlayingTrack = new NowPlayingDetails
            {
                AlbumArtist = _api.NowPlaying_GetFileTag(MetaDataType.AlbumArtist).Cleanup(),
                Genre = _api.NowPlaying_GetFileTag(MetaDataType.Genre).Cleanup(),
                TrackNo = _api.NowPlaying_GetFileTag(MetaDataType.TrackNo).Cleanup(),
                TrackCount = _api.NowPlaying_GetFileTag(MetaDataType.TrackCount).Cleanup(),
                DiscNo = _api.NowPlaying_GetFileTag(MetaDataType.DiscNo).Cleanup(),
                DiscCount = _api.NowPlaying_GetFileTag(MetaDataType.DiscCount).Cleanup(),
                Grouping = _api.NowPlaying_GetFileTag(MetaDataType.Grouping).Cleanup(),
                Publisher = _api.NowPlaying_GetFileTag(MetaDataType.Publisher).Cleanup(),
                RatingAlbum = _api.NowPlaying_GetFileTag(MetaDataType.RatingAlbum).Cleanup(),
                Composer = _api.NowPlaying_GetFileTag(MetaDataType.Composer).Cleanup(),
                Comment = _api.NowPlaying_GetFileTag(MetaDataType.Comment).Cleanup(),
                Encoder = _api.NowPlaying_GetFileTag(MetaDataType.Encoder).Cleanup(),

                Kind = _api.NowPlaying_GetFileProperty(FilePropertyType.Kind).Cleanup(),
                Format = _api.NowPlaying_GetFileProperty(FilePropertyType.Format).Cleanup(),
                Size = _api.NowPlaying_GetFileProperty(FilePropertyType.Size).Cleanup(),
                Channels = _api.NowPlaying_GetFileProperty(FilePropertyType.Channels).Cleanup(),
                SampleRate = _api.NowPlaying_GetFileProperty(FilePropertyType.SampleRate).Cleanup(),
                Bitrate = _api.NowPlaying_GetFileProperty(FilePropertyType.Bitrate).Cleanup(),
                DateModified = _api.NowPlaying_GetFileProperty(FilePropertyType.DateModified).Cleanup(),
                DateAdded = _api.NowPlaying_GetFileProperty(FilePropertyType.DateAdded).Cleanup(),
                LastPlayed = _api.NowPlaying_GetFileProperty(FilePropertyType.LastPlayed).Cleanup(),
                PlayCount = _api.NowPlaying_GetFileProperty(FilePropertyType.PlayCount).Cleanup(),
                SkipCount = _api.NowPlaying_GetFileProperty(FilePropertyType.SkipCount).Cleanup(),
                Duration = _api.NowPlaying_GetFileProperty(FilePropertyType.Duration).Cleanup()
            };

            return nowPlayingTrack;
        }

        private string GetNowPlayingFileUrl()
        {
            return _api.NowPlaying_GetFileUrl();
        }

        private string GetNowPlayingArtist()
        {
            return _api.NowPlaying_GetFileTag(MetaDataType.Artist);
        }

        private string GetNowPlayingTrackTitle()
        {
            return _api.NowPlaying_GetFileTag(MetaDataType.TrackTitle);
        }

        private string GetNowPlayingYear()
        {
            return _api.NowPlaying_GetFileTag(MetaDataType.Year);
        }

        private string GetNowPlayingAlbum()
        {
            return _api.NowPlaying_GetFileTag(MetaDataType.Album);
        }

        /// <summary>
        ///     When called plays the next track.
        /// </summary>
        /// <returns></returns>
        public void RequestNextTrack(string clientId)
        {
            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerNext,
                        _api.Player_PlayNextTrack()).ToJsonString()));
        }

        /// <summary>
        ///     When called stops the playback.
        /// </summary>
        /// <returns></returns>
        public void RequestStopPlayback(string clientId)
        {
            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerStop,
                        _api.Player_Stop()).ToJsonString()));
        }

        /// <summary>
        ///     When called changes the play/pause state or starts playing a track if the status is stopped.
        /// </summary>
        /// <returns></returns>
        public void RequestPlayPauseTrack(string clientId)
        {
            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerPlayPause,
                        _api.Player_PlayPause()).ToJsonString()));
        }

        /// <summary>
        ///     When called plays the previous track.
        /// </summary>
        /// <returns></returns>
        public void RequestPreviousTrack(string clientId)
        {
            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerPrevious,
                        _api.Player_PlayPreviousTrack()).ToJsonString()));
        }

        /// <summary>
        ///     When called plays the next album (MusicBee 3.1+ / API 50+).
        /// </summary>
        public void RequestNextAlbum(string clientId)
        {
            bool success = false;
            string message = "Feature not available";

            if (_api.Player_PlayNextAlbum != null)
            {
                success = _api.Player_PlayNextAlbum();
                message = success ? "Skipped to next album" : "Failed to skip album";
            }
            else
            {
                message = "Requires MusicBee 3.1+";
            }

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerNextAlbum,
                        new { success, message }).ToJsonString()));
        }

        /// <summary>
        ///     When called plays the previous album (MusicBee 3.1+ / API 50+).
        /// </summary>
        public void RequestPreviousAlbum(string clientId)
        {
            bool success = false;
            string message = "Feature not available";

            if (_api.Player_PlayPreviousAlbum != null)
            {
                success = _api.Player_PlayPreviousAlbum();
                message = success ? "Skipped to previous album" : "Failed to skip album";
            }
            else
            {
                message = "Requires MusicBee 3.1+";
            }

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerPreviousAlbum,
                        new { success, message }).ToJsonString()));
        }

        /// <summary>
        ///     When called if the volume string is an integer in the range [0,100] it
        ///     changes the volume to the specific value and returns the new value.
        ///     In any other case it just returns the current value for the volume.
        /// </summary>
        /// <param name="volume"> </param>
        public void RequestVolumeChange(int volume)
        {
            if (volume >= 0) _api.Player_SetVolume((float)volume / 100);

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerVolume,
                        (int)Math.Round(_api.Player_GetVolume() * 100, 1)).ToJsonString()));

            if (_api.Player_GetMute()) _api.Player_SetMute(false);
        }

        /// <summary>
        ///     Changes the player shuffle state. If the StateAction is Toggle then the current state is switched with it's
        ///     opposite,
        ///     if it is State the current state is dispatched with an Event.
        /// </summary>
        /// <param name="action"></param>
        public void RequestShuffleState(StateAction action)
        {
            if (action == StateAction.Toggle) 
            {
                _userChangingShuffle = true;
                _api.Player_SetShuffle(!_api.Player_GetShuffle());
                _userChangingShuffle = false;
            }

            EventBus.FireEvent(
                new MessageEvent(
                    EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerShuffle,
                        _api.Player_GetShuffle()).ToJsonString()));
        }

        /// <summary>
        ///     Changes the player shuffle and autodj state following the model of MusicBee.
        /// </summary>
        /// <param name="action"></param>
        public void RequestAutoDjShuffleState(StateAction action)
        {
            var shuffleEnabled = _api.Player_GetShuffle();
            var autoDjEnabled = _api.Player_GetAutoDjEnabled();

            if (action != StateAction.Toggle) return;
            if (shuffleEnabled && !autoDjEnabled)
            {
                var success = _api.Player_StartAutoDj();
                if (success) _shuffleState = ShuffleState.AutoDj;
            }
            else if (autoDjEnabled)
            {
                _api.Player_EndAutoDj();
            }
            else
            {
                var success = _api.Player_SetShuffle(true);
                if (success) _shuffleState = ShuffleState.Shuffle;
            }

            var socketMessage = new SocketMessage(Constants.PlayerShuffle, _shuffleState);
            EventBus.FireEvent(new MessageEvent(EventType.ReplyAvailable, socketMessage.SerializeToString()));
        }

        public ShuffleState GetShuffleState()
        {
            var shuffleEnabled = _api.Player_GetShuffle();
            var autoDjEnabled = _api.Player_GetAutoDjEnabled();
            var state = ShuffleState.Off;
            if (shuffleEnabled && !autoDjEnabled)
                state = ShuffleState.Shuffle;
            else if (autoDjEnabled) state = ShuffleState.AutoDj;

            return state;
        }

        /// <summary>
        ///     Changes the player mute state. If the StateAction is Toggle then the current state is switched with it's opposite,
        ///     if it is State the current state is dispatched with an Event.
        /// </summary>
        /// <param name="action"></param>
        public void RequestMuteState(StateAction action)
        {
            if (action == StateAction.Toggle) _api.Player_SetMute(!_api.Player_GetMute());

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerMute,
                        _api.Player_GetMute()).ToJsonString()));
        }

        /// <summary>
        /// </summary>
        /// <param name="action"></param>
        public void RequestScrobblerState(StateAction action)
        {
            if (action == StateAction.Toggle) _api.Player_SetScrobbleEnabled(!_api.Player_GetScrobbleEnabled());

            EventBus.FireEvent(
                new MessageEvent(
                    EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerScrobble,
                        _api.Player_GetScrobbleEnabled()).ToJsonString()));
        }

        /// <summary>
        ///     If the action equals toggle then it changes the repeat state, in any other case
        ///     it just returns the current value of the repeat.
        /// </summary>
        /// <param name="action">toggle or state</param>
        /// <returns>Repeat state: None, All, One</returns>
        public void RequestRepeatState(StateAction action)
        {
            if (action == StateAction.Toggle)
                switch (_api.Player_GetRepeat())
                {
                    case RepeatMode.None:
                        _api.Player_SetRepeat(RepeatMode.All);
                        break;
                    case RepeatMode.All:
                        _api.Player_SetRepeat(RepeatMode.One);
                        break;
                    case RepeatMode.One:
                        _api.Player_SetRepeat(RepeatMode.None);
                        break;
                }

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerRepeat,
                        _api.Player_GetRepeat()).ToJsonString()));
        }

        public void RequestNowPlayingListOrdered(string clientId, int offset = 0, int limit = 100)
        {
            _nowPlayingService.RequestNowPlayingListOrdered(clientId, offset, limit);
        }

        public void RequestNowPlayingListPage(string clientId, int offset = 0, int limit = 4000)
        {
            _nowPlayingService.RequestNowPlayingListPage(clientId, offset, limit);
        }

        public void RequestNowPlayingList(string clientId)
        {
            _nowPlayingService.RequestNowPlayingList(clientId);
        }

        public void RequestOutputDevice(string clientId)
        {
            _api.Player_GetOutputDevices(out var deviceNames, out var activeDeviceName);

            var currentDevices = new OutputDevice(deviceNames, activeDeviceName);

            SendReply(new SocketMessage(Constants.PlayerOutput,
                currentDevices).ToJsonString(), clientId);
        }

        /// <summary>
        ///     If the given rating string is not null or empty and the value of the string is a float number in the [0,5]
        ///     the function will set the new rating as the current track's new track rating. In any other case it will
        ///     just return the rating for the current track.
        /// </summary>
        /// <param name="rating">New Track Rating</param>
        /// <param name="clientId"> </param>
        /// <returns>Track Rating</returns>
        public void RequestTrackRating(string rating, string clientId)
        {
            _loveRatingService.RequestTrackRating(rating, clientId);
        }

        /// <summary>
        ///     Requests the Now Playing track lyrics. If the lyrics are available then they are dispatched along with
        ///     and event. If not, and the ApiRevision is equal or greater than r17 a request for the downloaded lyrics
        ///     is initiated. The lyrics are dispatched along with and event when ready.
        /// </summary>
        public void RequestNowPlayingTrackLyrics()
        {
            _nowPlayingService.RequestNowPlayingTrackLyrics();
        }

        /// <summary>
        ///     Requests the Now Playing Track Cover.
        /// </summary>
        public void RequestNowPlayingTrackCover()
        {
            _nowPlayingService.RequestNowPlayingTrackCover();
        }

        /// <summary>
        /// </summary>
        /// <param name="request"></param>
        public void RequestPlayPosition(string request)
        {
            _nowPlayingService.RequestPlayPosition(request);
        }

        /// <summary>
        ///     Searches in the Now playing list for the track specified and plays it.
        /// </summary>
        public void NowPlayingPlay(string index, bool isAndroid)
        {
            _nowPlayingService.NowPlayingPlay(index, isAndroid);
        }

        /// <summary>
        /// </summary>
        public void NowPlayingListRemoveTrack(int index, string clientId)
        {
            _nowPlayingService.NowPlayingListRemoveTrack(index, clientId);
        }

        /// <summary>
        ///     This function requests or changes the AutoDJ functionality's state.
        /// </summary>
        /// <param name="action">
        ///     The action can be either toggle or state.
        /// </param>
        public void RequestAutoDjState(StateAction action)
        {
            if (action == StateAction.Toggle)
            {
                if (!_api.Player_GetAutoDjEnabled())
                    _api.Player_StartAutoDj();
                else
                    _api.Player_EndAutoDj();
            }

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerAutoDj,
                        _api.Player_GetAutoDjEnabled()).ToJsonString()));
        }

        /// <summary>
        ///     This function is used to change the playing track's last.fm love rating.
        /// </summary>
        /// <param name="action">
        ///     The action can be either love, or ban.
        /// </param>
        /// <param name="clientId"></param>
        public void RequestLoveStatus(string action, string clientId)
        {
            _loveRatingService.RequestLoveStatus(action, clientId);
        }

        /// <summary>
        /// Sets the rating for a library track by file path.
        /// Used by two-way sync (P2.3).
        /// </summary>
        public bool SetLibraryTrackRating(string filePath, float rating)
        {
            return _loveRatingService.SetLibraryTrackRating(filePath, rating);
        }

        /// <summary>
        /// Sets the love/ban status for a library track by file path.
        /// Used by two-way sync (P2.4).
        /// </summary>
        public bool SetLibraryTrackLove(string filePath, string status)
        {
            return _loveRatingService.SetLibraryTrackLove(filePath, status);
        }

        /// <summary>
        ///     The function checks the MusicBee api and gets all the available playlist urls.
        /// </summary>
        /// <param name="clientId"></param>
        public void GetAvailablePlaylistUrls(string clientId)
        {
            _api.Playlist_QueryPlaylists();
            var playlists = new List<Playlist>();
            while (true)
            {
                var url = _api.Playlist_QueryGetNextPlaylist();

                if (string.IsNullOrEmpty(url)) break;

                var name = _api.Playlist_GetName(url);

                var playlist = new Playlist
                {
                    Name = name,
                    Url = url
                };
                playlists.Add(playlist);
            }

            var data = new SocketMessage(Constants.PlaylistList, playlists).ToJsonString();
            EventBus.FireEvent(new MessageEvent(EventType.ReplyAvailable, data, clientId));
        }

        public void PlayPlaylist(string clientId, string url)
        {
            var success = _api.Playlist_PlayNow(url);
            var data = new SocketMessage(Constants.PlaylistPlay, success).ToJsonString();
            EventBus.FireEvent(new MessageEvent(EventType.ReplyAvailable, data, clientId));
        }

        /// <summary>
        /// Queue a playlist to the end of the now playing list (appends without replacing)
        /// </summary>
        public void QueuePlaylist(string clientId, string url)
        {
            bool success = false;
            string[] files;

            if (_api.Playlist_QueryFilesEx(url, out files) && files != null && files.Length > 0)
            {
                success = _api.NowPlayingList_QueueFilesLast(files);
            }

            var data = new SocketMessage(Constants.PlaylistQueue, success).ToJsonString();
            EventBus.FireEvent(new MessageEvent(EventType.ReplyAvailable, data, clientId));
        }

        public void LibraryPlayAll(string clientId, bool shuffle = false)
        {
            _librarySearchService.LibraryPlayAll(clientId, shuffle);
        }

        /// <summary>
        /// </summary>
        public void RequestPlayerStatus(string clientId)
        {
            _librarySearchService.RequestPlayerStatus(clientId);
        }

        /// <summary>
        /// </summary>
        public void RequestTrackInfo(string clientId)
        {
            _trackInfoService.RequestTrackInfo(clientId);
        }

        public void RequestTrackDetails(string clientId)
        {
            _trackInfoService.RequestTrackDetails(clientId);
        }

        public void SetTrackTag(string tagName, string value, string clientId)
        {
            _trackInfoService.SetTrackTag(tagName, value, clientId);
        }

        /// <summary>
        ///     Moves a track of the now playing list to a new position.
        /// </summary>
        /// <param name="clientId">The Id of the client that initiated the request</param>
        /// <param name="from">The initial position</param>
        /// <param name="to">The final position</param>
        public void RequestNowPlayingMove(string clientId, int from, int to)
        {
            _nowPlayingService.RequestNowPlayingMove(clientId, from, to);
        }

        private static string XmlFilter(string[] tags, string query, bool isStrict,
            SearchSource source = SearchSource.None)
        {
            short src;
            if (source != SearchSource.None)
            {
                src = (short)source;
            }
            else
            {
                var userDefaults = UserSettings.Instance.Source != SearchSource.None;
                src = (short)
                    (userDefaults
                        ? UserSettings.Instance.Source
                        : SearchSource.Library);
            }


            var filter = new XElement("Source",
                new XAttribute("Type", src));

            var conditions = new XElement("Conditions",
                new XAttribute("CombineMethod", "Any"));
            foreach (var tag in tags)
            {
                var condition = new XElement("Condition",
                    new XAttribute("Field", tag),
                    new XAttribute("Comparison", isStrict ? "Is" : "Contains"),
                    new XAttribute("Value", query));
                conditions.Add(condition);
            }

            filter.Add(conditions);

            return filter.ToString();
        }

        /// <summary>
        ///     Calls the API to get albums matching the specified parameter.
        /// </summary>
        public void LibrarySearchAlbums(string albumName, string clientId)
        {
            _librarySearchService.LibrarySearchAlbums(albumName, clientId);
        }

        /// <summary>
        ///     Used to get all the albums by the specified artist.
        /// </summary>
        public void LibraryGetArtistAlbums(string artist, string clientId)
        {
            _librarySearchService.LibraryGetArtistAlbums(artist, clientId);
        }

        /// <summary>
        /// </summary>
        public void LibrarySearchArtist(string artist, string clientId)
        {
            _librarySearchService.LibrarySearchArtist(artist, clientId);
        }


        public void LibraryGetGenreArtists(string genre, string clientId)
        {
            _libraryBrowseService.LibraryGetGenreArtists(genre, clientId);
        }

        public void LibrarySearchGenres(string genre, string clientId)
        {
            _libraryBrowseService.LibrarySearchGenres(genre, clientId);
        }

        public void LibraryBrowseGenres(string clientId, int offset = 0, int limit = 4000)
        {
            _libraryBrowseService.LibraryBrowseGenres(clientId, offset, limit);
        }

        public void LibraryBrowseArtists(string clientId, int offset = 0, int limit = 4000, bool albumArtists = false)
        {
            _libraryBrowseService.LibraryBrowseArtists(clientId, offset, limit, albumArtists);
        }

        public static AlbumData CreateAlbum(string queryResult)
        {
            if (string.IsNullOrEmpty(queryResult)) return new AlbumData(string.Empty, string.Empty);

            var albumInfo = queryResult.Split('\0');

            albumInfo = albumInfo.Select(s => s.Cleanup()).ToArray();

            if (albumInfo.Length == 0) return new AlbumData(string.Empty, string.Empty);
            if (albumInfo.Length == 1) return new AlbumData(albumInfo[0], string.Empty);

            if (albumInfo.Length == 2 && queryResult.StartsWith("\0")) return new AlbumData(albumInfo[1], string.Empty);
            if (albumInfo.Length == 2) return new AlbumData(albumInfo[0], albumInfo[1]);

            var current = albumInfo.Length == 3
                ? new AlbumData(albumInfo[1], albumInfo[2])
                : new AlbumData(albumInfo[0], albumInfo[1]);

            return current;
        }

        public void LibraryBrowseAlbums(string clientId, int offset = 0, int limit = 4000)
        {
            _libraryBrowseService.LibraryBrowseAlbums(clientId, offset, limit);
        }

        public void LibraryBrowseTracks(string clientId, int offset = 0, int limit = 4000)
        {
            var tracks = new List<Track>();
            // Extended metadata only for v4.5+ clients
            var clientProtocol = Authenticator.ClientProtocolVersion(clientId);
            var includeExtendedMetadata = clientProtocol >= 4.5;

            if (_api.Library_QueryFiles(null))
                while (true)
                {
                    var currentTrack = _api.Library_QueryGetNextFile();
                    if (string.IsNullOrEmpty(currentTrack)) break;

                    int.TryParse(_api.Library_GetFileTag(currentTrack, MetaDataType.TrackNo), out var trackNumber);
                    int.TryParse(_api.Library_GetFileTag(currentTrack, MetaDataType.DiscNo), out var discNumber);

                    var track = new Track
                    {
                        Artist = GetArtistForTrack(currentTrack),
                        Title = GetTitleForTrack(currentTrack),
                        Album = GetAlbumForTrack(currentTrack),
                        AlbumArtist = GetAlbumArtistForTrack(currentTrack),
                        Genre = GetGenreForTrack(currentTrack),
                        Disc = discNumber,
                        TrackNo = trackNumber,
                        Src = currentTrack
                    };

                    // Extended metadata - v4.5 protocol feature (client requested v4.5+)
                    if (includeExtendedMetadata)
                    {
                        track.Year = _api.Library_GetFileTag(currentTrack, MetaDataType.Year);
                        track.Rating = _api.Library_GetFileTag(currentTrack, MetaDataType.Rating);
                        track.Bitrate = _api.Library_GetFileProperty(currentTrack, FilePropertyType.Bitrate);
                        track.Format = _api.Library_GetFileProperty(currentTrack, FilePropertyType.Format);

                        // Extended metadata fields (v4.5)
                        int.TryParse(_api.Library_GetFileProperty(currentTrack, FilePropertyType.PlayCount) ?? "0", out var playCount);
                        int.TryParse(_api.Library_GetFileProperty(currentTrack, FilePropertyType.SkipCount) ?? "0", out var skipCount);
                        track.PlayCount = playCount;
                        track.SkipCount = skipCount;
                        track.LastPlayed = _api.Library_GetFileProperty(currentTrack, FilePropertyType.LastPlayed);
                        track.DateAdded = _api.Library_GetFileProperty(currentTrack, FilePropertyType.DateAdded);

                        // Love status: Llfm = Loved, Blfm = Banned, empty = Neither
                        var ratingLove = _api.Library_GetFileTag(currentTrack, MetaDataType.RatingLove) ?? "";
                        track.Loved = ratingLove.Contains("L") ? "L" : (ratingLove.Contains("B") ? "B" : "");
                    }

                    tracks.Add(track);
                }

            var total = tracks.Count;
            var realLimit = offset + limit > total ? total - offset : limit;
            var message = new SocketMessage
            {
                Context = Constants.LibraryBrowseTracks,
                Data = new Page<Track>
                {
                    Data = offset > total ? new List<Track>() : tracks.GetRange(offset, realLimit),
                    Offset = offset,
                    Limit = limit,
                    Total = total
                }
            };

            var messageEvent = new MessageEvent(EventType.ReplyAvailable, message.ToJsonString(), clientId);
            EventBus.FireEvent(messageEvent);
        }

        private string GetGenreForTrack(string currentTrack)
        {
            return _api.Library_GetFileTag(currentTrack, MetaDataType.Genre).Cleanup();
        }

        private string GetAlbumArtistForTrack(string currentTrack)
        {
            return _api.Library_GetFileTag(currentTrack, MetaDataType.AlbumArtist).Cleanup();
        }

        private string GetAlbumForTrack(string currentTrack)
        {
            return _api.Library_GetFileTag(currentTrack, MetaDataType.Album).Cleanup();
        }

        private string GetTitleForTrack(string currentTrack)
        {
            return _api.Library_GetFileTag(currentTrack, MetaDataType.TrackTitle).Cleanup();
        }

        private string GetArtistForTrack(string currentTrack)
        {
            return _api.Library_GetFileTag(currentTrack, MetaDataType.Artist).Cleanup();
        }

        /// <summary>
        /// Broadcasts a library event to all subscribed clients.
        /// </summary>
        /// <param name="eventType">The event type (e.g., "tagchanged", "fileadded")</param>
        /// <param name="filePath">The file path that changed</param>
        /// <param name="additionalData">Optional additional event data</param>
        private void BroadcastLibraryEvent(string eventType, string filePath, object additionalData = null)
        {
            try
            {
                var subscriptions = LibrarySubscriptionManager.Instance.GetSubscriptionsForEvent(eventType);
                if (!subscriptions.Any())
                {
                    return;
                }

                foreach (var subscription in subscriptions)
                {
                    try
                    {
                        object eventData;
                        if (subscription.IncludeMetadata)
                        {
                            // Include full track metadata
                            eventData = new
                            {
                                path = filePath,
                                eventType,
                                artist = GetArtistForTrack(filePath),
                                album = GetAlbumForTrack(filePath),
                                albumArtist = GetAlbumArtistForTrack(filePath),
                                title = GetTitleForTrack(filePath),
                                genre = GetGenreForTrack(filePath),
                                year = _api.Library_GetFileTag(filePath, MetaDataType.Year).Cleanup(),
                                rating = _api.Library_GetFileTag(filePath, MetaDataType.Rating).Cleanup(),
                                trackNo = _api.Library_GetFileTag(filePath, MetaDataType.TrackNo).Cleanup(),
                                discNo = _api.Library_GetFileTag(filePath, MetaDataType.DiscNo).Cleanup(),
                                loved = (_api.Library_GetFileTag(filePath, MetaDataType.RatingLove) ?? "").Contains("L") ? "L" : "",
                                additional = additionalData
                            };
                        }
                        else
                        {
                            // Minimal data - just path and event type
                            eventData = new
                            {
                                path = filePath,
                                eventType,
                                additional = additionalData
                            };
                        }

                        string messageContext;
                        switch (eventType)
                        {
                            case LibrarySubscription.EventTypes_.TagChanged:
                                messageContext = Constants.LibraryTagChanged;
                                break;
                            case LibrarySubscription.EventTypes_.FileAdded:
                                messageContext = Constants.LibraryFileAdded;
                                break;
                            case LibrarySubscription.EventTypes_.FileDeleted:
                                messageContext = Constants.LibraryFileDeleted;
                                break;
                            case LibrarySubscription.EventTypes_.RatingChanged:
                                messageContext = Constants.LibraryRatingChanged;
                                break;
                            case LibrarySubscription.EventTypes_.PlayCountChanged:
                                messageContext = Constants.LibraryPlayCountChanged;
                                break;
                            default:
                                messageContext = Constants.LibraryTagChanged;
                                break;
                        }

                        var message = new SocketMessage(messageContext, eventData).ToJsonString();
                        SendReply(message, subscription.ClientId);

                        _logger.Debug("Broadcast {0} to client {1}: {2}", eventType, subscription.ClientId, filePath);
                    }
                    catch (Exception ex)
                    {
                        _logger.Error(ex, "Error broadcasting {0} to client {1}", eventType, subscription.ClientId);
                    }
                }
            }
            catch (Exception ex)
            {
                _logger.Error(ex, "Error in BroadcastLibraryEvent: {0}", eventType);
            }
        }

        public void LibrarySearchTitle(string title, string clientId)
        {
            _libraryBrowseService.LibrarySearchTitle(title, clientId);
        }

        public void LibraryGetAlbumTracks(string album, string client)
        {
            _libraryBrowseService.LibraryGetAlbumTracks(album, client);
        }

        public void RequestRadioStations(string clientId, int offset = 0, int limit = 4000)
        {
            _radioService.RequestRadioStations(clientId, offset, limit);
        }

        #region Podcast Support (MusicBee 3.1+ / API 51+)

        public void RequestPodcastSubscriptions(string clientId)
        {
            _podcastService.RequestPodcastSubscriptions(clientId);
        }

        public void RequestPodcastEpisodes(string clientId, string subscriptionId)
        {
            _podcastService.RequestPodcastEpisodes(clientId, subscriptionId);
        }

        public void RequestPodcastPlay(string clientId, string episodeUrl)
        {
            _podcastService.RequestPodcastPlay(clientId, episodeUrl);
        }

        #endregion

        /// <summary>
        /// </summary>
        /// <param name="queue"></param>
        /// <param name="tag"></param>
        /// <param name="query"></param>
        public void RequestQueueFiles(QueueType queue, MetaTag tag, string query)
        {
            string[] trackList;
            if (tag == MetaTag.Title && queue == QueueType.PlayNow)
                trackList = new[] { query };
            else
                trackList = GetUrlsForTag(tag, query);

            QueueFiles(queue, trackList, query);
        }

        /// <summary>
        ///     Takes a given query string and searches the Now Playing list for any track with a matching title or artist.
        ///     The title is checked first.
        /// </summary>
        /// <param name="query">The string representing the query</param>
        /// <param name="clientId">Client</param>
        public void NowPlayingSearch(string query, string clientId)
        {
            var result = false;
            _api.NowPlayingList_QueryFiles(XmlFilter(new[] { "ArtistPeople", "Title" }, query, false));

            while (true)
            {
                var currentTrack = _api.NowPlayingList_QueryGetNextFile();
                if (string.IsNullOrEmpty(currentTrack)) break;
                var artist = _api.Library_GetFileTag(currentTrack, MetaDataType.Artist);
                var title = _api.Library_GetFileTag(currentTrack, MetaDataType.TrackTitle);

                if (title.IndexOf(query, StringComparison.OrdinalIgnoreCase) < 0 &&
                    artist.IndexOf(query, StringComparison.OrdinalIgnoreCase) < 0) continue;
                result = _api.NowPlayingList_PlayNow(currentTrack);
                break;
            }

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.NowPlayingListSearch, result).ToJsonString(), clientId));
        }

        public string[] GetUrlsForTag(MetaTag tag, string query)
        {
            var filter = string.Empty;
            string[] tracks = { };
            switch (tag)
            {
                case MetaTag.Artist:
                    filter = XmlFilter(new[] { "ArtistPeople" }, query, true);
                    break;
                case MetaTag.Album:
                    filter = XmlFilter(new[] { "Album" }, query, true);
                    break;
                case MetaTag.Genre:
                    filter = XmlFilter(new[] { "Genre" }, query, true);
                    break;
                case MetaTag.Title:
                    filter = "";
                    break;
            }


            _api.Library_QueryFilesEx(filter, out tracks);

            var list = tracks.Select(file => new MetaData
                {
                    File = file,
                    Artist = _api.Library_GetFileTag(file, MetaDataType.Artist),
                    AlbumArtist = _api.Library_GetFileTag(file, MetaDataType.AlbumArtist),
                    Album = _api.Library_GetFileTag(file, MetaDataType.Album),
                    Title = _api.Library_GetFileTag(file, MetaDataType.TrackTitle),
                    Genre = _api.Library_GetFileTag(file, MetaDataType.Genre),
                    Year = _api.Library_GetFileTag(file, MetaDataType.Year),
                    TrackNo = _api.Library_GetFileTag(file, MetaDataType.TrackNo),
                    Disc = _api.Library_GetFileTag(file, MetaDataType.DiscNo)
                })
                .ToList();
            list.Sort();
            tracks = list.Select(r => r.File)
                .ToArray();

            return tracks;
        }

        public void RequestPlay(string clientId)
        {
            var state = _api.Player_GetPlayState();

            if (state != PlayState.Playing) _api.Player_PlayPause();
        }

        public void RequestPausePlayback(string clientId)
        {
            var state = _api.Player_GetPlayState();

            if (state == PlayState.Playing) _api.Player_PlayPause();
        }

        public void SendVisualizerNotEnabled(string clientId, bool forListRequest = false)
        {
            _visualizerService.SendVisualizerNotEnabled(clientId, forListRequest);
        }

        public void RequestVisualizerList(string clientId)
        {
            _visualizerService.RequestVisualizerList(clientId);
        }

        public void RequestVisualizer(string visualizerName, string state, string clientId)
        {
            _visualizerService.RequestVisualizer(visualizerName, state, clientId);
        }

        /// <summary>
        ///     Gets a Page of playlists from the plugin api and sends it to the client that requested it.
        /// </summary>
        /// <param name="clientId">The id of the client performing the request</param>
        /// <param name="offset">The starting position (zero based) of the dataset</param>
        /// <param name="limit">The number of elements in the dataset</param>
        public void GetAvailablePlaylistUrls(string clientId, int offset, int limit)
        {
            _api.Playlist_QueryPlaylists();
            var playlists = new List<Playlist>();
            while (true)
            {
                var url = _api.Playlist_QueryGetNextPlaylist();

                if (string.IsNullOrEmpty(url)) break;

                var name = _api.Playlist_GetName(url);

                var playlist = new Playlist
                {
                    Name = name,
                    Url = url
                };
                playlists.Add(playlist);
            }

            var total = playlists.Count;
            var realLimit = offset + limit > total ? total - offset : limit;
            var message = new SocketMessage
            {
                Context = Constants.PlaylistList,
                Data = new Page<Playlist>
                {
                    Data = offset > total ? new List<Playlist>() : playlists.GetRange(offset, realLimit),
                    Offset = offset,
                    Limit = limit,
                    Total = total
                }
            };
            var messageEvent = new MessageEvent(EventType.ReplyAvailable, message.ToJsonString(), clientId);
            EventBus.FireEvent(messageEvent);
        }

        public bool QueueFiles(QueueType queue, string[] data, string query = "")
        {
            switch (queue)
            {
                case QueueType.Next:
                    return _api.NowPlayingList_QueueFilesNext(data);
                case QueueType.Last:
                    return _api.NowPlayingList_QueueFilesLast(data);
                case QueueType.PlayNow:
                    if (data == null || data.Length == 0) return false;
                    _api.NowPlayingList_Clear();
                    _api.NowPlayingList_QueueFilesLast(data);
                    return _api.NowPlayingList_PlayNow(data[0]);
                case QueueType.AddAndPlay:
                    _api.NowPlayingList_Clear();
                    _api.NowPlayingList_QueueFilesLast(data);
                    return _api.NowPlayingList_PlayNow(query);
                default:
                    return false;
            }
        }

        public void SwitchOutputDevice(string outputDevice, string clientId)
        {
            _api.Player_SetOutputDevice(outputDevice);
            RequestOutputDevice(clientId);
        }

        private string CacheCover(string track)
        {
            var locations = PictureLocations.EmbedInFile |
                            PictureLocations.LinkToSource |
                            PictureLocations.LinkToOrganisedCopy;

            var pictureUrl = string.Empty;
            var data = new byte[] { };

            _api.Library_GetArtworkEx(
                track,
                0,
                true,
                out locations,
                out pictureUrl,
                out data
            );


            if (!pictureUrl.IsNullOrEmpty()) return Utilities.StoreCoverToCache(pictureUrl);

            var hash = data?.Length > 0 ? Utilities.StoreCoverToCache(data) : string.Empty;
            return hash;
        }

        private void PrepareCache()
        {
            var watch = Stopwatch.StartNew();
            var identifiers = GetIdentifiers();

            _logger.Debug($"Detected {identifiers.Count} albums");

            _api.Library_QueryLookupTable(null, null, null);

            if (!_api.Library_QueryFiles(null))
            {
                _logger.Debug("No result in query");
                return;
            }

            var paths = new Dictionary<string, string>();
            var modified = new Dictionary<string, string>();

            while (true)
            {
                var currentTrack = _api.Library_QueryGetNextFile();
                if (string.IsNullOrEmpty(currentTrack)) break;

                var album = GetAlbumForTrack(currentTrack);
                var artist = GetAlbumArtistForTrack(currentTrack);
                var fileModified = _api.Library_GetFileProperty(currentTrack, FilePropertyType.DateModified);

                try
                {
                    var key = Utilities.CoverIdentifier(artist, album);

                    if (!identifiers.Contains(key)) continue;

                    paths[key] = currentTrack;
                    modified[key] = fileModified;
                }
                catch (Exception e)
                {
                    _logger.Error(e, $"Failed creating identifier for {album} by {artist}");
                }
            }

            CoverCache.Instance.WarmUpCache(paths, modified);
            watch.Stop();
            _logger.Debug($"Cover cache preparation: {watch.ElapsedMilliseconds} ms");
        }

        private void BuildCoverCache()
        {
            var watch = Stopwatch.StartNew();
            CoverCache.Instance.Build(CacheCover);
            watch.Stop();
            _logger.Debug($"Cover cache task complete after: {watch.ElapsedMilliseconds} ms");
        }

        private List<string> GetIdentifiers()
        {
            var identifiers = new List<string>();
            if (!_api.Library_QueryLookupTable("album", "albumartist" + '\0' + "album", null)) return identifiers;
            try
            {
                var data = _api.Library_QueryGetLookupTableValue(null)
                    .Split(new[] { "\0\0" }, StringSplitOptions.None)
                    .Where(s => !string.IsNullOrEmpty(s))
                    .Select(s => s.Trim())
                    .Select(CreateAlbum)
                    .Select(album => Utilities.CoverIdentifier(album.Artist, album.Album))
                    .Distinct()
                    .ToList();

                identifiers.AddRange(data);
            }
            catch (IndexOutOfRangeException ex)
            {
                _logger.Error(ex, "While loading album data");
            }

            return identifiers;
        }

        public void RequestCover(string clientId, string artist, string album, string clientHash, string size)
        {
            _coverService.RequestCover(clientId, artist, album, clientHash, size);
        }

        public void RequestCoverPage(string clientId, int offset, int limit)
        {
            _coverService.RequestCoverPage(clientId, offset, limit);
        }
    }
}