using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using MusicBeePlugin.AndroidRemote.Model;
using MusicBeePlugin.AndroidRemote.Networking;
using MusicBeePlugin.AndroidRemote.Settings;
using MusicBeePlugin.Properties;
using MusicBeePlugin.Tools;
using NLog;

namespace MusicBeePlugin
{
    /// <summary>
    ///     Represents the Settings and monitoring dialog of the plugin.
    /// </summary>
    public partial class InfoWindow : Form, SocketTester.IConnectionListener
    {
        private readonly Logger _logger = LogManager.GetCurrentClassLogger();
        private BindingList<string> _ipAddressBinding;
        private IOnDebugSelectionChanged _onDebugSelectionChangedListener;
        private IOnInvalidateCacheListener _onInvalidateCacheListener;
        private SocketTester _socketTester;

        /// <summary>
        /// </summary>
        public InfoWindow()
        {
            InitializeComponent();
            _ipAddressBinding = new BindingList<string>();
        }

        public void OnConnectionResult(bool isConnected)
        {
            UpdateSocketStatus(isConnected);
        }

        /// <summary>
        ///     Updates the visual indicator with the current Socket server status.
        /// </summary>
        /// <param name="isRunning"></param>
        public void UpdateSocketStatus(bool isRunning)
        {
            if (InvokeRequired)
            {
                Invoke(new MethodInvoker(() => UpdateSocketStatus(isRunning)));
                return;
            }

            if (isRunning)
            {
                statusLabel.Text = @"Running";
                statusLabel.ForeColor = Color.Green;
            }
            else
            {
                statusLabel.Text = @"Stopped";
                statusLabel.ForeColor = Color.Red;
            }
        }

        public void UpdateCacheState(string cached)
        {
            if (InvokeRequired)
            {
                Invoke(new MethodInvoker(() => UpdateCacheState(cached)));
                return;
            }

            coversCacheValue.Text = cached;
        }

        private void HelpButtonClick(object sender, EventArgs e)
        {
            Process.Start("https://mbrc.kelsos.net/help/");
        }

        private void AriaHelpButtonClick(object sender, EventArgs e)
        {
            Process.Start("https://halrad.com/MBXRemote/initiator-actions.html");
        }

        private void InfoWindowLoad(object sender, EventArgs e)
        {
            var settings = UserSettings.Instance;
            internalIPList.DataSource = NetworkTools.GetPrivateAddressList();
            // Get build epoch from assembly metadata (set by build script)
            var assembly = System.Reflection.Assembly.GetExecutingAssembly();
            var epochAttr = assembly.GetCustomAttributes(typeof(System.Reflection.AssemblyMetadataAttribute), false)
                .OfType<System.Reflection.AssemblyMetadataAttribute>()
                .FirstOrDefault(a => a.Key == "BuildEpoch");
            var epochHex = epochAttr?.Value ?? "00000000";
            versionLabel.Text = $"{settings.CurrentVersion ?? "1.5.0"}+{epochHex}";

            // Clamp port value to valid range
            var port = settings.ListeningPort;
            if (port < portNumericUpDown.Minimum) port = (uint)portNumericUpDown.Minimum;
            if (port > portNumericUpDown.Maximum) port = (uint)portNumericUpDown.Maximum;
            portNumericUpDown.Value = port;

            UpdateFilteringSelection(settings.FilterSelection);

            UpdateSocketStatus(SocketServer.Instance.IsRunning);
            allowedAddressesComboBox.DataSource = _ipAddressBinding;

            if (settings.Source == SearchSource.None) settings.Source |= SearchSource.Library;

            debugEnabled.Checked = settings.DebugLogEnabled;
            firewallCheckbox.Checked = settings.UpdateFirewall;
            ariaCheckbox.Checked = settings.AriaEnabled;
            experimentalCheckbox.Checked = settings.ExperimentalFeaturesEnabled;

            _logger.Debug($"Selected source is -> {settings.Source}");

            _socketTester = new SocketTester { ConnectionListener = this };
            _socketTester.VerifyConnection();
            coversCacheValue.Text = CoverCache.Instance.State;
        }

        private void SelectionFilteringComboBoxSelectedIndexChanged(object sender, EventArgs e)
        {
            switch (selectionFilteringComboBox.SelectedIndex)
            {
                case 0:
                    addressLabel.Enabled = false;
                    ipAddressInputTextBox.Enabled = false;
                    rangeNumericUpDown.Enabled = false;
                    addAddressButton.Enabled = false;
                    removeAddressButton.Enabled = false;
                    allowedAddressesComboBox.Enabled = false;
                    allowedLabel.Enabled = false;
                    UserSettings.Instance.FilterSelection = FilteringSelection.All;
                    break;
                case 1:
                    addressLabel.Enabled = true;
                    ipAddressInputTextBox.Enabled = true;
                    rangeNumericUpDown.Enabled = true;
                    addAddressButton.Enabled = false;
                    removeAddressButton.Enabled = false;
                    allowedAddressesComboBox.Enabled = false;
                    allowedLabel.Enabled = false;
                    UserSettings.Instance.FilterSelection = FilteringSelection.Range;
                    // Load saved range values
                    ipAddressInputTextBox.Text = UserSettings.Instance.BaseIp ?? "";
                    var lastOctet = UserSettings.Instance.LastOctetMax;
                    if (lastOctet < rangeNumericUpDown.Minimum) lastOctet = (uint)rangeNumericUpDown.Minimum;
                    if (lastOctet > rangeNumericUpDown.Maximum) lastOctet = (uint)rangeNumericUpDown.Maximum;
                    rangeNumericUpDown.Value = lastOctet;
                    break;
                case 2:
                    addressLabel.Enabled = true;
                    ipAddressInputTextBox.Enabled = true;
                    rangeNumericUpDown.Enabled = false;
                    addAddressButton.Enabled = true;
                    removeAddressButton.Enabled = true;
                    allowedAddressesComboBox.Enabled = true;
                    allowedLabel.Enabled = true;
                    UserSettings.Instance.FilterSelection = FilteringSelection.Specific;
                    // Load existing whitelist into binding
                    var existingList = UserSettings.Instance.IpAddressList;
                    var ipList = existingList != null ? new List<string>(existingList) : new List<string>();
                    _ipAddressBinding = new BindingList<string>(ipList);
                    allowedAddressesComboBox.DataSource = null;
                    allowedAddressesComboBox.DataSource = _ipAddressBinding;
                    break;
            }
        }

        private void UpdateFilteringSelection(FilteringSelection selection)
        {
            try
            {
                switch (selection)
                {
                    case FilteringSelection.All:
                        selectionFilteringComboBox.SelectedIndex = 0;
                        break;
                    case FilteringSelection.Range:
                        ipAddressInputTextBox.Text = UserSettings.Instance.BaseIp ?? "";
                        // Clamp value to valid range
                        var lastOctet = UserSettings.Instance.LastOctetMax;
                        if (lastOctet < rangeNumericUpDown.Minimum) lastOctet = (uint)rangeNumericUpDown.Minimum;
                        if (lastOctet > rangeNumericUpDown.Maximum) lastOctet = (uint)rangeNumericUpDown.Maximum;
                        rangeNumericUpDown.Value = lastOctet;
                        selectionFilteringComboBox.SelectedIndex = 1;
                        break;
                    case FilteringSelection.Specific:
                        // Ensure list is never null - create fresh list from source
                        var sourceList = UserSettings.Instance.IpAddressList;
                        var ipList = sourceList != null ? new List<string>(sourceList) : new List<string>();
                        _ipAddressBinding = new BindingList<string>(ipList);
                        // Rebind the ComboBox to the new list
                        allowedAddressesComboBox.DataSource = null;
                        allowedAddressesComboBox.DataSource = _ipAddressBinding;
                        selectionFilteringComboBox.SelectedIndex = 2;
                        break;
                    default:
                        selectionFilteringComboBox.SelectedIndex = 0;
                        break;
                }
            }
            catch (Exception ex)
            {
                // If anything fails, default to "All" selection
                System.Diagnostics.Debug.WriteLine($"UpdateFilteringSelection error: {ex.Message}");
                selectionFilteringComboBox.SelectedIndex = 0;
            }
        }

        private void HandleSaveButtonClick(object sender, EventArgs e)
        {
            UserSettings.Instance.ListeningPort = (uint)portNumericUpDown.Value;

            switch (selectionFilteringComboBox.SelectedIndex)
            {
                case 0:
                    break;
                case 1:
                    UserSettings.Instance.BaseIp = ipAddressInputTextBox.Text;
                    UserSettings.Instance.LastOctetMax = (uint)rangeNumericUpDown.Value;
                    break;
                case 2:
                    UserSettings.Instance.IpAddressList = new List<string>(_ipAddressBinding);
                    break;
            }

            UserSettings.Instance.UpdateFirewall = firewallCheckbox.Checked;
            UserSettings.Instance.AriaEnabled = ariaCheckbox.Checked;
            UserSettings.Instance.ExperimentalFeaturesEnabled = experimentalCheckbox.Checked;
            UserSettings.Instance.SaveSettings();
            

            if (firewallCheckbox.Checked) UpdateFirewallRules(UserSettings.Instance.ListeningPort);

            _socketTester.VerifyConnection();
        }

        private void AddAddressButtonClick(object sender, EventArgs e)
        {
            if (!IsAddressValid()) return;
            if (!_ipAddressBinding.Contains(ipAddressInputTextBox.Text))
                _ipAddressBinding.Add(ipAddressInputTextBox.Text);
        }

        private void RemoveAddressButtonClick(object sender, EventArgs e)
        {
            _ipAddressBinding.Remove(allowedAddressesComboBox.Text);
        }

        private bool IsAddressValid()
        {
            const string pattern =
                @"\b(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\b";
            return Regex.IsMatch(ipAddressInputTextBox.Text, pattern);
        }

        private void HandleIpAddressInputTextBoxTextChanged(object sender, EventArgs e)
        {
            var isAddressValid = IsAddressValid();
            ipAddressInputTextBox.BackColor = isAddressValid ? Color.LightGreen : Color.Red;
            if (!isAddressValid || selectionFilteringComboBox.SelectedIndex != 1) return;

            try
            {
                var addressSplit = ipAddressInputTextBox.Text.Split(".".ToCharArray(),
                    StringSplitOptions.RemoveEmptyEntries);
                if (addressSplit.Length >= 4 && int.TryParse(addressSplit[3], out var lastOctet))
                {
                    rangeNumericUpDown.Minimum = Math.Max(0, Math.Min(255, lastOctet));
                }
            }
            catch
            {
                // Ignore parsing errors
            }
        }

        private void DebugCheckboxCheckedChanged(object sender, EventArgs e)
        {
            var settings = UserSettings.Instance;
            settings.DebugLogEnabled = debugEnabled.Checked;
            _onDebugSelectionChangedListener?.SelectionChanged(settings.DebugLogEnabled);
        }

        private void OpenLogButtonClick(object sender, EventArgs e)
        {
            if (File.Exists(UserSettings.Instance.FullLogPath))
                Process.Start(UserSettings.Instance.FullLogPath);
            else
                MessageBox.Show(Resources.InfoWindow_OpenLogButtonClick_Log_file_doesn_t_exist);
        }

        private void OnCacheInvalidateButtonPressed(object sender, EventArgs e)
        {
            _onInvalidateCacheListener?.InvalidateCache();
        }

        public void SetOnDebugSelectionListener(IOnDebugSelectionChanged onDebugSelectionChangedListener)
        {
            _onDebugSelectionChangedListener = onDebugSelectionChangedListener;
        }

        /// <summary>
        ///     When called it will execute the firewall-utility passing the port settings
        ///     needed by the plugin.
        /// </summary>
        private void UpdateFirewallRules(uint port)
        {
            var cmd = $"{AppDomain.CurrentDomain.BaseDirectory}\\Plugins\\firewall-utility.exe";
            if (!File.Exists(cmd)) return;
            var startInfo = new ProcessStartInfo(cmd)
            {
                Verb = "runas",
                Arguments = $"-s {port}"
            };
            try
            {
                Process.Start(startInfo);
            }
            catch (System.ComponentModel.Win32Exception)
            {
                // User canceled UAC prompt - this is expected behavior, not an error
            }
        }

        public void SetOnInvalidateCacheListener(IOnInvalidateCacheListener onInvalidateCacheListener)
        {
            _onInvalidateCacheListener = onInvalidateCacheListener;
        }

        public interface IOnDebugSelectionChanged
        {
            void SelectionChanged(bool enabled);
        }

        public interface IOnInvalidateCacheListener
        {
            void InvalidateCache();
        }
    }
}