using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using NLog;

namespace MusicBeePlugin.AndroidRemote.Subscriptions
{
    /// <summary>
    /// Manages library event subscriptions for connected clients.
    /// Thread-safe singleton that tracks which clients want to receive library events.
    /// </summary>
    public class LibrarySubscriptionManager
    {
        private static readonly Logger Logger = LogManager.GetCurrentClassLogger();
        private static readonly Lazy<LibrarySubscriptionManager> _instance =
            new Lazy<LibrarySubscriptionManager>(() => new LibrarySubscriptionManager());

        /// <summary>
        /// Singleton instance.
        /// </summary>
        public static LibrarySubscriptionManager Instance => _instance.Value;

        /// <summary>
        /// Thread-safe dictionary of client subscriptions.
        /// Key: ClientId, Value: LibrarySubscription
        /// </summary>
        private readonly ConcurrentDictionary<string, LibrarySubscription> _subscriptions =
            new ConcurrentDictionary<string, LibrarySubscription>(StringComparer.OrdinalIgnoreCase);

        private LibrarySubscriptionManager() { }

        /// <summary>
        /// Subscribe a client to library events.
        /// </summary>
        /// <param name="clientId">The client's connection ID.</param>
        /// <param name="eventTypes">Event types to subscribe to.</param>
        /// <param name="includeMetadata">Whether to include full track metadata in notifications.</param>
        /// <returns>The created subscription.</returns>
        public LibrarySubscription Subscribe(string clientId, IEnumerable<string> eventTypes, bool includeMetadata = true)
        {
            if (string.IsNullOrEmpty(clientId))
            {
                Logger.Warn("Attempted to subscribe with empty clientId");
                return null;
            }

            var subscription = new LibrarySubscription
            {
                ClientId = clientId,
                EventTypes = new HashSet<string>(eventTypes ?? LibrarySubscription.EventTypes_.All, StringComparer.OrdinalIgnoreCase),
                IncludeMetadata = includeMetadata,
                SubscribedAt = DateTime.UtcNow
            };

            _subscriptions.AddOrUpdate(clientId, subscription, (key, existing) => subscription);

            Logger.Info("Client {0} subscribed to library events: [{1}], includeMetadata={2}",
                clientId, string.Join(", ", subscription.EventTypes), includeMetadata);

            return subscription;
        }

        /// <summary>
        /// Unsubscribe a client from all library events.
        /// </summary>
        /// <param name="clientId">The client's connection ID.</param>
        /// <returns>True if the client was subscribed and is now unsubscribed.</returns>
        public bool Unsubscribe(string clientId)
        {
            if (string.IsNullOrEmpty(clientId))
                return false;

            var removed = _subscriptions.TryRemove(clientId, out var subscription);

            if (removed)
            {
                Logger.Info("Client {0} unsubscribed from library events", clientId);
            }

            return removed;
        }

        /// <summary>
        /// Get all client IDs subscribed to a specific event type.
        /// </summary>
        /// <param name="eventType">The event type (e.g., "tagchanged", "fileadded").</param>
        /// <returns>Collection of client IDs subscribed to this event type.</returns>
        public IEnumerable<string> GetSubscribedClients(string eventType)
        {
            return _subscriptions.Values
                .Where(s => s.IsSubscribedTo(eventType))
                .Select(s => s.ClientId)
                .ToList();
        }

        /// <summary>
        /// Get all subscriptions for clients subscribed to a specific event type.
        /// </summary>
        /// <param name="eventType">The event type.</param>
        /// <returns>Collection of subscriptions.</returns>
        public IEnumerable<LibrarySubscription> GetSubscriptionsForEvent(string eventType)
        {
            return _subscriptions.Values
                .Where(s => s.IsSubscribedTo(eventType))
                .ToList();
        }

        /// <summary>
        /// Check if a specific client is subscribed to a specific event type.
        /// </summary>
        public bool IsSubscribed(string clientId, string eventType)
        {
            if (_subscriptions.TryGetValue(clientId, out var subscription))
            {
                return subscription.IsSubscribedTo(eventType);
            }
            return false;
        }

        /// <summary>
        /// Get the subscription for a specific client.
        /// </summary>
        public LibrarySubscription GetSubscription(string clientId)
        {
            _subscriptions.TryGetValue(clientId, out var subscription);
            return subscription;
        }

        /// <summary>
        /// Check if any clients are subscribed to library events.
        /// </summary>
        public bool HasSubscribers => !_subscriptions.IsEmpty;

        /// <summary>
        /// Get the count of subscribed clients.
        /// </summary>
        public int SubscriberCount => _subscriptions.Count;

        /// <summary>
        /// Called when a client disconnects to clean up their subscription.
        /// </summary>
        /// <param name="clientId">The disconnected client's ID.</param>
        public void OnClientDisconnected(string clientId)
        {
            if (Unsubscribe(clientId))
            {
                Logger.Debug("Cleaned up library subscription for disconnected client {0}", clientId);
            }
        }

        /// <summary>
        /// Get all current subscriptions (for debugging/monitoring).
        /// </summary>
        public IEnumerable<LibrarySubscription> GetAllSubscriptions()
        {
            return _subscriptions.Values.ToList();
        }

        /// <summary>
        /// Clear all subscriptions (for testing or shutdown).
        /// </summary>
        public void ClearAll()
        {
            var count = _subscriptions.Count;
            _subscriptions.Clear();
            Logger.Info("Cleared all {0} library subscriptions", count);
        }
    }
}
