using System;
using System.Collections.Generic;

namespace MusicBeePlugin.AndroidRemote.Subscriptions
{
    /// <summary>
    /// Represents a client's subscription to library events.
    /// </summary>
    public class LibrarySubscription
    {
        /// <summary>
        /// The client ID (from socket connection).
        /// </summary>
        public string ClientId { get; set; }

        /// <summary>
        /// Set of event types the client is subscribed to.
        /// Values: "tagchanged", "fileadded", "filedeleted", "ratingchanged", "playcountchanged"
        /// </summary>
        public HashSet<string> EventTypes { get; set; } = new HashSet<string>(StringComparer.OrdinalIgnoreCase);

        /// <summary>
        /// If true, include full track metadata in event notifications.
        /// If false, only send file path and changed values.
        /// </summary>
        public bool IncludeMetadata { get; set; }

        /// <summary>
        /// When the subscription was created.
        /// </summary>
        public DateTime SubscribedAt { get; set; } = DateTime.UtcNow;

        /// <summary>
        /// Known event type constants.
        /// </summary>
        public static class EventTypes_
        {
            public const string TagChanged = "tagchanged";
            public const string FileAdded = "fileadded";
            public const string FileDeleted = "filedeleted";
            public const string RatingChanged = "ratingchanged";
            public const string PlayCountChanged = "playcountchanged";

            public static readonly string[] All = new[]
            {
                TagChanged, FileAdded, FileDeleted, RatingChanged, PlayCountChanged
            };
        }

        /// <summary>
        /// Check if this subscription includes a specific event type.
        /// </summary>
        public bool IsSubscribedTo(string eventType)
        {
            return EventTypes.Contains(eventType);
        }

        public override string ToString()
        {
            return $"LibrarySubscription[Client={ClientId}, Events=[{string.Join(",", EventTypes)}], IncludeMetadata={IncludeMetadata}]";
        }
    }
}
