using System;
using System.Windows.Forms;
using MusicBeePlugin.AndroidRemote.Events;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Networking;
using NLog;
using static MusicBeePlugin.Plugin;

namespace MusicBeePlugin.AndroidRemote.Services
{
    /// <summary>
    /// Handles visualizer operations: list, show, hide.
    /// Extracted from Plugin.cs to reduce monolith size.
    /// </summary>
    public class VisualizerService
    {
        private readonly Logger _logger = LogManager.GetCurrentClassLogger();

        public VisualizerService()
        {
        }

        /// <summary>
        /// Sends a response indicating that visualizer control is not enabled (experimental features disabled)
        /// </summary>
        public void SendVisualizerNotEnabled(string clientId, bool forListRequest = false)
        {
            // Use the correct message type based on what was requested
            var messageType = forListRequest ? Constants.PlayerVisualizerList : Constants.PlayerVisualizer;
            var response = new SocketMessage(messageType, new
            {
                error = "Not Enabled",
                message = "Experimental features are disabled in plugin settings",
                visualizers = Array.Empty<string>()  // Include empty array so client can handle it
            });
            SendReply(response.ToJsonString(), clientId);
        }

        /// <summary>
        /// Request list of available visualizers.
        /// </summary>
        public void RequestVisualizerList(string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            _logger.Debug("Getting visualizer list");

            string[] visualizerNames = null;
            string defaultVisualizer = null;
            WindowState defaultState = WindowState.Off;
            WindowState currentState = WindowState.Off;

            bool success = api.MB_GetVisualiserInformation(
                out visualizerNames,
                out defaultVisualizer,
                out defaultState,
                out currentState);

            if (success && visualizerNames != null)
            {
                var response = new
                {
                    visualizers = visualizerNames,
                    defaultVisualizer = defaultVisualizer,
                    defaultState = defaultState.ToString(),
                    currentState = currentState.ToString()
                };

                var message = new SocketMessage(Constants.PlayerVisualizerList, response);
                SendReply(message.ToJsonString(), clientId);

                _logger.Info($"Sent visualizer list with {visualizerNames.Length} items");
            }
            else
            {
                var errorMessage = new SocketMessage(Constants.PlayerVisualizerList, new { error = "Failed to get visualizer list" });
                SendReply(errorMessage.ToJsonString(), clientId);
                _logger.Warn("Failed to get visualizer list");
            }
        }

        /// <summary>
        /// Request to show or hide a specific visualizer.
        /// </summary>
        public void RequestVisualizer(string visualizerName, string state, string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            try
            {
                _logger.Info($"RequestVisualizer called with name='{visualizerName}', state='{state}'");

                // Parse the window state
                WindowState windowState = WindowState.Normal;
                switch (state?.ToLower())
                {
                    case "off":
                    case "hide":
                    case "close":
                        windowState = WindowState.Off;
                        break;
                    case "fullscreen":
                    case "full":
                        windowState = WindowState.Fullscreen;
                        break;
                    case "desktop":
                        windowState = WindowState.Desktop;
                        break;
                    case "normal":
                    case "window":
                    default:
                        windowState = WindowState.Normal;
                        break;
                }

                _logger.Debug($"Parsed window state: {windowState} ({(int)windowState})");

                bool success = false;

                // Log current state before attempting change
                string[] currentVisualizers;
                string defaultVis;
                WindowState defaultState, currentState;
                if (api.MB_GetVisualiserInformation(out currentVisualizers, out defaultVis, out defaultState, out currentState))
                {
                    _logger.Debug($"Before change - Current state: {currentState}, Default: '{defaultVis}'");
                    if (currentVisualizers != null)
                    {
                        _logger.Debug($"Available visualizers: {string.Join(", ", currentVisualizers)}");
                    }
                }

                // Visualizer operations need to run on the UI thread
                var hwnd = api.MB_GetWindowHandle();
                var mb = (Form)Control.FromHandle(hwnd);

                // Capture variables for closure
                var visName = visualizerName;
                var winState = windowState;
                var defVis = defaultVis;

                // Execute on UI thread
                mb.Invoke(new MethodInvoker(() =>
                {
                    // Show the requested visualizer
                    if (winState == WindowState.Off)
                    {
                        _logger.Debug("Turning off visualizer");
                        success = api.MB_ShowVisualiser(null, WindowState.Off);
                        _logger.Debug($"Turn off result: {success}");
                    }
                    else if (!string.IsNullOrEmpty(visName))
                    {
                        _logger.Debug($"Attempting to show visualizer '{visName}' in {winState} state");

                        // Try showing the visualizer with the exact name
                        success = api.MB_ShowVisualiser(visName, winState);
                        _logger.Debug($"Show visualizer '{visName}' result: {success}");

                        // If that didn't work and it's the default, try with null
                        if (!success && (visName == defVis || visName.ToLower() == "default"))
                        {
                            _logger.Debug("Trying null for default visualizer");
                            success = api.MB_ShowVisualiser(null, winState);
                            _logger.Debug($"Show default visualizer with null result: {success}");
                        }
                    }
                    else
                    {
                        // Empty name, try to show default
                        _logger.Debug("No visualizer name specified, trying to show default");
                        success = api.MB_ShowVisualiser(null, winState);
                        _logger.Debug($"Show default result: {success}");
                    }
                }));

                // Log state after attempting change
                if (api.MB_GetVisualiserInformation(out currentVisualizers, out defaultVis, out defaultState, out currentState))
                {
                    _logger.Debug($"After change - Current state: {currentState}");
                }

                if (success)
                {
                    var response = new SocketMessage(Constants.PlayerVisualizer, new
                    {
                        success = success,
                        message = $"Visualizer '{visualizerName}' shown in {windowState} state"
                    });
                    SendReply(response.ToJsonString(), clientId);
                }
                else
                {
                    var response = new SocketMessage(Constants.PlayerVisualizer, new
                    {
                        success = false,
                        message = $"Failed to show visualizer '{visualizerName}'"
                    });
                    SendReply(response.ToJsonString(), clientId);
                    _logger.Warn($"Failed to show visualizer '{visualizerName}'");
                }
            }
            catch (Exception ex)
            {
                _logger.Error($"Exception in RequestVisualizer: {ex}");
                var response = new SocketMessage(Constants.PlayerVisualizer, new
                {
                    success = false,
                    message = $"Exception: {ex.Message}"
                });
                SendReply(response.ToJsonString(), clientId);
            }
        }
    }
}
