using System;
using MusicBeePlugin.AndroidRemote.Events;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Networking;
using MusicBeePlugin.AndroidRemote.Utilities;
using NLog;
using static MusicBeePlugin.Plugin;

namespace MusicBeePlugin.AndroidRemote.Services
{
    /// <summary>
    /// Handles track information retrieval and tag modification.
    /// Extracted from Plugin.cs to reduce monolith size.
    /// </summary>
    public class TrackInfoService
    {
        private readonly Logger _logger = LogManager.GetCurrentClassLogger();

        public TrackInfoService()
        {
        }

        /// <summary>
        /// Get basic track info (v2 protocol).
        /// </summary>
        public NowPlayingTrack GetTrackInfo()
        {
            var api = Plugin.Instance.MbApiInterface;
            var nowPlayingTrack = new NowPlayingTrack
            {
                Artist = api.NowPlaying_GetFileTag(MetaDataType.Artist),
                Album = api.NowPlaying_GetFileTag(MetaDataType.Album),
                Year = api.NowPlaying_GetFileTag(MetaDataType.Year)
            };
            nowPlayingTrack.SetTitle(api.NowPlaying_GetFileTag(MetaDataType.TrackTitle), api.NowPlaying_GetFileUrl());
            return nowPlayingTrack;
        }

        /// <summary>
        /// Get extended track info (v3+ protocol).
        /// </summary>
        public NowPlayingTrackV2 GetTrackInfoV2()
        {
            var api = Plugin.Instance.MbApiInterface;
            var fileUrl = api.NowPlaying_GetFileUrl();
            var nowPlayingTrack = new NowPlayingTrackV2
            {
                Artist = api.NowPlaying_GetFileTag(MetaDataType.Artist),
                Album = api.NowPlaying_GetFileTag(MetaDataType.Album),
                Year = api.NowPlaying_GetFileTag(MetaDataType.Year),
                Path = fileUrl
            };
            nowPlayingTrack.SetTitle(api.NowPlaying_GetFileTag(MetaDataType.TrackTitle), fileUrl);
            return nowPlayingTrack;
        }

        /// <summary>
        /// Get detailed track information including file properties.
        /// </summary>
        public NowPlayingDetails GetPlayingTrackDetails()
        {
            var api = Plugin.Instance.MbApiInterface;
            var nowPlayingTrack = new NowPlayingDetails
            {
                AlbumArtist = api.NowPlaying_GetFileTag(MetaDataType.AlbumArtist).Cleanup(),
                Genre = api.NowPlaying_GetFileTag(MetaDataType.Genre).Cleanup(),
                TrackNo = api.NowPlaying_GetFileTag(MetaDataType.TrackNo).Cleanup(),
                TrackCount = api.NowPlaying_GetFileTag(MetaDataType.TrackCount).Cleanup(),
                DiscNo = api.NowPlaying_GetFileTag(MetaDataType.DiscNo).Cleanup(),
                DiscCount = api.NowPlaying_GetFileTag(MetaDataType.DiscCount).Cleanup(),
                Grouping = api.NowPlaying_GetFileTag(MetaDataType.Grouping).Cleanup(),
                Publisher = api.NowPlaying_GetFileTag(MetaDataType.Publisher).Cleanup(),
                RatingAlbum = api.NowPlaying_GetFileTag(MetaDataType.RatingAlbum).Cleanup(),
                Composer = api.NowPlaying_GetFileTag(MetaDataType.Composer).Cleanup(),
                Comment = api.NowPlaying_GetFileTag(MetaDataType.Comment).Cleanup(),
                Encoder = api.NowPlaying_GetFileTag(MetaDataType.Encoder).Cleanup(),

                Kind = api.NowPlaying_GetFileProperty(FilePropertyType.Kind).Cleanup(),
                Format = api.NowPlaying_GetFileProperty(FilePropertyType.Format).Cleanup(),
                Size = api.NowPlaying_GetFileProperty(FilePropertyType.Size).Cleanup(),
                Channels = api.NowPlaying_GetFileProperty(FilePropertyType.Channels).Cleanup(),
                SampleRate = api.NowPlaying_GetFileProperty(FilePropertyType.SampleRate).Cleanup(),
                Bitrate = api.NowPlaying_GetFileProperty(FilePropertyType.Bitrate).Cleanup(),
                DateModified = api.NowPlaying_GetFileProperty(FilePropertyType.DateModified).Cleanup(),
                DateAdded = api.NowPlaying_GetFileProperty(FilePropertyType.DateAdded).Cleanup(),
                LastPlayed = api.NowPlaying_GetFileProperty(FilePropertyType.LastPlayed).Cleanup(),
                PlayCount = api.NowPlaying_GetFileProperty(FilePropertyType.PlayCount).Cleanup(),
                SkipCount = api.NowPlaying_GetFileProperty(FilePropertyType.SkipCount).Cleanup(),
                Duration = api.NowPlaying_GetFileProperty(FilePropertyType.Duration).Cleanup()
            };

            return nowPlayingTrack;
        }

        /// <summary>
        /// Request track info for a client.
        /// </summary>
        public void RequestTrackInfo(string clientId)
        {
            var protocolVersion = Authenticator.ClientProtocolVersion(clientId);
            var message = protocolVersion > 2
                ? new SocketMessage(Constants.NowPlayingTrack, GetTrackInfoV2())
                : new SocketMessage(Constants.NowPlayingTrack, GetTrackInfo());

            var messageEvent = new MessageEvent(EventType.ReplyAvailable, message.ToJsonString(), clientId);
            EventBus.FireEvent(messageEvent);
        }

        /// <summary>
        /// Request detailed track info for a client.
        /// </summary>
        public void RequestTrackDetails(string clientId)
        {
            var message = new SocketMessage(Constants.NowPlayingDetails, GetPlayingTrackDetails());

            var messageEvent = new MessageEvent(EventType.ReplyAvailable, message.ToJsonString(), clientId);
            EventBus.FireEvent(messageEvent);
        }

        /// <summary>
        /// Set a tag on the current track.
        /// </summary>
        public void SetTrackTag(string tagName, string value, string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            var currentTrack = api.NowPlaying_GetFileUrl();
            var metadataType = MetaDataType.AlbumArtistRaw;
            switch (tagName)
            {
                case "TrackTitle":
                    metadataType = MetaDataType.TrackTitle;
                    break;
                case "Artist":
                    metadataType = MetaDataType.Artist;
                    break;
                case "Album":
                    metadataType = MetaDataType.Album;
                    break;
                case "Year":
                    metadataType = MetaDataType.Year;
                    break;
                case "AlbumArtist":
                    metadataType = MetaDataType.AlbumArtist;
                    break;
                case "Genre":
                    metadataType = MetaDataType.Genre;
                    break;
                case "TrackNo":
                    metadataType = MetaDataType.TrackNo;
                    break;
                case "TrackCount":
                    metadataType = MetaDataType.TrackCount;
                    break;
                case "DiscNo":
                    metadataType = MetaDataType.DiscNo;
                    break;
                case "DiscCount":
                    metadataType = MetaDataType.DiscCount;
                    break;
                case "Grouping":
                    metadataType = MetaDataType.Grouping;
                    break;
                case "Publisher":
                    metadataType = MetaDataType.Publisher;
                    break;
                case "RatingAlbum":
                    metadataType = MetaDataType.RatingAlbum;
                    break;
                case "Composer":
                    metadataType = MetaDataType.Composer;
                    break;
                case "Comment":
                    metadataType = MetaDataType.Comment;
                    break;
                case "Encoder":
                    metadataType = MetaDataType.Encoder;
                    break;
                case "Lyrics":
                    metadataType = MetaDataType.Lyrics;
                    break;
            }

            if (metadataType == MetaDataType.AlbumArtistRaw)
            {
                _logger.Error("Requested tag type not found");
                return;
            }

            try
            {
                api.Library_SetFileTag(currentTrack, metadataType, value);
                api.Library_CommitTagsToFile(currentTrack);
                api.MB_RefreshPanels();

                EventBus.FireEvent(
                    new MessageEvent(EventType.ReplyAvailable,
                        new SocketMessage(Constants.NowPlayingDetails, GetPlayingTrackDetails())));
            }
            catch (Exception e)
            {
                _logger.Error(e, "On Tag Change call");
            }
        }
    }
}
