using System;
using System.Collections.Generic;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Networking;
using NLog;
using static MusicBeePlugin.Plugin;

namespace MusicBeePlugin.AndroidRemote.Services
{
    /// <summary>
    /// Handles podcast operations: subscriptions, episodes, playback.
    /// Extracted from Plugin.cs to reduce monolith size.
    /// </summary>
    public class PodcastService
    {
        private readonly Logger _logger = LogManager.GetCurrentClassLogger();

        public PodcastService()
        {
        }

        /// <summary>
        /// Gets the list of podcast subscriptions (requires MusicBee 3.1+).
        /// </summary>
        public void RequestPodcastSubscriptions(string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            var subscriptions = new List<AndroidRemote.Model.Entities.PodcastSubscription>();
            string errorMessage = null;

            try
            {
                if (api.Podcasts_QuerySubscriptions == null)
                {
                    errorMessage = "Requires MusicBee 3.1+";
                }
                else
                {
                    string[] ids;
                    if (api.Podcasts_QuerySubscriptions(null, out ids) && ids != null)
                    {
                        foreach (var id in ids)
                        {
                            string[] details;
                            if (api.Podcasts_GetSubscription(id, out details) && details != null && details.Length >= 6)
                            {
                                subscriptions.Add(new AndroidRemote.Model.Entities.PodcastSubscription
                                {
                                    Id = id,
                                    Title = details[(int)SubscriptionMetaDataType.Title],
                                    Grouping = details[(int)SubscriptionMetaDataType.Grouping],
                                    Genre = details[(int)SubscriptionMetaDataType.Genre],
                                    Description = details[(int)SubscriptionMetaDataType.Description],
                                    DownloadedCount = int.TryParse(details[(int)SubscriptionMetaDataType.DounloadedCount], out var count) ? count : 0
                                });
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                _logger.Error(ex, "Error getting podcast subscriptions");
                errorMessage = ex.Message;
            }

            var response = new SocketMessage
            {
                Context = Constants.PodcastSubscriptions,
                Data = new
                {
                    success = errorMessage == null,
                    error = errorMessage,
                    subscriptions = subscriptions,
                    total = subscriptions.Count
                }
            };

            SendReply(response.ToJsonString(), clientId);
        }

        /// <summary>
        /// Gets the episodes for a podcast subscription (requires MusicBee 3.1+).
        /// </summary>
        public void RequestPodcastEpisodes(string clientId, string subscriptionId)
        {
            var api = Plugin.Instance.MbApiInterface;
            var episodes = new List<AndroidRemote.Model.Entities.PodcastEpisode>();
            string errorMessage = null;

            try
            {
                if (api.Podcasts_GetSubscriptionEpisodes == null)
                {
                    errorMessage = "Requires MusicBee 3.1+";
                }
                else if (string.IsNullOrEmpty(subscriptionId))
                {
                    errorMessage = "Subscription ID required";
                }
                else
                {
                    string[] urls;
                    if (api.Podcasts_GetSubscriptionEpisodes(subscriptionId, out urls) && urls != null)
                    {
                        for (int i = 0; i < urls.Length; i++)
                        {
                            string[] details;
                            if (api.Podcasts_GetSubscriptionEpisode(subscriptionId, i, out details) && details != null && details.Length >= 7)
                            {
                                episodes.Add(new AndroidRemote.Model.Entities.PodcastEpisode
                                {
                                    Id = details[(int)EpisodeMetaDataType.Id],
                                    SubscriptionId = subscriptionId,
                                    Title = details[(int)EpisodeMetaDataType.Title],
                                    DateTime = details[(int)EpisodeMetaDataType.DateTime],
                                    Description = details[(int)EpisodeMetaDataType.Description],
                                    Duration = details[(int)EpisodeMetaDataType.Duration],
                                    IsDownloaded = details[(int)EpisodeMetaDataType.IsDownloaded] == "True",
                                    HasBeenPlayed = details[(int)EpisodeMetaDataType.HasBeenPlayed] == "True",
                                    Url = urls[i]
                                });
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                _logger.Error(ex, "Error getting podcast episodes");
                errorMessage = ex.Message;
            }

            var response = new SocketMessage
            {
                Context = Constants.PodcastEpisodes,
                Data = new
                {
                    success = errorMessage == null,
                    error = errorMessage,
                    subscriptionId = subscriptionId,
                    episodes = episodes,
                    total = episodes.Count
                }
            };

            SendReply(response.ToJsonString(), clientId);
        }

        /// <summary>
        /// Plays a podcast episode by URL.
        /// </summary>
        public void RequestPodcastPlay(string clientId, string episodeUrl)
        {
            var api = Plugin.Instance.MbApiInterface;
            bool success = false;
            string message = "Unknown error";

            try
            {
                if (string.IsNullOrEmpty(episodeUrl))
                {
                    message = "Episode URL required";
                }
                else
                {
                    // Queue the episode and play it
                    success = api.NowPlayingList_QueueNext(episodeUrl);
                    if (success)
                    {
                        api.Player_PlayNextTrack();
                        message = "Playing episode";
                    }
                    else
                    {
                        message = "Failed to queue episode";
                    }
                }
            }
            catch (Exception ex)
            {
                _logger.Error(ex, "Error playing podcast episode");
                message = ex.Message;
            }

            var response = new SocketMessage
            {
                Context = Constants.PodcastPlay,
                Data = new { success, message, url = episodeUrl }
            };

            SendReply(response.ToJsonString(), clientId);
        }
    }
}
