using System.Collections.Generic;
using MusicBeePlugin.AndroidRemote.Events;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Networking;
using NLog;

namespace MusicBeePlugin.AndroidRemote.Services
{
    /// <summary>
    /// Handles playlist operations: list, play, queue.
    /// Extracted from Plugin.cs to reduce monolith size.
    /// </summary>
    public class PlaylistService
    {
        private readonly Logger _logger = LogManager.GetCurrentClassLogger();

        public PlaylistService()
        {
        }

        /// <summary>
        /// Gets all available playlist URLs.
        /// </summary>
        public void GetAvailablePlaylistUrls(string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            api.Playlist_QueryPlaylists();
            var playlists = new List<Playlist>();
            while (true)
            {
                var url = api.Playlist_QueryGetNextPlaylist();

                if (string.IsNullOrEmpty(url)) break;

                var name = api.Playlist_GetName(url);

                var playlist = new Playlist
                {
                    Name = name,
                    Url = url
                };
                playlists.Add(playlist);
            }

            var data = new SocketMessage(Constants.PlaylistList, playlists).ToJsonString();
            EventBus.FireEvent(new MessageEvent(EventType.ReplyAvailable, data, clientId));
        }

        /// <summary>
        /// Gets a paged list of playlists.
        /// </summary>
        public void GetAvailablePlaylistUrls(string clientId, int offset, int limit)
        {
            var api = Plugin.Instance.MbApiInterface;
            api.Playlist_QueryPlaylists();
            var playlists = new List<Playlist>();
            while (true)
            {
                var url = api.Playlist_QueryGetNextPlaylist();

                if (string.IsNullOrEmpty(url)) break;

                var name = api.Playlist_GetName(url);

                var playlist = new Playlist
                {
                    Name = name,
                    Url = url
                };
                playlists.Add(playlist);
            }

            var total = playlists.Count;
            var realLimit = offset + limit > total ? total - offset : limit;
            var message = new SocketMessage
            {
                Context = Constants.PlaylistList,
                Data = new Page<Playlist>
                {
                    Data = offset > total ? new List<Playlist>() : playlists.GetRange(offset, realLimit),
                    Offset = offset,
                    Limit = limit,
                    Total = total
                }
            };
            var messageEvent = new MessageEvent(EventType.ReplyAvailable, message.ToJsonString(), clientId);
            EventBus.FireEvent(messageEvent);
        }

        /// <summary>
        /// Plays a playlist immediately.
        /// </summary>
        public void PlayPlaylist(string clientId, string url)
        {
            var api = Plugin.Instance.MbApiInterface;
            var success = api.Playlist_PlayNow(url);
            var data = new SocketMessage(Constants.PlaylistPlay, success).ToJsonString();
            EventBus.FireEvent(new MessageEvent(EventType.ReplyAvailable, data, clientId));
        }

        /// <summary>
        /// Queue a playlist to the end of the now playing list (appends without replacing).
        /// </summary>
        public void QueuePlaylist(string clientId, string url)
        {
            var api = Plugin.Instance.MbApiInterface;
            bool success = false;
            string[] files;

            if (api.Playlist_QueryFilesEx(url, out files) && files != null && files.Length > 0)
            {
                success = api.NowPlayingList_QueueFilesLast(files);
            }

            var data = new SocketMessage(Constants.PlaylistQueue, success).ToJsonString();
            EventBus.FireEvent(new MessageEvent(EventType.ReplyAvailable, data, clientId));
        }
    }
}
