using System;
using System.Windows.Forms;
using MusicBeePlugin.AndroidRemote.Entities;
using MusicBeePlugin.AndroidRemote.Enumerations;
using MusicBeePlugin.AndroidRemote.Events;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Networking;
using MusicBeePlugin.AndroidRemote.Utilities;
using NLog;
using static MusicBeePlugin.Plugin;

namespace MusicBeePlugin.AndroidRemote.Services
{
    /// <summary>
    /// Handles player control operations: play, pause, stop, next, previous, volume, etc.
    /// Extracted from Plugin.cs to reduce monolith size.
    /// </summary>
    public class PlayerControlService
    {
        private readonly Logger _logger = LogManager.GetCurrentClassLogger();

        public PlayerControlService()
        {
        }

        /// <summary>
        /// When called plays the next track.
        /// </summary>
        public void RequestNextTrack(string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerNext,
                        api.Player_PlayNextTrack()).ToJsonString()));
        }

        /// <summary>
        /// When called stops the playback.
        /// </summary>
        public void RequestStopPlayback(string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerStop,
                        api.Player_Stop()).ToJsonString()));
        }

        /// <summary>
        /// When called changes the play/pause state or starts playing a track if the status is stopped.
        /// </summary>
        public void RequestPlayPauseTrack(string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerPlayPause,
                        api.Player_PlayPause()).ToJsonString()));
        }

        /// <summary>
        /// When called plays the previous track.
        /// </summary>
        public void RequestPreviousTrack(string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerPrevious,
                        api.Player_PlayPreviousTrack()).ToJsonString()));
        }

        /// <summary>
        /// When called plays the next album (MusicBee 3.1+ / API 50+).
        /// </summary>
        public void RequestNextAlbum(string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            bool success = false;
            string message = "Feature not available";

            if (api.Player_PlayNextAlbum != null)
            {
                success = api.Player_PlayNextAlbum();
                message = success ? "Skipped to next album" : "Failed to skip album";
            }
            else
            {
                message = "Requires MusicBee 3.1+";
            }

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerNextAlbum,
                        new { success, message }).ToJsonString()));
        }

        /// <summary>
        /// When called plays the previous album (MusicBee 3.1+ / API 50+).
        /// </summary>
        public void RequestPreviousAlbum(string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            bool success = false;
            string message = "Feature not available";

            if (api.Player_PlayPreviousAlbum != null)
            {
                success = api.Player_PlayPreviousAlbum();
                message = success ? "Skipped to previous album" : "Failed to skip album";
            }
            else
            {
                message = "Requires MusicBee 3.1+";
            }

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerPreviousAlbum,
                        new { success, message }).ToJsonString()));
        }

        /// <summary>
        /// Changes the volume or returns current volume.
        /// </summary>
        /// <param name="volume">Volume 0-100, or -1 to just query</param>
        public void RequestVolumeChange(int volume)
        {
            var api = Plugin.Instance.MbApiInterface;
            if (volume >= 0) api.Player_SetVolume((float)volume / 100);

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerVolume,
                        (int)Math.Round(api.Player_GetVolume() * 100, 1)).ToJsonString()));

            if (api.Player_GetMute()) api.Player_SetMute(false);
        }

        /// <summary>
        /// Changes the player shuffle state.
        /// </summary>
        public void RequestShuffleState(StateAction action)
        {
            var api = Plugin.Instance.MbApiInterface;
            if (action == StateAction.Toggle)
            {
                api.Player_SetShuffle(!api.Player_GetShuffle());
            }

            EventBus.FireEvent(
                new MessageEvent(
                    EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerShuffle,
                        api.Player_GetShuffle()).ToJsonString()));
        }

        /// <summary>
        /// Changes the player mute state.
        /// </summary>
        public void RequestMuteState(StateAction action)
        {
            var api = Plugin.Instance.MbApiInterface;
            if (action == StateAction.Toggle) api.Player_SetMute(!api.Player_GetMute());

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerMute,
                        api.Player_GetMute()).ToJsonString()));
        }

        /// <summary>
        /// Toggles or returns scrobbler state.
        /// </summary>
        public void RequestScrobblerState(StateAction action)
        {
            var api = Plugin.Instance.MbApiInterface;
            if (action == StateAction.Toggle) api.Player_SetScrobbleEnabled(!api.Player_GetScrobbleEnabled());

            EventBus.FireEvent(
                new MessageEvent(
                    EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerScrobble,
                        api.Player_GetScrobbleEnabled()).ToJsonString()));
        }

        /// <summary>
        /// Changes the repeat state: None -> All -> One -> None
        /// </summary>
        public void RequestRepeatState(StateAction action)
        {
            var api = Plugin.Instance.MbApiInterface;
            if (action == StateAction.Toggle)
            {
                switch (api.Player_GetRepeat())
                {
                    case RepeatMode.None:
                        api.Player_SetRepeat(RepeatMode.All);
                        break;
                    case RepeatMode.All:
                        api.Player_SetRepeat(RepeatMode.One);
                        break;
                    case RepeatMode.One:
                        api.Player_SetRepeat(RepeatMode.None);
                        break;
                }
            }

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerRepeat,
                        api.Player_GetRepeat()).ToJsonString()));
        }

        /// <summary>
        /// Toggles or returns AutoDJ state.
        /// </summary>
        public void RequestAutoDjState(StateAction action)
        {
            var api = Plugin.Instance.MbApiInterface;
            if (action == StateAction.Toggle)
            {
                if (!api.Player_GetAutoDjEnabled())
                    api.Player_StartAutoDj();
                else
                    api.Player_EndAutoDj();
            }

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.PlayerAutoDj,
                        api.Player_GetAutoDjEnabled()).ToJsonString()));
        }

        /// <summary>
        /// Gets output devices and sends to client.
        /// </summary>
        public void RequestOutputDevice(string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            api.Player_GetOutputDevices(out var deviceNames, out var activeDeviceName);

            var currentDevices = new OutputDevice(deviceNames, activeDeviceName);

            Plugin.SendReply(new SocketMessage(Constants.PlayerOutput,
                currentDevices).ToJsonString(), clientId);
        }

        /// <summary>
        /// Switches the output device.
        /// </summary>
        public void SwitchOutputDevice(string outputDevice, string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            api.Player_SetOutputDevice(outputDevice);
            RequestOutputDevice(clientId);
        }
    }
}
