using System.Collections.Generic;
using MusicBeePlugin.AndroidRemote.Events;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Networking;
using NLog;
using static MusicBeePlugin.Plugin;
using static MusicBeePlugin.AndroidRemote.Utilities.Utilities;

namespace MusicBeePlugin.AndroidRemote.Services
{
    /// <summary>
    /// Handles now playing list operations.
    /// Extracted from Plugin.cs to reduce monolith size.
    /// </summary>
    public class NowPlayingService
    {
        private readonly Logger _logger = LogManager.GetCurrentClassLogger();

        public NowPlayingService()
        {
        }

        /// <summary>
        /// Get the now playing list ordered from current position.
        /// </summary>
        public void RequestNowPlayingListOrdered(string clientId, int offset = 0, int limit = 100)
        {
            var api = Plugin.Instance.MbApiInterface;
            api.NowPlayingList_QueryFiles(null);

            var tracks = new List<NowPlaying>();
            var position = 1;
            var itemIndex = api.NowPlayingList_GetCurrentIndex();
            while (position <= limit)
            {
                var trackPath = api.NowPlayingList_GetListFileUrl(itemIndex);

                if (string.IsNullOrEmpty(trackPath))
                    break;

                var artist = api.Library_GetFileTag(trackPath, MetaDataType.Artist);
                var album = api.Library_GetFileTag(trackPath, MetaDataType.Album);
                var albumArtist = api.Library_GetFileTag(trackPath, MetaDataType.AlbumArtist);
                var title = api.Library_GetFileTag(trackPath, MetaDataType.TrackTitle);

                if (string.IsNullOrEmpty(title))
                {
                    var index = trackPath.LastIndexOf('\\');
                    title = trackPath.Substring(index + 1);
                }

                NowPlaying track;
                if (IsAndroid(clientId))
                    track = new NowPlaying
                    {
                        Artist = string.IsNullOrEmpty(artist) ? "Unknown Artist" : artist,
                        Album = album,
                        AlbumArtist = albumArtist,
                        Title = title,
                        Position = itemIndex,
                        Path = trackPath
                    };
                else
                    track = new NowPlaying
                    {
                        Artist = artist,
                        Album = album,
                        AlbumArtist = albumArtist,
                        Title = title,
                        Position = itemIndex,
                        Path = trackPath
                    };

                tracks.Add(track);
                itemIndex = api.NowPlayingList_GetNextIndex(position);
                position++;
            }

            var total = tracks.Count;
            var currentPlayingIndex = api.NowPlayingList_GetCurrentIndex();
            var message = new SocketMessage
            {
                Context = Constants.NowPlayingList,
                Data = new
                {
                    data = tracks,
                    offset = offset,
                    limit = limit,
                    total = total,
                    playingIndex = currentPlayingIndex
                }
            };
            var messageEvent = new MessageEvent(EventType.ReplyAvailable, message.ToJsonString(), clientId);
            EventBus.FireEvent(messageEvent);
        }

        /// <summary>
        /// Get a page of the now playing list.
        /// </summary>
        public void RequestNowPlayingListPage(string clientId, int offset = 0, int limit = 4000)
        {
            var api = Plugin.Instance.MbApiInterface;
            api.NowPlayingList_QueryFiles(null);

            var tracks = new List<NowPlaying>();
            var position = 1;
            while (true)
            {
                var trackPath = api.NowPlayingList_QueryGetNextFile();
                if (string.IsNullOrEmpty(trackPath))
                    break;

                var artist = api.Library_GetFileTag(trackPath, MetaDataType.Artist);
                var title = api.Library_GetFileTag(trackPath, MetaDataType.TrackTitle);

                if (string.IsNullOrEmpty(title))
                {
                    var index = trackPath.LastIndexOf('\\');
                    title = trackPath.Substring(index + 1);
                }

                var track = new NowPlaying
                {
                    Artist = string.IsNullOrEmpty(artist) ? "Unknown Artist" : artist,
                    Title = title,
                    Position = position,
                    Path = trackPath
                };

                tracks.Add(track);
                position++;
            }

            var total = tracks.Count;
            var realLimit = offset + limit > total ? total - offset : limit;
            var message = new SocketMessage
            {
                Context = Constants.NowPlayingList,
                Data = new Page<NowPlaying>
                {
                    Data = offset > total ? new List<NowPlaying>() : tracks.GetRange(offset, realLimit),
                    Offset = offset,
                    Limit = limit,
                    Total = total
                }
            };
            var messageEvent = new MessageEvent(EventType.ReplyAvailable, message.ToJsonString(), clientId);
            EventBus.FireEvent(messageEvent);
        }

        /// <summary>
        /// Get the full now playing list.
        /// </summary>
        public void RequestNowPlayingList(string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            api.NowPlayingList_QueryFiles(null);

            var trackList = new List<NowPlayingListTrack>();
            var position = 1;
            while (position <= 5000)
            {
                var trackPath = api.NowPlayingList_QueryGetNextFile();
                if (string.IsNullOrEmpty(trackPath))
                    break;

                var artist = api.Library_GetFileTag(trackPath, MetaDataType.Artist);
                var title = api.Library_GetFileTag(trackPath, MetaDataType.TrackTitle);

                if (string.IsNullOrEmpty(title))
                {
                    var index = trackPath.LastIndexOf('\\');
                    title = trackPath.Substring(index + 1);
                }

                var track = new NowPlayingListTrack
                {
                    Artist = string.IsNullOrEmpty(artist) ? "Unknown Artist" : artist,
                    Title = title,
                    Position = position,
                    Path = trackPath
                };
                trackList.Add(track);
                position++;
            }

            var jsonString = new SocketMessage(Constants.NowPlayingList, trackList).ToJsonString();
            var messageEvent = new MessageEvent(EventType.ReplyAvailable, jsonString, clientId);
            EventBus.FireEvent(messageEvent);
        }

        /// <summary>
        /// Request or set play position.
        /// </summary>
        public void RequestPlayPosition(string request)
        {
            var api = Plugin.Instance.MbApiInterface;
            if (!request.Contains("status"))
            {
                if (int.TryParse(request, out var newPosition)) api.Player_SetPosition(newPosition);
            }

            var currentPosition = api.Player_GetPosition();
            var totalDuration = api.NowPlaying_GetDuration();

            var position = new
            {
                current = currentPosition,
                total = totalDuration
            };

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.NowPlayingPosition, position).ToJsonString()));
        }

        /// <summary>
        /// Play a track from the now playing list.
        /// </summary>
        public void NowPlayingPlay(string index, bool isAndroid)
        {
            var api = Plugin.Instance.MbApiInterface;
            var result = false;
            if (int.TryParse(index, out var trackIndex))
            {
                api.NowPlayingList_QueryFiles(null);
                var trackToPlay = api.NowPlayingList_GetListFileUrl(isAndroid ? trackIndex - 1 : trackIndex);
                if (!string.IsNullOrEmpty(trackToPlay))
                    result = api.NowPlayingList_PlayNow(trackToPlay);
            }

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.NowPlayingListPlay,
                        result).ToJsonString()));
        }

        /// <summary>
        /// Remove a track from the now playing list.
        /// </summary>
        public void NowPlayingListRemoveTrack(int index, string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            var reply = new
            {
                success = api.NowPlayingList_RemoveAt(index),
                index
            };
            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.NowPlayingListRemove,
                        reply).ToJsonString(), clientId));
        }

        /// <summary>
        /// Move a track in the now playing list.
        /// </summary>
        public void RequestNowPlayingMove(string clientId, int from, int to)
        {
            var api = Plugin.Instance.MbApiInterface;
            int[] aFrom = { from };
            int dIn;
            if (from > to)
                dIn = to - 1;
            else
                dIn = to;

            var result = api.NowPlayingList_MoveFiles(aFrom, dIn);

            var reply = new
            {
                success = result,
                from,
                to
            };
            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.NowPlayingListMove, reply).ToJsonString(), clientId));
        }

        /// <summary>
        /// Request now playing track cover.
        /// </summary>
        public void RequestNowPlayingTrackCover()
        {
            var api = Plugin.Instance.MbApiInterface;
            if (!string.IsNullOrEmpty(api.NowPlaying_GetArtwork()))
            {
                EventBus.FireEvent(new MessageEvent(EventType.NowPlayingCoverChange,
                    api.NowPlaying_GetArtwork()));
            }
            else if (api.ApiRevision >= 17)
            {
                var cover = api.NowPlaying_GetDownloadedArtwork();
                EventBus.FireEvent(new MessageEvent(EventType.NowPlayingCoverChange,
                    !string.IsNullOrEmpty(cover) ? cover : string.Empty));
            }
            else
            {
                EventBus.FireEvent(new MessageEvent(EventType.NowPlayingCoverChange, string.Empty));
            }
        }

        /// <summary>
        /// Request now playing track lyrics.
        /// </summary>
        public void RequestNowPlayingTrackLyrics()
        {
            var api = Plugin.Instance.MbApiInterface;
            var lyrics = api.NowPlaying_GetLyrics();

            if (!string.IsNullOrEmpty(lyrics))
            {
                //no additional actions
            }
            else if (api.ApiRevision >= 17)
            {
                lyrics = api.NowPlaying_GetDownloadedLyrics();
            }
            else
            {
                lyrics = string.Empty;
            }

            //update LyricsCoverModel cache
            EventBus.FireEvent(new MessageEvent(EventType.NowPlayingLyricsChange, lyrics));

            Plugin.BroadcastLyrics(lyrics);
        }
    }
}
