using System;
using System.Globalization;
using System.Windows.Forms;
using MusicBeePlugin.AndroidRemote.Enumerations;
using MusicBeePlugin.AndroidRemote.Events;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Networking;
using NLog;
using static MusicBeePlugin.Plugin;

namespace MusicBeePlugin.AndroidRemote.Services
{
    /// <summary>
    /// Handles love/ban status and rating operations for tracks.
    /// Extracted from Plugin.cs to reduce monolith size.
    /// </summary>
    public class LoveRatingService
    {
        private readonly Logger _logger = LogManager.GetCurrentClassLogger();

        public LoveRatingService()
        {
        }

        /// <summary>
        /// Request or toggle the love status.
        /// </summary>
        public void RequestLoveStatus(string action, string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            var hwnd = api.MB_GetWindowHandle();
            var mb = (Form)Control.FromHandle(hwnd);

            if (action.Equals("toggle", StringComparison.OrdinalIgnoreCase))
            {
                if (GetLfmStatus() == LastfmStatus.Love || GetLfmStatus() == LastfmStatus.Ban)
                    mb.Invoke(new MethodInvoker(SetLfmNormalStatus));
                else
                    mb.Invoke(new MethodInvoker(SetLfmLoveStatus));
            }
            else if (action.Equals("love", StringComparison.OrdinalIgnoreCase))
            {
                mb.Invoke(new MethodInvoker(SetLfmLoveStatus));
            }
            else if (action.Equals("ban", StringComparison.OrdinalIgnoreCase))
            {
                mb.Invoke(new MethodInvoker(SetLfmLoveBan));
            }
            else if (action.Equals("normal", StringComparison.OrdinalIgnoreCase))
            {
                mb.Invoke(new MethodInvoker(SetLfmNormalStatus));
            }
            else
            {
                SendLfmStatusMessage(GetLfmStatus());
            }
        }

        /// <summary>
        /// Set love status to normal (clear).
        /// </summary>
        private void SetLfmNormalStatus()
        {
            var api = Plugin.Instance.MbApiInterface;
            var fileUrl = api.NowPlaying_GetFileUrl();
            var success = api.Library_SetFileTag(fileUrl, MetaDataType.RatingLove, "");
            if (success)
            {
                api.Library_CommitTagsToFile(fileUrl);
                api.MB_RefreshPanels();
                SendLfmStatusMessage(LastfmStatus.Normal);
            }
        }

        /// <summary>
        /// Set love status.
        /// </summary>
        private void SetLfmLoveStatus()
        {
            var api = Plugin.Instance.MbApiInterface;
            var fileUrl = api.NowPlaying_GetFileUrl();
            var success = api.Library_SetFileTag(fileUrl, MetaDataType.RatingLove, "Llfm");
            if (success)
            {
                api.Library_CommitTagsToFile(fileUrl);
                api.MB_RefreshPanels();
                SendLfmStatusMessage(LastfmStatus.Love);
            }
        }

        /// <summary>
        /// Set ban status.
        /// </summary>
        private void SetLfmLoveBan()
        {
            var api = Plugin.Instance.MbApiInterface;
            var fileUrl = api.NowPlaying_GetFileUrl();
            var success = api.Library_SetFileTag(fileUrl, MetaDataType.RatingLove, "Blfm");
            if (success)
            {
                api.Library_CommitTagsToFile(fileUrl);
                api.MB_RefreshPanels();
                SendLfmStatusMessage(LastfmStatus.Ban);
            }
        }

        /// <summary>
        /// Get the current LFM status.
        /// </summary>
        public LastfmStatus GetLfmStatus()
        {
            var api = Plugin.Instance.MbApiInterface;
            LastfmStatus lastfmStatus;
            var apiReply = api.NowPlaying_GetFileTag(MetaDataType.RatingLove);
            if (apiReply.Equals("L") || apiReply.Equals("lfm") || apiReply.Equals("Llfm"))
                lastfmStatus = LastfmStatus.Love;
            else if (apiReply.Equals("B") || apiReply.Equals("Blfm"))
                lastfmStatus = LastfmStatus.Ban;
            else
                lastfmStatus = LastfmStatus.Normal;

            return lastfmStatus;
        }

        /// <summary>
        /// Send LFM status message.
        /// </summary>
        private static void SendLfmStatusMessage(LastfmStatus lastfmStatus)
        {
            var data = new SocketMessage(Constants.NowPlayingLfmRating, lastfmStatus).ToJsonString();
            EventBus.FireEvent(new MessageEvent(EventType.ReplyAvailable, data));
        }

        /// <summary>
        /// Request or set the track rating.
        /// </summary>
        public void RequestTrackRating(string rating, string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            var currentTrack = api.NowPlaying_GetFileUrl();
            var decimalSeparator = Convert.ToChar(CultureInfo.CurrentCulture.NumberFormat.NumberDecimalSeparator);

            try
            {
                rating = rating.Replace('.', decimalSeparator);
                float fRating;
                if (!float.TryParse(rating, out fRating)) fRating = -1;

                if ((fRating >= 0 && fRating <= 5) || rating == "")
                {
                    var value = rating == ""
                        ? rating.ToString(CultureInfo.CurrentCulture)
                        : fRating.ToString(CultureInfo.CurrentCulture);
                    api.Library_SetFileTag(currentTrack, MetaDataType.Rating, value);
                    api.Library_CommitTagsToFile(currentTrack);
                    api.Player_GetShowRatingTrack();
                    api.MB_RefreshPanels();
                }

                rating = api.Library_GetFileTag(currentTrack, MetaDataType.Rating).Replace(decimalSeparator, '.');

                EventBus.FireEvent(
                    new MessageEvent(EventType.ReplyAvailable,
                        new SocketMessage(Constants.NowPlayingRating,
                            rating).ToJsonString()));
            }
            catch (Exception e)
            {
                _logger.Error(e, "On Rating call");
            }
        }

        /// <summary>
        /// Sets the rating for a library track by file path.
        /// Used by two-way sync (P2.3).
        /// </summary>
        public bool SetLibraryTrackRating(string filePath, float rating)
        {
            try
            {
                var api = Plugin.Instance.MbApiInterface;
                var hwnd = api.MB_GetWindowHandle();
                var mb = (Form)Control.FromHandle(hwnd);

                bool success = false;
                mb.Invoke(new MethodInvoker(() =>
                {
                    var value = rating.ToString(CultureInfo.CurrentCulture);
                    success = api.Library_SetFileTag(filePath, MetaDataType.Rating, value);
                    if (success)
                    {
                        api.Library_CommitTagsToFile(filePath);
                        api.MB_RefreshPanels();
                    }
                }));

                _logger.Debug("SetLibraryTrackRating: {0} -> {1}, success={2}", filePath, rating, success);
                return success;
            }
            catch (Exception ex)
            {
                _logger.Error(ex, "SetLibraryTrackRating failed for {0}", filePath);
                return false;
            }
        }

        /// <summary>
        /// Sets the love/ban status for a library track by file path.
        /// Used by two-way sync (P2.4).
        /// </summary>
        public bool SetLibraryTrackLove(string filePath, string status)
        {
            try
            {
                var api = Plugin.Instance.MbApiInterface;
                var hwnd = api.MB_GetWindowHandle();
                var mb = (Form)Control.FromHandle(hwnd);

                // Map status to MusicBee value
                string mbValue;
                switch (status.ToLowerInvariant())
                {
                    case "love":
                        mbValue = "Llfm";
                        break;
                    case "ban":
                        mbValue = "Blfm";
                        break;
                    default:
                        mbValue = "";
                        break;
                }

                bool success = false;
                mb.Invoke(new MethodInvoker(() =>
                {
                    success = api.Library_SetFileTag(filePath, MetaDataType.RatingLove, mbValue);
                    if (success)
                    {
                        api.Library_CommitTagsToFile(filePath);
                        api.MB_RefreshPanels();
                    }
                }));

                _logger.Debug("SetLibraryTrackLove: {0} -> {1}, success={2}", filePath, status, success);
                return success;
            }
            catch (Exception ex)
            {
                _logger.Error(ex, "SetLibraryTrackLove failed for {0}", filePath);
                return false;
            }
        }
    }
}
