using System;
using System.IO;
using NLog;
using NLog.Config;
using NLog.Targets;

namespace MusicBeePlugin.AndroidRemote.Services
{
    /// <summary>
    /// Handles logging configuration with enhanced debug verbosity.
    /// Extracted from Plugin.cs to reduce monolith size.
    /// </summary>
    public static class LoggingService
    {
        private static readonly Logger _logger = LogManager.GetCurrentClassLogger();

        /// <summary>
        /// Initializes the logging configuration with appropriate verbosity.
        /// When debug is enabled, provides maximum verbosity including:
        /// - Trace level logging
        /// - Method names and line numbers
        /// - Thread IDs
        /// - Full stack traces for exceptions
        /// </summary>
        /// <param name="logFilePath">Path to the log file</param>
        /// <param name="debugEnabled">True for maximum verbosity</param>
        public static void Initialize(string logFilePath, bool debugEnabled)
        {
            var config = new LoggingConfiguration();
            var logLevel = debugEnabled ? LogLevel.Trace : LogLevel.Error;

            // Debug mode: smaller files, more archives for granular history
            // Normal mode: larger files, fewer archives
            var archiveSize = debugEnabled ? 1048576 : 4194304;  // 1MB debug, 4MB normal
            var maxArchives = debugEnabled ? 10 : 3;             // More archives for debug

            // Build archive filename pattern
            var logDir = Path.GetDirectoryName(logFilePath) ?? "";
            var logName = Path.GetFileNameWithoutExtension(logFilePath);
            var logExt = Path.GetExtension(logFilePath);
            var archiveFilePath = Path.Combine(logDir, $"{logName}.{{#}}{logExt}");

            // Enhanced layout for debug mode - includes everything useful for debugging
            var debugLayout = "${longdate} [${level:uppercase=true:padding=-5}] " +
                              "[Thread:${threadid:padding=3}] " +
                              "${callsite:className=true:methodName=true:includeSourcePath=false}:${callsite-linenumber}" +
                              "${newline}  ${message}" +
                              "${onexception:${newline}  EXCEPTION: ${exception:format=ToString,StackTrace}}" +
                              "${newline}";

            // Compact layout for production - just essentials
            var productionLayout = "${longdate} [${level:uppercase=true}] ${logger}: ${message}" +
                                   "${onexception:${newline}  ${exception:format=Message}}${newline}";

            var fileTarget = new FileTarget("file")
            {
                FileName = logFilePath,
                ArchiveFileName = archiveFilePath,
                ArchiveAboveSize = archiveSize,
                ArchiveEvery = FileArchivePeriod.Day,
                MaxArchiveFiles = maxArchives,
                Layout = debugEnabled ? debugLayout : productionLayout
            };

            config.AddTarget(fileTarget);
            config.AddRule(logLevel, LogLevel.Fatal, fileTarget);

#if DEBUG
            // Console output for development
            var consoleTarget = new ColoredConsoleTarget("console")
            {
                Layout = "${time} [${level:uppercase=true:padding=-5}] ${callsite:methodName=true}: ${message}"
            };
            config.AddTarget(consoleTarget);
            config.AddRule(LogLevel.Trace, LogLevel.Fatal, consoleTarget);

            // Visual Studio output window
            var debuggerTarget = new DebuggerTarget("debugger")
            {
                Layout = debugLayout
            };
            config.AddTarget(debuggerTarget);
            config.AddRule(LogLevel.Trace, LogLevel.Fatal, debuggerTarget);
#endif

            LogManager.Configuration = config;
            LogManager.ReconfigExistingLoggers();

            // Log startup info
            _logger.Info("=== LOGGING INITIALIZED ===");
            _logger.Info($"Log file: {logFilePath}");
            _logger.Info($"Debug mode: {debugEnabled}");
            _logger.Info($"Log level: {logLevel}");
            _logger.Info($"Archive size: {archiveSize / 1024} KB");
            _logger.Info($"Max archives: {maxArchives}");
        }

        /// <summary>
        /// Switches logging verbosity at runtime.
        /// </summary>
        public static void SetDebugMode(bool enabled, string logFilePath)
        {
            _logger.Info($"Switching debug mode to: {enabled}");
            Initialize(logFilePath, enabled);
        }

        /// <summary>
        /// Logs method entry for tracing (only in debug builds).
        /// </summary>
        [System.Diagnostics.Conditional("DEBUG")]
        public static void TraceMethodEntry(Logger logger, string methodName, params object[] args)
        {
            if (args.Length > 0)
                logger.Trace($">>> {methodName}({string.Join(", ", args)})");
            else
                logger.Trace($">>> {methodName}()");
        }

        /// <summary>
        /// Logs method exit for tracing (only in debug builds).
        /// </summary>
        [System.Diagnostics.Conditional("DEBUG")]
        public static void TraceMethodExit(Logger logger, string methodName, object result = null)
        {
            if (result != null)
                logger.Trace($"<<< {methodName} => {result}");
            else
                logger.Trace($"<<< {methodName}");
        }
    }
}
