using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml.Linq;
using MusicBeePlugin.AndroidRemote.Events;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Networking;
using MusicBeePlugin.AndroidRemote.Settings;
using MusicBeePlugin.AndroidRemote.Utilities;
using NLog;
using static MusicBeePlugin.Plugin;

namespace MusicBeePlugin.AndroidRemote.Services
{
    /// <summary>
    /// Handles library search operations: artists, albums, genres, tracks.
    /// Extracted from Plugin.cs to reduce monolith size.
    /// </summary>
    public class LibrarySearchService
    {
        private readonly Logger _logger = LogManager.GetCurrentClassLogger();

        public LibrarySearchService()
        {
        }

        /// <summary>
        /// Create XML filter for library queries.
        /// </summary>
        public static string XmlFilter(string[] tags, string query, bool isStrict,
            SearchSource source = SearchSource.None)
        {
            short src;
            if (source != SearchSource.None)
            {
                src = (short)source;
            }
            else
            {
                var userDefaults = UserSettings.Instance.Source != SearchSource.None;
                src = (short)
                    (userDefaults
                        ? UserSettings.Instance.Source
                        : SearchSource.Library);
            }

            var filter = new XElement("Source",
                new XAttribute("Type", src));

            var conditions = new XElement("Conditions",
                new XAttribute("CombineMethod", "Any"));
            foreach (var tag in tags)
            {
                var condition = new XElement("Condition",
                    new XAttribute("Field", tag),
                    new XAttribute("Comparison", isStrict ? "Is" : "Contains"),
                    new XAttribute("Value", query));
                conditions.Add(condition);
            }

            filter.Add(conditions);

            return filter.ToString();
        }

        /// <summary>
        /// Search for albums by name.
        /// </summary>
        public void LibrarySearchAlbums(string albumName, string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            var filter = XmlFilter(new[] { "Album" }, albumName, false);

            var albums = new List<AlbumData>();

            if (api.Library_QueryLookupTable("album", "albumartist" + '\0' + "album", filter))
                try
                {
                    foreach (
                        var entry in
                        new List<string>(api.Library_QueryGetLookupTableValue(null)
                            .Split(new[] { "\0\0" }, StringSplitOptions.None)))
                    {
                        if (string.IsNullOrEmpty(entry)) continue;
                        var albumInfo = entry.Split('\0');
                        if (albumInfo.Length < 2) continue;

                        var current = albumInfo.Length == 3
                            ? new AlbumData(albumInfo[1], albumInfo[2])
                            : new AlbumData(albumInfo[0], albumInfo[1]);
                        if (current.Album.IndexOf(albumName, StringComparison.OrdinalIgnoreCase) < 0) continue;

                        if (!albums.Contains(current))
                            albums.Add(current);
                        else
                            albums.ElementAt(albums.IndexOf(current)).IncreaseCount();
                    }
                }
                catch (IndexOutOfRangeException)
                {
                }

            api.Library_QueryLookupTable(null, null, null);

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.LibrarySearchAlbum,
                        albums).ToJsonString(), clientId));
        }

        /// <summary>
        /// Get albums for a specific artist.
        /// </summary>
        public void LibraryGetArtistAlbums(string artist, string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            var albumList = new List<AlbumData>();
            if (api.Library_QueryFiles(XmlFilter(new[] { "ArtistPeople" }, artist, true)))
                while (true)
                {
                    var currentFile = api.Library_QueryGetNextFile();
                    if (string.IsNullOrEmpty(currentFile)) break;
                    var current = new AlbumData(api.Library_GetFileTag(currentFile, MetaDataType.AlbumArtist),
                        api.Library_GetFileTag(currentFile, MetaDataType.Album));
                    if (!albumList.Contains(current))
                        albumList.Add(current);
                    else
                        albumList.ElementAt(albumList.IndexOf(current)).IncreaseCount();
                }

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.LibraryArtistAlbums,
                        albumList).ToJsonString(), clientId));
        }

        /// <summary>
        /// Search for artists by name.
        /// </summary>
        public void LibrarySearchArtist(string artist, string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            var artistList = new List<ArtistData>();

            if (api.Library_QueryLookupTable("artist", "count",
                    XmlFilter(new[] { "ArtistPeople" }, artist, false)))
                artistList.AddRange(api.Library_QueryGetLookupTableValue(null)
                    .Split(new[] { "\0\0" }, StringSplitOptions.None)
                    .Select(entry => entry.Split('\0'))
                    .Where(parts => parts.Length >= 2)
                    .Select(artistInfo => new ArtistData(artistInfo[0], int.Parse(artistInfo[1]))));

            api.Library_QueryLookupTable(null, null, null);

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.LibrarySearchArtist, artistList).ToJsonString(), clientId));
        }

        /// <summary>
        /// Play all library tracks.
        /// </summary>
        public void LibraryPlayAll(string clientId, bool shuffle = false)
        {
            var api = Plugin.Instance.MbApiInterface;
            bool success;

            if (shuffle)
            {
                success = api.NowPlayingList_PlayLibraryShuffled();
            }
            else
            {
                if (api.Player_GetAutoDjEnabled()) api.Player_EndAutoDj();

                api.Player_SetShuffle(false);

                string[] songsList = null;
                api.Library_QueryFilesEx(null, out songsList);
                if (songsList.Length > 0)
                {
                    api.NowPlayingList_Clear();
                    success = api.NowPlayingList_QueueFilesNext(songsList);
                    api.Player_PlayNextTrack();
                }
                else
                {
                    success = false;
                }
            }

            var data = new SocketMessage(Constants.LibraryPlayAll, success).ToJsonString();
            EventBus.FireEvent(new MessageEvent(EventType.ReplyAvailable, data, clientId));
        }

        /// <summary>
        /// Get player status for a client.
        /// </summary>
        public void RequestPlayerStatus(string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            var status = new Dictionary<string, object>
            {
                [Constants.PlayerRepeat] = api.Player_GetRepeat().ToString(),
                [Constants.PlayerMute] = api.Player_GetMute(),
                [Constants.PlayerShuffle] = Authenticator.ClientProtocolMisMatch(clientId)
                    ? (object)api.Player_GetShuffle()
                    : Plugin.Instance.GetShuffleState(),
                [Constants.PlayerScrobble] = api.Player_GetScrobbleEnabled(),
                [Constants.PlayerState] = api.Player_GetPlayState().ToString(),
                [Constants.PlayerVolume] = ((int)Math.Round(api.Player_GetVolume() * 100, 1)).ToString(
                    System.Globalization.CultureInfo.InvariantCulture)
            };

            var data = new SocketMessage(Constants.PlayerStatus, status).ToJsonString();
            EventBus.FireEvent(new MessageEvent(EventType.ReplyAvailable, data, clientId));
        }
    }
}
