using System;
using System.Collections.Generic;
using System.Linq;
using MusicBeePlugin.AndroidRemote.Events;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Networking;
using MusicBeePlugin.AndroidRemote.Utilities;
using NLog;
using static MusicBeePlugin.Plugin;

namespace MusicBeePlugin.AndroidRemote.Services
{
    /// <summary>
    /// Handles library browsing operations: genres, artists, albums, tracks.
    /// Extracted from Plugin.cs to reduce monolith size.
    /// </summary>
    public class LibraryBrowseService
    {
        private readonly Logger _logger = LogManager.GetCurrentClassLogger();

        public LibraryBrowseService()
        {
        }

        /// <summary>
        /// Get artists for a specific genre.
        /// </summary>
        public void LibraryGetGenreArtists(string genre, string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            var artistList = new List<ArtistData>();

            if (api.Library_QueryLookupTable("artist", "count", LibrarySearchService.XmlFilter(new[] { "Genre" }, genre, true)))
                artistList.AddRange(api.Library_QueryGetLookupTableValue(null)
                    .Split(new[] { "\0\0" }, StringSplitOptions.None)
                    .Select(entry => entry.Split('\0'))
                    .Where(parts => parts.Length >= 2)
                    .Select(artistInfo => new ArtistData(artistInfo[0], int.Parse(artistInfo[1]))));

            api.Library_QueryLookupTable(null, null, null);
            var message = new SocketMessage(Constants.LibraryGenreArtists, artistList).ToJsonString();
            var messageEvent = new MessageEvent(EventType.ReplyAvailable, message, clientId);
            EventBus.FireEvent(messageEvent);
        }

        /// <summary>
        /// Search for genres by name.
        /// </summary>
        public void LibrarySearchGenres(string genre, string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            var genreList = new List<GenreData>();
            var query = LibrarySearchService.XmlFilter(new[] { "Genre" }, genre, false);
            if (api.Library_QueryLookupTable("genre", "count", query))
                genreList.AddRange(api.Library_QueryGetLookupTableValue(null)
                    .Split(new[] { "\0\0" }, StringSplitOptions.None)
                    .Select(entry => entry.Split(new[] { '\0' }, StringSplitOptions.None))
                    .Where(parts => parts.Length >= 2)
                    .Select(genreInfo => new GenreData(genreInfo[0], int.Parse(genreInfo[1]))));

            api.Library_QueryLookupTable(null, null, null);

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.LibrarySearchGenre,
                        genreList).ToJsonString(), clientId));
        }

        /// <summary>
        /// Browse all genres with pagination.
        /// </summary>
        public void LibraryBrowseGenres(string clientId, int offset = 0, int limit = 4000)
        {
            var api = Plugin.Instance.MbApiInterface;
            var genres = new List<GenreData>();
            if (api.Library_QueryLookupTable("genre", "count", null))
                genres.AddRange(api.Library_QueryGetLookupTableValue(null)
                    .Split(new[] { "\0\0" }, StringSplitOptions.None)
                    .Select(entry => entry.Split(new[] { '\0' }, StringSplitOptions.None))
                    .Where(parts => parts.Length >= 2)
                    .Select(genreInfo => new GenreData(genreInfo[0].Cleanup(), int.Parse(genreInfo[1]))));

            api.Library_QueryLookupTable(null, null, null);

            // Calculate artist counts for each genre
            var artistCounts = new Dictionary<string, HashSet<string>>();
            if (api.Library_QueryLookupTable("genre", "genre" + '\0' + "artist", null))
            {
                try
                {
                    var genreArtistData = api.Library_QueryGetLookupTableValue(null)
                        .Split(new[] { "\0\0" }, StringSplitOptions.None)
                        .Where(s => !string.IsNullOrEmpty(s))
                        .Select(s => s.Trim().Split('\0'))
                        .Where(parts => parts.Length >= 2);

                    foreach (var parts in genreArtistData)
                    {
                        var genreName = parts[0].Cleanup();
                        var artist = parts[1].Cleanup();
                        if (!artistCounts.ContainsKey(genreName))
                            artistCounts[genreName] = new HashSet<string>();
                        artistCounts[genreName].Add(artist);
                    }
                }
                catch (Exception ex)
                {
                    _logger.Error(ex, "While calculating artist counts for genres");
                }
            }
            api.Library_QueryLookupTable(null, null, null);

            // Update genres with artist counts
            foreach (var genreItem in genres)
            {
                if (artistCounts.TryGetValue(genreItem.Genre, out var artists))
                    genreItem.ArtistCount = artists.Count;
            }

            var total = genres.Count;
            var realLimit = offset + limit > total ? total - offset : limit;

            var message = new SocketMessage
            {
                Context = Constants.LibraryBrowseGenres,
                Data = new Page<GenreData>
                {
                    Data = offset > total ? new List<GenreData>() : genres.GetRange(offset, realLimit),
                    Offset = offset,
                    Limit = limit,
                    Total = total
                }
            };

            var messageEvent = new MessageEvent(EventType.ReplyAvailable, message.ToJsonString(), clientId);
            EventBus.FireEvent(messageEvent);
        }

        /// <summary>
        /// Browse all artists with pagination.
        /// </summary>
        public void LibraryBrowseArtists(string clientId, int offset = 0, int limit = 4000, bool albumArtists = false)
        {
            var api = Plugin.Instance.MbApiInterface;
            var artists = new List<ArtistData>();
            var artistType = "artist";
            if (albumArtists) artistType = "albumartist";

            if (api.Library_QueryLookupTable(artistType, "count", null))
                artists.AddRange(api.Library_QueryGetLookupTableValue(null)
                    .Split(new[] { "\0\0" }, StringSplitOptions.None)
                    .Select(entry => entry.Split('\0'))
                    .Where(parts => parts.Length >= 2)
                    .Select(artistInfo => new ArtistData(artistInfo[0].Cleanup(), int.Parse(artistInfo[1]))));

            api.Library_QueryLookupTable(null, null, null);

            // Calculate album counts for each artist
            var albumCounts = new Dictionary<string, int>();
            if (api.Library_QueryLookupTable("album", artistType + '\0' + "album", null))
            {
                try
                {
                    var albumData = api.Library_QueryGetLookupTableValue(null)
                        .Split(new[] { "\0\0" }, StringSplitOptions.None)
                        .Where(s => !string.IsNullOrEmpty(s))
                        .Select(s => s.Trim().Split('\0'))
                        .Where(parts => parts.Length >= 2);

                    foreach (var parts in albumData)
                    {
                        var artist = parts[0].Cleanup();
                        if (!albumCounts.ContainsKey(artist))
                            albumCounts[artist] = 0;
                        albumCounts[artist]++;
                    }
                }
                catch (Exception ex)
                {
                    _logger.Error(ex, "While calculating album counts for artists");
                }
            }
            api.Library_QueryLookupTable(null, null, null);

            // Update artists with album counts
            foreach (var artist in artists)
            {
                if (albumCounts.TryGetValue(artist.Artist, out var albumCount))
                    artist.AlbumCount = albumCount;
            }

            var total = artists.Count;
            var realLimit = offset + limit > total ? total - offset : limit;
            var message = new SocketMessage
            {
                Context = Constants.LibraryBrowseArtists,
                Data = new Page<ArtistData>
                {
                    Data = offset > total ? new List<ArtistData>() : artists.GetRange(offset, realLimit),
                    Offset = offset,
                    Limit = limit,
                    Total = total
                }
            };

            var messageEvent = new MessageEvent(EventType.ReplyAvailable, message.ToJsonString(), clientId);
            EventBus.FireEvent(messageEvent);
        }

        /// <summary>
        /// Browse all albums with pagination.
        /// </summary>
        public void LibraryBrowseAlbums(string clientId, int offset = 0, int limit = 4000)
        {
            var api = Plugin.Instance.MbApiInterface;
            var albums = new List<AlbumData>();

            if (api.Library_QueryLookupTable("album", "albumartist" + '\0' + "album", null))
                try
                {
                    List<AlbumData> data;
                    try
                    {
                        data = api.Library_QueryGetLookupTableValue(null)
                            .Split(new[] { "\0\0" }, StringSplitOptions.None)
                            .Where(s => !string.IsNullOrEmpty(s))
                            .Select(s => s.Trim())
                            .Select(Plugin.CreateAlbum)
                            .Distinct()
                            .ToList();
                    }
                    catch (IndexOutOfRangeException ex)
                    {
                        _logger.Error(ex, "While loading album data");
                        data = new List<AlbumData>();
                    }

                    albums.AddRange(data);
                }
                catch (Exception ex)
                {
                    _logger.Error(ex, "While loading album data");
                }

            api.Library_QueryLookupTable(null, null, null);

            var total = albums.Count;
            var realLimit = offset + limit > total ? total - offset : limit;

            var message = new SocketMessage
            {
                Context = Constants.LibraryBrowseAlbums,
                Data = new Page<AlbumData>
                {
                    Data = offset > total ? new List<AlbumData>() : albums.GetRange(offset, realLimit),
                    Offset = offset,
                    Limit = limit,
                    Total = total
                }
            };
            var messageEvent = new MessageEvent(EventType.ReplyAvailable, message.ToJsonString(), clientId);
            EventBus.FireEvent(messageEvent);
        }

        /// <summary>
        /// Search for tracks by title.
        /// </summary>
        public void LibrarySearchTitle(string title, string clientId)
        {
            var api = Plugin.Instance.MbApiInterface;
            var tracks = new List<Track>();
            if (api.Library_QueryFiles(LibrarySearchService.XmlFilter(new[] { "Title" }, title, false)))
                while (true)
                {
                    var currentTrack = api.Library_QueryGetNextFile();
                    if (string.IsNullOrEmpty(currentTrack)) break;

                    int.TryParse(api.Library_GetFileTag(currentTrack, MetaDataType.TrackNo), out var trackNumber);
                    var src = currentTrack;

                    tracks.Add(new Track(api.Library_GetFileTag(currentTrack, MetaDataType.Artist),
                        api.Library_GetFileTag(currentTrack, MetaDataType.TrackTitle),
                        trackNumber, src));
                }

            api.Library_QueryLookupTable(null, null, null);
            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.LibrarySearchTitle, tracks).ToJsonString(), clientId));
        }

        /// <summary>
        /// Get tracks for a specific album.
        /// </summary>
        public void LibraryGetAlbumTracks(string album, string client)
        {
            var api = Plugin.Instance.MbApiInterface;
            var trackList = new List<Track>();
            if (api.Library_QueryFiles(LibrarySearchService.XmlFilter(new[] { "Album" }, album, true)))
                while (true)
                {
                    var currentTrack = api.Library_QueryGetNextFile();
                    if (string.IsNullOrEmpty(currentTrack)) break;

                    int.TryParse(api.Library_GetFileTag(currentTrack, MetaDataType.TrackNo), out var trackNumber);
                    int.TryParse(api.Library_GetFileTag(currentTrack, MetaDataType.DiscNo), out var discNumber);
                    var src = currentTrack;

                    trackList.Add(new Track
                    {
                        Artist = api.Library_GetFileTag(currentTrack, MetaDataType.Artist),
                        Title = api.Library_GetFileTag(currentTrack, MetaDataType.TrackTitle),
                        AlbumArtist = api.Library_GetFileTag(currentTrack, MetaDataType.AlbumArtist),
                        Disc = discNumber,
                        TrackNo = trackNumber,
                        Src = src
                    });
                }

            EventBus.FireEvent(
                new MessageEvent(EventType.ReplyAvailable,
                    new SocketMessage(Constants.LibraryAlbumTracks, trackList).ToJsonString(), client));
        }
    }
}
