using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using MusicBeePlugin.AndroidRemote.Events;
using MusicBeePlugin.AndroidRemote.Model;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Networking;
using NLog;
using ServiceStack;
using static MusicBeePlugin.Plugin;
using static MusicBeePlugin.AndroidRemote.Utilities.Utilities;

namespace MusicBeePlugin.AndroidRemote.Services
{
    /// <summary>
    /// Handles album cover operations: caching, retrieval, and requests.
    /// Extracted from Plugin.cs to reduce monolith size.
    /// </summary>
    public class CoverService
    {
        private readonly Logger _logger = LogManager.GetCurrentClassLogger();

        public CoverService()
        {
        }

        /// <summary>
        /// Cache the cover for a track.
        /// </summary>
        public string CacheCover(string track)
        {
            var api = Plugin.Instance.MbApiInterface;
            var locations = PictureLocations.EmbedInFile |
                            PictureLocations.LinkToSource |
                            PictureLocations.LinkToOrganisedCopy;

            var pictureUrl = string.Empty;
            var data = new byte[] { };

            api.Library_GetArtworkEx(
                track,
                0,
                true,
                out locations,
                out pictureUrl,
                out data
            );

            if (!pictureUrl.IsNullOrEmpty()) return StoreCoverToCache(pictureUrl);

            var hash = data?.Length > 0 ? StoreCoverToCache(data) : string.Empty;
            return hash;
        }

        /// <summary>
        /// Prepare the cover cache by warming it up with track paths.
        /// </summary>
        public void PrepareCache()
        {
            var api = Plugin.Instance.MbApiInterface;
            var watch = Stopwatch.StartNew();
            var identifiers = GetIdentifiers();

            _logger.Debug($"Detected {identifiers.Count} albums");

            api.Library_QueryLookupTable(null, null, null);

            if (!api.Library_QueryFiles(null))
            {
                _logger.Debug("No result in query");
                return;
            }

            var paths = new Dictionary<string, string>();
            var modified = new Dictionary<string, string>();

            while (true)
            {
                var currentTrack = api.Library_QueryGetNextFile();
                if (string.IsNullOrEmpty(currentTrack)) break;

                var album = GetAlbumForTrack(currentTrack);
                var artist = GetAlbumArtistForTrack(currentTrack);
                var fileModified = api.Library_GetFileProperty(currentTrack, FilePropertyType.DateModified);

                try
                {
                    var key = CoverIdentifier(artist, album);

                    if (!identifiers.Contains(key)) continue;

                    paths[key] = currentTrack;
                    modified[key] = fileModified;
                }
                catch (Exception e)
                {
                    _logger.Error(e, $"Failed creating identifier for {album} by {artist}");
                }
            }

            CoverCache.Instance.WarmUpCache(paths, modified);
            watch.Stop();
            _logger.Debug($"Cover cache preparation: {watch.ElapsedMilliseconds} ms");
        }

        /// <summary>
        /// Build the cover cache.
        /// </summary>
        public void BuildCoverCache()
        {
            var watch = Stopwatch.StartNew();
            CoverCache.Instance.Build(CacheCover);
            watch.Stop();
            _logger.Debug($"Cover cache task complete after: {watch.ElapsedMilliseconds} ms");
        }

        /// <summary>
        /// Get album identifiers for cache.
        /// </summary>
        public List<string> GetIdentifiers()
        {
            var api = Plugin.Instance.MbApiInterface;
            var identifiers = new List<string>();
            if (!api.Library_QueryLookupTable("album", "albumartist" + '\0' + "album", null)) return identifiers;
            try
            {
                var data = api.Library_QueryGetLookupTableValue(null)
                    .Split(new[] { "\0\0" }, StringSplitOptions.None)
                    .Where(s => !string.IsNullOrEmpty(s))
                    .Select(s => s.Trim())
                    .Select(Plugin.CreateAlbum)
                    .Select(album => CoverIdentifier(album.Artist, album.Album))
                    .Distinct()
                    .ToList();

                identifiers.AddRange(data);
            }
            catch (IndexOutOfRangeException ex)
            {
                _logger.Error(ex, "While loading album data");
            }

            return identifiers;
        }

        /// <summary>
        /// Get cover with specific size.
        /// </summary>
        public AlbumCoverPayload GetCoverSize(string artist, string album, string size)
        {
            var api = Plugin.Instance.MbApiInterface;
            AlbumCoverPayload payload;
            var isNumeric = int.TryParse(size, out var coverSize);
            var original = size == "original";
            if (!original && !isNumeric)
            {
                payload = GetAlbumCoverStatusPayload(400);
            }
            else
            {
                var key = CoverIdentifier(artist, album);
                var path = CoverCache.Instance.Lookup(key);
                if (path.IsNullOrEmpty())
                {
                    payload = GetAlbumCoverStatusPayload(404);
                }
                else
                {
                    var locations = PictureLocations.EmbedInFile |
                                    PictureLocations.LinkToSource |
                                    PictureLocations.LinkToOrganisedCopy;

                    var pictureUrl = string.Empty;
                    var data = new byte[] { };

                    api.Library_GetArtworkEx(
                        path,
                        0,
                        true,
                        out locations,
                        out pictureUrl,
                        out data
                    );

                    if (!pictureUrl.IsNullOrEmpty())
                        payload = new AlbumCoverPayload
                        {
                            Cover = original
                                ? FileToBase64(pictureUrl)
                                : ImageResizeFile(pictureUrl, coverSize, coverSize),
                            Status = 200,
                            Hash = Sha1HashFile(pictureUrl)
                        };
                    else if (data?.Length > 0)
                        payload = new AlbumCoverPayload
                        {
                            Cover = original
                                ? Convert.ToBase64String(data)
                                : ImageResize(data, coverSize, coverSize),
                            Status = 200,
                            Hash = Sha1Hash(data)
                        };
                    else
                        payload = GetAlbumCoverStatusPayload(404);
                }
            }

            return payload;
        }

        /// <summary>
        /// Request cover for an album.
        /// </summary>
        public void RequestCover(string clientId, string artist, string album, string clientHash, string size)
        {
            var stopwatch = Stopwatch.StartNew();
            AlbumCoverPayload payload;

            try
            {
                if (artist == null)
                    payload = GetAlbumCoverStatusPayload(500);
                else if (album.IsNullOrEmpty())
                    payload = GetAlbumCoverStatusPayload(400);
                else
                    payload = size != null
                        ? GetCoverSize(artist, album, size)
                        : GetAlbumCover(artist, album, clientHash);
            }
            catch (Exception e)
            {
                _logger.Error(e);
                payload = GetAlbumCoverStatusPayload(500);
            }

            var message = new SocketMessage(Constants.LibraryAlbumCover, payload);
            Plugin.SendReply(message.ToJsonString(), clientId);
            stopwatch.Stop();
            _logger.Debug($"cover request took {stopwatch.Elapsed} ms");
        }

        /// <summary>
        /// Get album cover from cache or fetch it.
        /// </summary>
        public AlbumCoverPayload GetAlbumCover(string artist, string album, string clientHash)
        {
            var cache = CoverCache.Instance;
            var key = CoverIdentifier(artist, album);
            string hash;
            if (cache.IsCached(key))
            {
                hash = cache.GetCoverHash(key);
                return hash.IsEmpty() ? GetAlbumCoverStatusPayload(404) : GetAlbumCoverFromCache(clientHash, hash);
            }

            var path = cache.Lookup(key);
            if (path.IsNullOrEmpty()) return GetAlbumCoverStatusPayload(404);

            hash = CacheCover(path);
            if (hash.IsEmpty()) return GetAlbumCoverStatusPayload(404);

            cache.Update(key, hash);
            return GetAlbumCoverFromCache(clientHash, hash);
        }

        /// <summary>
        /// Create status payload.
        /// </summary>
        public static AlbumCoverPayload GetAlbumCoverStatusPayload(int status)
        {
            return new AlbumCoverPayload
            {
                Status = status
            };
        }

        /// <summary>
        /// Get cover from cache with hash check.
        /// </summary>
        public static AlbumCoverPayload GetAlbumCoverFromCache(string clientHash, string hash)
        {
            if (clientHash.IsEmpty() || clientHash != hash)
                return new AlbumCoverPayload
                {
                    Cover = GetCoverFromCache(hash),
                    Status = 200,
                    Hash = hash
                };

            return GetAlbumCoverStatusPayload(304);
        }

        /// <summary>
        /// Request a page of covers.
        /// </summary>
        public void RequestCoverPage(string clientId, int offset, int limit)
        {
            var stopwatch = Stopwatch.StartNew();
            var keys = CoverCache.Instance.Keys();
            var page = keys.Skip(offset).Take(limit);

            var data = (from key in page
                let hash = CoverCache.Instance.GetCoverHash(key)
                let path = CoverCache.Instance.Lookup(key)
                let cover = GetCoverFromCache(hash)
                select new AlbumCoverPayload
                {
                    Artist = GetAlbumArtistForTrack(path),
                    Album = GetAlbumForTrack(path),
                    Cover = cover,
                    Hash = hash,
                    Status = cover.IsEmpty() ? 404 : 200
                }).ToList();

            var message = new SocketMessage
            {
                Context = Constants.LibraryAlbumCover,
                Data = new Page<AlbumCoverPayload>
                {
                    Data = offset > keys.Count ? new List<AlbumCoverPayload>() : data,
                    Offset = offset,
                    Limit = limit,
                    Total = keys.Count
                }
            };

            Plugin.SendReply(message.ToJsonString(), clientId);
            stopwatch.Stop();
            _logger.Debug($"cover page from: {offset} with {limit} limit, request took {stopwatch.Elapsed} ms");
        }

        // Helper methods that delegate to Plugin
        private static string GetAlbumForTrack(string path)
        {
            return Plugin.Instance.MbApiInterface.Library_GetFileTag(path, MetaDataType.Album);
        }

        private static string GetAlbumArtistForTrack(string path)
        {
            return Plugin.Instance.MbApiInterface.Library_GetFileTag(path, MetaDataType.AlbumArtist);
        }
    }
}
