﻿using System;
using System.Net;
using System.Net.Sockets;
using System.Text;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Settings;
using Newtonsoft.Json.Linq;
using NLog;

namespace MusicBeePlugin.AndroidRemote.Networking
{
    public class SocketTester
    {
        private readonly Logger _logger = LogManager.GetCurrentClassLogger();

        public IConnectionListener ConnectionListener { get; set; }

        public void VerifyConnection()
        {
            Socket client = null;
            try
            {
                var port = UserSettings.Instance.ListeningPort;
                _logger.Debug($"=== SOCKET TESTER VERIFY CONNECTION STARTED ===");
                _logger.Debug($"Thread: {System.Threading.Thread.CurrentThread.ManagedThreadId}");
                _logger.Debug($"Connecting to loopback:{port}");

                var ipEndpoint = new IPEndPoint(IPAddress.Loopback, (int)port);
                client = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);
                _logger.Debug($"Socket created: Handle={client.Handle}");
                client.BeginConnect(ipEndpoint, ConnectCallback, client);
            }
            catch (Exception e)
            {
                _logger.Log(LogLevel.Debug, e, "Tester Connection error");
                ConnectionListener?.OnConnectionResult(false);

                // Clean up socket on error
                CleanupSocket(client, "VerifyConnection");
            }
        }

        private void CleanupSocket(Socket socket, string context)
        {
            if (socket == null)
            {
                _logger.Debug($"  [{context}] Socket was null, nothing to cleanup");
                return;
            }

            try
            {
                // Safe property access for logging (may throw if socket in bad state)
                string socketInfo = "unknown";
                try { socketInfo = $"Connected={socket.Connected}, Handle={socket.Handle}"; }
                catch { socketInfo = "properties inaccessible"; }
                _logger.Debug($"  [{context}] Cleaning up socket: {socketInfo}");

                socket.Close();
                socket.Dispose();
                _logger.Debug($"  [{context}] Socket disposed successfully");
            }
            catch (Exception ex)
            {
                _logger.Debug($"  [{context}] Socket cleanup error (non-fatal): {ex.Message}");
            }
        }

        private void ReceiveCallback(IAsyncResult ar)
        {
            Socket client = null;
            try
            {
                var state = (StateObject)ar.AsyncState;
                client = state.WorkSocket;

                _logger.Debug($"ReceiveCallback: Socket Handle={client?.Handle}, Connected={client?.Connected}");

                var received = client.EndReceive(ar);
                _logger.Debug($"ReceiveCallback: Received {received} bytes");

                var chars = new char[received + 1];
                var decoder = Encoding.UTF8.GetDecoder();
                decoder.GetChars(state.Buffer, 0, received, chars, 0);
                var message = new string(chars);
                var json = JObject.Parse(message);
                var verified = (string)json["context"] == Constants.VerifyConnection;

                _logger.Log(LogLevel.Info, $"Connection verified: {verified}");
                ConnectionListener?.OnConnectionResult(verified);

                _logger.Debug("ReceiveCallback: Shutting down and disposing socket");
                client.Shutdown(SocketShutdown.Both);
                client.Close();
                client.Dispose();
                _logger.Debug("ReceiveCallback: Socket disposed successfully");
            }
            catch (Exception e)
            {
                _logger.Log(LogLevel.Debug, e, "Tester Connection error");
                ConnectionListener?.OnConnectionResult(false);

                // Clean up socket on error
                CleanupSocket(client, "ReceiveCallback");
            }
        }

        private void ConnectCallback(IAsyncResult ar)
        {
            Socket client = null;
            try
            {
                client = (Socket)ar.AsyncState;
                _logger.Debug($"ConnectCallback: Ending connection, Handle={client?.Handle}");

                client.EndConnect(ar);
                _logger.Debug($"ConnectCallback: Connected={client.Connected}, RemoteEndPoint={client.RemoteEndPoint}");

                var state = new StateObject { WorkSocket = client };
                client.BeginReceive(state.Buffer, 0, StateObject.BufferSize, 0, ReceiveCallback, state);
                client.ReceiveTimeout = 3000;
                var payload = Payload();
                _logger.Debug($"ConnectCallback: Sending {payload.Length} byte payload");
                client.Send(payload);
                _logger.Debug("ConnectCallback: Payload sent, waiting for receive callback");
            }
            catch (Exception e)
            {
                _logger.Log(LogLevel.Debug, e, "Tester Connection error");
                ConnectionListener?.OnConnectionResult(false);

                // Clean up socket on error
                CleanupSocket(client, "ConnectCallback");
            }
        }

        private static byte[] Payload()
        {
            var socketMessage = new SocketMessage(Constants.VerifyConnection, string.Empty);
            var payload = Encoding.UTF8.GetBytes(socketMessage.ToJsonString() + "\r\n");
            return payload;
        }

        // State object for receiving data from remote device.
        private class StateObject
        {
            // Size of receive buffer.
            public const int BufferSize = 256;

            // Receive buffer.
            public readonly byte[] Buffer = new byte[BufferSize];

            // Received data string.
            public StringBuilder Sb = new StringBuilder();

            // Client socket.
            public Socket WorkSocket;
        }

        public interface IConnectionListener
        {
            void OnConnectionResult(bool isConnected);
        }
    }
}