using System;
using System.Collections.Concurrent;
using System.Globalization;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading.Tasks;
using System.Timers;
using MusicBeePlugin.AndroidRemote.Events;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Settings;
using MusicBeePlugin.AndroidRemote.Threading;
using MusicBeePlugin.AndroidRemote.Utilities;
using MusicBeePlugin.Tools;
using NLog;

namespace MusicBeePlugin.AndroidRemote.Networking
{
    /// <summary>
    ///     The socket server.
    /// </summary>
    public sealed class SocketServer : IDisposable
    {
        private const string NewLine = "\r\n";

        private readonly ConcurrentDictionary<string, Socket> _availableWorkerSockets;

        private readonly ProtocolHandler _handler;
        private readonly Logger _logger = LogManager.GetCurrentClassLogger();
        private readonly TaskFactory _messageProcessingFactory;

        private bool _isRunning;

        /// <summary>
        ///     The main socket. This is the Socket that listens for new client connections.
        /// </summary>
        private Socket _mainSocket;

        private Timer _pingTimer;

        /// <summary>
        ///     The worker callback.
        /// </summary>
        private AsyncCallback _workerCallback;

        /// <summary>
        /// </summary>
        private SocketServer()
        {
            _handler = new ProtocolHandler();
            IsRunning = false;
            _availableWorkerSockets = new ConcurrentDictionary<string, Socket>();

            // Use limited concurrency scheduler to prevent socket thread blocking
            // while still allowing parallel message processing (2 concurrent tasks)
            var scheduler = new LimitedTaskScheduler(2);
            _messageProcessingFactory = new TaskFactory(scheduler);
        }


        /// <summary>
        ///     Returns the Instance of the singleton socketserver
        /// </summary>
        public static SocketServer Instance { get; } = new SocketServer();

        /// <summary>
        /// </summary>
        public bool IsRunning
        {
            get => _isRunning;
            private set
            {
                _isRunning = value;
                EventBus.FireEvent(new MessageEvent(EventType.SocketStatusChange, _isRunning));
            }
        }

        /// <summary>
        ///     Disposes anything Related to the socket server at the end of life of the Object.
        /// </summary>
        public void Dispose()
        {
            try
            {
                _logger.Debug("=== SOCKET SERVER DISPOSE STARTED ===");
                _logger.Debug($"Thread: {System.Threading.Thread.CurrentThread.ManagedThreadId}");

                // Dispose ping timer
                if (_pingTimer != null)
                {
                    _logger.Debug($"Disposing _pingTimer: Enabled={_pingTimer.Enabled}");
                    _pingTimer.Stop();
                    _pingTimer.Dispose();
                    _pingTimer = null;
                    _logger.Debug("_pingTimer disposed");
                }

                // Close all worker sockets
                var workerCount = _availableWorkerSockets.Count;
                _logger.Debug($"Closing {workerCount} worker sockets");
                int closedCount = 0;
                foreach (var kvp in _availableWorkerSockets)
                {
                    var clientId = kvp.Key;
                    var wSocket = kvp.Value;
                    if (wSocket == null)
                    {
                        _logger.Debug($"  Worker socket for {clientId} was null");
                        continue;
                    }
                    try
                    {
                        // Safe property access for logging (may throw if socket in bad state)
                        string socketInfo = "unknown";
                        try { socketInfo = $"Connected={wSocket.Connected}, Handle={wSocket.Handle}"; }
                        catch { socketInfo = "properties inaccessible"; }
                        _logger.Debug($"  Closing worker socket: ClientId={clientId}, {socketInfo}");

                        wSocket.Close();
                        wSocket.Dispose();
                        closedCount++;
                    }
                    catch (Exception ex)
                    {
                        _logger.Debug($"  Worker socket {clientId} cleanup exception: {ex.Message}");
                    }
                }
                _availableWorkerSockets.Clear();
                _logger.Debug($"Closed {closedCount}/{workerCount} worker sockets, dictionary cleared");

                // Dispose main socket
                if (_mainSocket != null)
                {
                    try
                    {
                        // Safe property access for logging (may throw if socket in bad state)
                        string socketInfo = "unknown";
                        try { socketInfo = $"Connected={_mainSocket.Connected}, Handle={_mainSocket.Handle}"; }
                        catch { socketInfo = "properties inaccessible"; }
                        _logger.Debug($"Disposing _mainSocket: {socketInfo}");

                        _mainSocket.Close();
                        _mainSocket.Dispose();
                        _logger.Debug("_mainSocket disposed");
                    }
                    catch (Exception ex)
                    {
                        _logger.Debug($"_mainSocket cleanup exception: {ex.Message}");
                    }
                    finally
                    {
                        _mainSocket = null;
                    }
                }
                else
                {
                    _logger.Debug("_mainSocket was already null");
                }

                _logger.Debug("=== SOCKET SERVER DISPOSE COMPLETED ===");
            }
            catch (Exception ex)
            {
                _logger.Debug($"Exception during SocketServer.Dispose(): {ex.Message}");
            }
        }


        /// <summary>
        /// </summary>
        /// <param name="clientId"> </param>
        public void KickClient(string clientId)
        {
            try
            {
                if (!_availableWorkerSockets.TryRemove(clientId, out var workerSocket)) return;
                workerSocket.Close();
                workerSocket.Dispose();
            }
            catch (Exception ex)
            {
                _logger.Error(ex, "While kicking a client");
            }
        }

        /// <summary>
        ///     It stops the SocketServer.
        /// </summary>
        /// <returns></returns>
        public void Stop()
        {
            _logger.Debug("Stopping socket service");
            try
            {
                _pingTimer?.Stop();
                _pingTimer?.Dispose();
                _pingTimer = null;

                _mainSocket?.Close();

                foreach (var wSocket in _availableWorkerSockets.Values)
                {
                    if (wSocket == null) continue;
                    wSocket.Close();
                    wSocket.Dispose();
                }

                _mainSocket = null;
            }
            catch (Exception ex)
            {
                _logger.Error(ex, "While stopping the socket service");
            }
            finally
            {
                IsRunning = false;
            }
        }

        /// <summary>
        ///     It starts the SocketServer.
        /// </summary>
        /// <returns></returns>
        public void Start()
        {
            _logger.Debug($"Socket starts listening on port: {UserSettings.Instance.ListeningPort}");
            try
            {
                _mainSocket = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);
                // Create the listening socket.    
                var ipLocal = new IPEndPoint(IPAddress.Any, Convert.ToInt32(UserSettings.Instance.ListeningPort));
                // Bind to local IP address.
                _mainSocket.Bind(ipLocal);
                // Start Listening.
                _mainSocket.Listen(4);
                // Create the call back for any client connections.
                _mainSocket.BeginAccept(OnClientConnect, null);
                IsRunning = true;

                _pingTimer = new Timer(15000);
                _pingTimer.Elapsed += PingTimerOnElapsed;
                _pingTimer.Enabled = true;
            }
            catch (SocketException se)
            {
                _logger.Error(se, "While starting the socket service");
            }
        }

        private void PingTimerOnElapsed(object sender, ElapsedEventArgs e)
        {
            Send(new SocketMessage("ping", string.Empty).ToJsonString());
            _logger.Debug($"Ping: {DateTime.UtcNow}");
        }

        /// <summary>
        ///     Restarts the main socket that is listening for new clients.
        ///     Useful when the user wants to change the listening port.
        /// </summary>
        public void RestartSocket()
        {
            Stop();
            Start();
        }

        // this is the call back function,
        private void OnClientConnect(IAsyncResult ar)
        {
            try
            {
                // Here we complete/end the BeginAccept asynchronous call
                // by calling EndAccept() - Which returns the reference
                // to a new Socket object.
                var workerSocket = _mainSocket.EndAccept(ar);

                // Validate If client should connect using IpFilterHelper
                var ipAddress = ((IPEndPoint)workerSocket.RemoteEndPoint).Address;
                var ipString = ipAddress.ToString();
                var isAllowed = IpFilterHelper.IsIpAllowed(ipAddress);

                if (!isAllowed)
                {
                    workerSocket.Send(
                        Encoding.UTF8.GetBytes(
                            new SocketMessage(Constants.NotAllowed, string.Empty).ToJsonString()));
                    workerSocket.Close();
                    _logger.Debug($"Client {ipString} was force disconnected IP was not in the allowed addresses");
                    _mainSocket.BeginAccept(OnClientConnect, null);
                    return;
                }

                var clientId = IdGenerator.GetUniqueKey();

                if (!_availableWorkerSockets.TryAdd(clientId, workerSocket)) return;
                // Inform the the Protocol Handler that a new Client has been connected, prepare for handshake.
                EventBus.FireEvent(new MessageEvent(EventType.ActionClientConnected, string.Empty, clientId));

                // Let the worker Socket do the further processing 
                // for the just connected client.
                WaitForData(workerSocket, clientId);
            }
            catch (ObjectDisposedException)
            {
                _logger.Debug(
                    $"{DateTime.Now.ToString(CultureInfo.InvariantCulture)} : OnClientConnection: Socket has been closed\n");
            }
            catch (SocketException se)
            {
                _logger.Debug(se, "On client connect");
            }
            catch (Exception ex)
            {
                _logger.Debug(
                    $"{DateTime.Now.ToString(CultureInfo.InvariantCulture)} : OnClientConnect Exception : {ex.Message}\n");
            }
            finally
            {
                try
                {
                    // Since the main Socket is now free, it can go back and
                    // wait for the other clients who are attempting to connect
                    _mainSocket.BeginAccept(OnClientConnect, null);
                }
                catch (Exception e)
                {
                    _logger.Debug(DateTime.Now.ToString(CultureInfo.InvariantCulture) +
                                  " : OnClientConnect Exception : " + e.Message + "\n");
                }
            }
        }

        // Start waiting for data from the client
        private void WaitForData(Socket socket, string clientId, SocketPacket packet = null)
        {
            try
            {
                if (_workerCallback == null)
                    // Specify the call back function which is to be
                    // invoked when there is any write activity by the
                    // connected client.
                    _workerCallback = OnDataReceived;

                var socketPacket = packet ?? new SocketPacket(socket, clientId);

                socket.BeginReceive(socketPacket.DataBuffer, 0, socketPacket.DataBuffer.Length, SocketFlags.None,
                    _workerCallback, socketPacket);
            }
            catch (SocketException se)
            {
                _logger.Debug(se, "On WaitForData");
                if (se.ErrorCode == 10053)
                    EventBus.FireEvent(new MessageEvent(EventType.ActionClientDisconnected, string.Empty, clientId));
            }
        }

        // This is the call back function which will be invoked when the socket
        // detects any client writing of data on the stream
        private void OnDataReceived(IAsyncResult ar)
        {
            var clientId = string.Empty;
            try
            {
                var socketData = (SocketPacket)ar.AsyncState;
                // Complete the BeginReceive() asynchronous call by EndReceive() method
                // which will return the number of characters written to the stream
                // by the client.

                clientId = socketData.ClientId;

                var iRx = socketData.MCurrentSocket.EndReceive(ar);
                var chars = new char[iRx + 1];

                var decoder = Encoding.UTF8.GetDecoder();

                decoder.GetChars(socketData.DataBuffer, 0, iRx, chars, 0);
                if (chars.Length == 1 && chars[0] == 0)
                {
                    socketData.MCurrentSocket.Close();
                    socketData.MCurrentSocket.Dispose();
                    return;
                }

                var message = new string(chars).Replace("\0", "");

                if (string.IsNullOrEmpty(message))
                    return;

                if (!message.EndsWith("\r\n"))
                {
                    // Limit partial message buffer to 2MB to prevent memory exhaustion from malformed clients
                    // Note: Must be large enough to handle base64-encoded album artwork (can be 200-500KB+)
                    const int maxPartialSize = 2097152; // 2MB
                    if (socketData.Partial.Length + message.Length > maxPartialSize)
                    {
                        _logger.Warn($"Client {socketData.ClientId} exceeded max partial message size ({maxPartialSize} bytes), disconnecting");
                        try
                        {
                            socketData.MCurrentSocket.Close();
                            socketData.MCurrentSocket.Dispose();
                        }
                        catch (Exception ex)
                        {
                            _logger.Debug($"Socket cleanup failed for {socketData.ClientId}: {ex.Message}");
                        }
                        _availableWorkerSockets.TryRemove(socketData.ClientId, out _);
                        return;
                    }
                    socketData.Partial.Append(message);
                    WaitForData(socketData.MCurrentSocket, socketData.ClientId, socketData);
                    return;
                }

                if (socketData.Partial.Length > 0)
                {
                    message = socketData.Partial.Append(message).ToString();
                    socketData.Partial.Clear();
                }

                // Process message asynchronously to avoid blocking socket receive thread
                // This prevents DoS under load and allows concurrent client handling
                var messageToProcess = message;
                var clientIdToProcess = socketData.ClientId;
                _messageProcessingFactory.StartNew(() =>
                {
                    try
                    {
                        _handler.ProcessIncomingMessage(messageToProcess, clientIdToProcess);
                    }
                    catch (Exception ex)
                    {
                        _logger.Error(ex, $"Error processing message from {clientIdToProcess}");
                    }
                });

                // Continue the waiting for data on the Socket.
                WaitForData(socketData.MCurrentSocket, socketData.ClientId, socketData);
            }
            catch (ObjectDisposedException)
            {
                EventBus.FireEvent(new MessageEvent(EventType.ActionClientDisconnected, string.Empty, clientId));
                _logger.Debug(DateTime.Now.ToString(CultureInfo.InvariantCulture) +
                              " : OnDataReceived: Socket has been closed\n");
            }
            catch (SocketException se)
            {
                if (se.ErrorCode == 10054) // Error code for Connection reset by peer
                {
                    if (_availableWorkerSockets.ContainsKey(clientId))
                        _availableWorkerSockets.TryRemove(clientId, out _);
                    EventBus.FireEvent(new MessageEvent(EventType.ActionClientDisconnected, string.Empty, clientId));
                }
                else
                {
                    _logger.Error(se, "On DataReceive");
                }
            }
        }

        /// <summary>
        /// </summary>
        /// <param name="message"></param>
        /// <param name="clientId"></param>
        public void Send(string message, string clientId)
        {
            _logger.Debug($"sending-{clientId}:{message}");

            if (clientId.Equals("all"))
            {
                Send(message);
                return;
            }

            try
            {
                var data = Encoding.UTF8.GetBytes(message + NewLine);
                if (_availableWorkerSockets.TryGetValue(clientId, out var wSocket))
                {
                    try
                    {
                        wSocket.Send(data);
                    }
                    catch (SocketException)
                    {
                        // Socket was closed between check and send - remove it
                        RemoveDeadSocket(clientId);
                    }
                    catch (ObjectDisposedException)
                    {
                        // Socket was disposed - remove it
                        RemoveDeadSocket(clientId);
                    }
                }
            }
            catch (Exception ex)
            {
                _logger.Error(ex, "While sending message to specific client");
            }
        }

        private void RemoveDeadSocket(string clientId)
        {
            _availableWorkerSockets.TryRemove(clientId, out var worker);
            worker?.Dispose();
        }


        public void Broadcast(BroadcastEvent broadcastEvent)
        {
            _logger.Debug($"broadcasting message {broadcastEvent}");

            try
            {
                foreach (var key in _availableWorkerSockets.Keys)
                {
                    if (!_availableWorkerSockets.TryGetValue(key, out var worker)) continue;
                    var isConnected = worker != null && worker.Connected;
                    if (!isConnected)
                    {
                        RemoveDeadSocket(key);
                        EventBus.FireEvent(new MessageEvent(EventType.ActionClientDisconnected, string.Empty, key));
                    }

                    if (!isConnected || !Authenticator.IsClientAuthenticated(key) ||
                        !Authenticator.IsClientBroadcastEnabled(key)) continue;

                    var clientProtocol = Authenticator.ClientProtocolVersion(key);
                    var message = broadcastEvent.GetMessage((int)clientProtocol);
                    var data = Encoding.UTF8.GetBytes(message + NewLine);
                    try
                    {
                        worker.Send(data);
                    }
                    catch (SocketException)
                    {
                        RemoveDeadSocket(key);
                    }
                    catch (ObjectDisposedException)
                    {
                        RemoveDeadSocket(key);
                    }
                }
            }
            catch (Exception ex)
            {
                _logger.Error(ex, "While sending message to all available clients");
            }
        }

        /// <summary>
        /// </summary>
        /// <param name="message"></param>
        public void Send(string message)
        {
            _logger.Debug($"sending-all: {message}");

            try
            {
                var data = Encoding.UTF8.GetBytes(message + NewLine);

                foreach (var key in _availableWorkerSockets.Keys)
                {
                    if (!_availableWorkerSockets.TryGetValue(key, out var worker)) continue;
                    var isConnected = worker != null && worker.Connected;
                    if (!isConnected)
                    {
                        RemoveDeadSocket(key);
                        EventBus.FireEvent(new MessageEvent(EventType.ActionClientDisconnected, string.Empty, key));
                    }

                    if (isConnected && Authenticator.IsClientAuthenticated(key) &&
                        Authenticator.IsClientBroadcastEnabled(key))
                    {
                        try
                        {
                            worker.Send(data);
                        }
                        catch (SocketException)
                        {
                            RemoveDeadSocket(key);
                        }
                        catch (ObjectDisposedException)
                        {
                            RemoveDeadSocket(key);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                _logger.Error(ex, "While sending message to all available clients");
            }
        }
    }
}