﻿using System;
using System.Collections.Generic;
using System.Net;
using System.Net.Sockets;
using System.Text;
using MusicBeePlugin.AndroidRemote.Settings;
using MusicBeePlugin.Tools;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using NLog;

namespace MusicBeePlugin.AndroidRemote.Networking
{
    internal class ServiceDiscovery
    {
        private const int Port = 45345;
        private static readonly IPAddress MulticastAddress = IPAddress.Parse("239.1.5.10");
        private readonly Logger _logger = LogManager.GetCurrentClassLogger();
        private readonly List<UdpClient> _udpClients = new List<UdpClient>();

        private ServiceDiscovery()
        {
        }


        public static ServiceDiscovery Instance { get; } = new ServiceDiscovery();

        public void Start()
        {
            var ips = NetworkTools.GetAddressList();

            ips.ForEach(address =>
            {
                UdpClient udpClient = null;
                try
                {
                    _logger.Debug($"Starting discovery listener at {MulticastAddress}:{Port} for interface {address}");
                    udpClient = new UdpClient(AddressFamily.InterNetwork) { EnableBroadcast = true };
                    udpClient.Client.SetSocketOption(SocketOptionLevel.Socket, SocketOptionName.ReuseAddress, true);
                    udpClient.Client.Bind(new IPEndPoint(address, Port));
                    udpClient.JoinMulticastGroup(MulticastAddress, address);
                    udpClient.BeginReceive(OnDataReceived, udpClient);
                    _udpClients.Add(udpClient);
                }
                catch (Exception ex)
                {
                    _logger.Error(ex, $"Failed to start discovery on {address}");
                    // Clean up if setup failed partway through
                    try
                    {
                        udpClient?.Close();
                        udpClient?.Dispose();
                    }
                    catch
                    {
                        // Ignore cleanup errors
                    }
                }
            });
        }

        public void Stop()
        {
            _logger.Debug("=== SERVICE DISCOVERY STOP STARTED ===");
            _logger.Debug($"Thread: {System.Threading.Thread.CurrentThread.ManagedThreadId}");
            _logger.Debug($"UDP client count: {_udpClients.Count}");

            try
            {
                var clientIndex = 0;
                foreach (var udpClient in _udpClients)
                {
                    clientIndex++;
                    try
                    {
                        if (udpClient != null)
                        {
                            var localEndpoint = "unknown";
                            try
                            {
                                localEndpoint = udpClient.Client?.LocalEndPoint?.ToString() ?? "null";
                            }
                            catch (Exception ex)
                            {
                                _logger.Debug($"Could not get LocalEndPoint: {ex.Message}");
                            }

                            _logger.Debug($"  Closing UDP client {clientIndex}/{_udpClients.Count}: LocalEndpoint={localEndpoint}");
                            try
                            {
                                udpClient.DropMulticastGroup(MulticastAddress);
                            }
                            catch (Exception ex)
                            {
                                _logger.Debug($"Could not drop multicast group: {ex.Message}");
                            }
                            udpClient.Close();
                            udpClient.Dispose();
                            _logger.Debug($"  UDP client {clientIndex} disposed successfully");
                        }
                        else
                        {
                            _logger.Debug($"  UDP client {clientIndex} was null, skipping");
                        }
                    }
                    catch (Exception ex)
                    {
                        _logger.Error(ex, $"Error disposing UDP client {clientIndex}");
                    }
                }
                _udpClients.Clear();
                _logger.Debug("UDP client list cleared");
            }
            catch (Exception ex)
            {
                _logger.Error(ex, "While stopping service discovery");
            }

            _logger.Debug("=== SERVICE DISCOVERY STOP COMPLETED ===");
        }

        private void OnDataReceived(IAsyncResult ar)
        {
            var udpClient = (UdpClient)ar.AsyncState;
            var mEndPoint = new IPEndPoint(IPAddress.Any, Port);

            byte[] request;
            try
            {
                request = udpClient.EndReceive(ar, ref mEndPoint);
            }
            catch (ObjectDisposedException)
            {
                // Client was disposed during shutdown - this is expected, just exit
                return;
            }
            catch (SocketException)
            {
                // Socket was closed - this is expected during shutdown
                return;
            }

            var mRequest = Encoding.UTF8.GetString(request);
            var incoming = JObject.Parse(mRequest);

            _logger.Debug($"Discovery incoming message {mRequest}");

            var discovery = ((string)incoming["context"])?.Contains("discovery") ?? false;
            if (discovery)
            {
                var addresses = NetworkTools.GetPrivateAddressList();
                var ipString = (string)incoming["address"];
                if (string.IsNullOrEmpty(ipString))
                {
                    var notify = ErrorMessage("missing address");
                    SendResponse(notify, mEndPoint, udpClient);
                }
                else
                {
                    var interfaceAddress = InterfaceAddress(ipString, addresses);
                    var notify = DiscoveryResponse(interfaceAddress);
                    _logger.Debug($"Replying to {ipString} discovery message with {notify}");
                    SendResponse(notify, mEndPoint, udpClient);
                }
            }
            else
            {
                var notify = ErrorMessage("unsupported action");
                SendResponse(notify, mEndPoint, udpClient);
            }

            try
            {
                udpClient.BeginReceive(OnDataReceived, udpClient);
            }
            catch (ObjectDisposedException)
            {
                // Client was disposed - don't try to receive more
            }
            catch (SocketException ex)
            {
                // Socket error (e.g., connection reset by remote host) - log and don't restart receive
                _logger.Debug($"Socket error in discovery receive loop: {ex.Message}");
            }
        }

        private void SendResponse(Dictionary<string, object> notify, IPEndPoint mEndPoint, UdpClient udpClient)
        {
            var response = Encoding.UTF8.GetBytes(JsonConvert.SerializeObject(notify));
            udpClient.Send(response, response.Length, mEndPoint);
        }

        private static Dictionary<string, object> ErrorMessage(string errorMessage)
        {
            var notify = new Dictionary<string, object>
            {
                { "context", "error" },
                { "description", errorMessage }
            };
            return notify;
        }

        private static Dictionary<string, object> DiscoveryResponse(string interfaceAddress)
        {
            var notify = new Dictionary<string, object>
            {
                { "context", "notify" },
                { "address", interfaceAddress },
                { "name", Environment.GetEnvironmentVariable("COMPUTERNAME") },
                { "port", UserSettings.Instance.ListeningPort }
            };
            return notify;
        }

        private static string InterfaceAddress(string ipString, List<string> addresses)
        {
            var clientAddress = IPAddress.Parse(ipString);
            var interfaceAddress = string.Empty;

            // Handle localhost - return first available address
            if (IPAddress.IsLoopback(clientAddress))
            {
                return addresses.Count > 0 ? addresses[0] : "127.0.0.1";
            }

            foreach (var address in addresses)
            {
                var ifAddress = IPAddress.Parse(address);
                var subnetMask = NetworkTools.GetSubnetMask(address);

                var firstNetwork = NetworkTools.GetNetworkAddress(ifAddress, subnetMask);
                var secondNetwork = NetworkTools.GetNetworkAddress(clientAddress, subnetMask);
                if (!firstNetwork.Equals(secondNetwork)) continue;
                interfaceAddress = ifAddress.ToString();
                break;
            }

            // Fallback to first address if no match found
            if (string.IsNullOrEmpty(interfaceAddress) && addresses.Count > 0)
            {
                interfaceAddress = addresses[0];
            }

            return interfaceAddress;
        }
    }
}