﻿using System;
using System.Collections.Generic;
using System.Linq;
using MusicBeePlugin.AndroidRemote.Events;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Utilities;
using Newtonsoft.Json.Linq;
using NLog;

namespace MusicBeePlugin.AndroidRemote.Networking
{
    internal class ProtocolHandler
    {
        private readonly Logger _logger = LogManager.GetCurrentClassLogger();

        /// <summary>
        ///     Processes the incoming message and answer's sending back the needed data.
        /// </summary>
        /// <param name="incomingMessage">The incoming message.</param>
        /// <param name="clientId"> </param>
        public void ProcessIncomingMessage(string incomingMessage, string clientId)
        {
            _logger.Debug($"Received by client: {clientId} message --> {incomingMessage}");

            try
            {
                var msgList = new List<SocketMessage>();
                if (string.IsNullOrEmpty(incomingMessage)) return;

                // Strip BOM and other invisible characters that can corrupt JSON parsing
                incomingMessage = incomingMessage.TrimStart('\uFEFF', '\u200B', '\uFFFE');

                try
                {
                    msgList.AddRange(from msg
                            in incomingMessage
                                .Split(new[] { "\r\n" },
                                    StringSplitOptions.RemoveEmptyEntries)
                        let cleanMsg = msg.TrimStart('\uFEFF', '\u200B', '\uFFFE')
                        where !cleanMsg.Equals("\n") && !string.IsNullOrWhiteSpace(cleanMsg)
                        select new SocketMessage(JObject.Parse(cleanMsg)));
                }
                catch (Exception ex)
                {
                    _logger.Error(ex, $"while processing message -> {incomingMessage} from {clientId}");
                }

                var client = Authenticator.Client(clientId);

                foreach (var msg in msgList)
                {
                    if (msg.Context == Constants.VerifyConnection)
                    {
                        EventBus.FireEvent(
                            new MessageEvent(EventType.ReplyAvailable,
                                new SocketMessage(Constants.VerifyConnection,
                                    string.Empty).ToJsonString(), clientId));
                        return;
                    }

                    if (client.PacketNumber == 0 && msg.Context != Constants.Player)
                    {
                        EventBus.FireEvent(new MessageEvent(EventType.ActionForceClientDisconnect, string.Empty,
                            clientId));
                        return;
                    }

                    if (client.PacketNumber == 1 && msg.Context != Constants.Protocol)
                    {
                        EventBus.FireEvent(new MessageEvent(EventType.ActionForceClientDisconnect, string.Empty,
                            clientId));
                        return;
                    }

                    if (msg.Context == Constants.Protocol && msg.Data is JObject data)
                    {
                        // Safe null checks for protocol handshake data
                        if (data["no_broadcast"] != null)
                        {
                            client.BroadcastsEnabled = !(bool)data["no_broadcast"];
                        }

                        // Protocol version can be int or double, handle both
                        if (data["protocol_version"] != null)
                        {
                            var versionToken = data["protocol_version"];
                            client.ClientProtocolVersion = versionToken.Type == JTokenType.Integer
                                ? (int)versionToken
                                : (double)versionToken;
                        }
                    }

                    EventBus.FireEvent(new MessageEvent(msg.Context, msg.Data, clientId));
                }

                client.IncreasePacketNumber();
            }
            catch (Exception ex)
            {
                _logger.Error(ex, $"Processing message failed --> {incomingMessage} from {clientId}");
            }
        }
    }
}