using System.IO;
using MusicBeePlugin.AndroidRemote.Interfaces;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Networking;
using MusicBeePlugin.AndroidRemote.Settings;

namespace MusicBeePlugin.AndroidRemote.Commands.Requests
{
    /// <summary>
    /// Returns MusicBee storage paths. Requires ARiA to be enabled for security.
    /// Used for cache validation and finding iTunes XML export.
    /// </summary>
    internal class RequestStoragePath : ICommand
    {
        public void Execute(IEvent eEvent)
        {
            // Security gate: only return paths when ARiA is enabled
            if (!UserSettings.Instance.AriaEnabled)
            {
                SocketServer.Instance.Send(new SocketMessage(Constants.PluginStoragePath,
                    new { error = "Not Enabled", message = "ARiA must be enabled to access storage paths" })
                    .ToJsonString());
                return;
            }

            // Get MusicBee's base path by removing the mb_remote\ suffix from plugin storage path
            var pluginPath = UserSettings.Instance.StoragePath;
            var musicBeePath = pluginPath;

            // StoragePath ends with "mb_remote\" - strip it to get MusicBee base path
            if (pluginPath.EndsWith("mb_remote\\") || pluginPath.EndsWith("mb_remote/"))
            {
                musicBeePath = pluginPath.Substring(0, pluginPath.Length - 10);
            }
            else if (pluginPath.EndsWith("mb_remote"))
            {
                musicBeePath = pluginPath.Substring(0, pluginPath.Length - 9);
            }

            // Common locations for iTunes XML
            var itunesXmlPath = Path.Combine(musicBeePath, "iTunes Music Library.xml");
            var libraryPath = Path.Combine(musicBeePath, "Library");
            var itunesXmlInLibrary = Path.Combine(libraryPath, "iTunes Music Library.xml");

            // Check which path exists
            string actualItunesPath = null;
            if (File.Exists(itunesXmlPath))
                actualItunesPath = itunesXmlPath;
            else if (File.Exists(itunesXmlInLibrary))
                actualItunesPath = itunesXmlInLibrary;

            // Find settings file
            var settingsPath = Path.Combine(musicBeePath, "MusicBee3Settings.ini");
            if (!File.Exists(settingsPath))
            {
                // Try parent directory (some installations)
                var parentPath = Path.GetDirectoryName(musicBeePath);
                if (parentPath != null)
                {
                    var altPath = Path.Combine(parentPath, "MusicBee3Settings.ini");
                    if (File.Exists(altPath))
                        settingsPath = altPath;
                }
            }

            SocketServer.Instance.Send(new SocketMessage(Constants.PluginStoragePath, new
            {
                musicBeePath = musicBeePath.TrimEnd('\\', '/'),
                pluginPath = pluginPath.TrimEnd('\\', '/'),
                settingsPath = File.Exists(settingsPath) ? settingsPath : null,
                itunesXml = actualItunesPath,
                itunesXmlExists = actualItunesPath != null
            }).ToJsonString());
        }
    }
}
