using System;
using System.Linq;
using MusicBeePlugin.AndroidRemote.Interfaces;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Networking;
using MusicBeePlugin.AndroidRemote.Subscriptions;
using Newtonsoft.Json.Linq;
using NLog;

namespace MusicBeePlugin.AndroidRemote.Commands.Requests
{
    /// <summary>
    /// Subscribes a client to library event notifications.
    /// Request: { "eventTypes": ["tagchanged", "fileadded"], "includeMetadata": true }
    /// Response: { "success": true, "subscribedTo": ["tagchanged", "fileadded"] }
    /// </summary>
    internal class RequestLibrarySubscribe : ICommand
    {
        private static readonly Logger Logger = LogManager.GetCurrentClassLogger();

        public void Execute(IEvent eEvent)
        {
            try
            {
                var data = eEvent.Data as JObject;

                // Parse event types (optional - defaults to all)
                string[] eventTypes = null;
                if (data?["eventTypes"] is JArray eventTypesArray)
                {
                    eventTypes = eventTypesArray.Select(t => t.ToString()).ToArray();
                }

                // Parse includeMetadata (optional - defaults to true)
                var includeMetadata = true;
                if (data?["includeMetadata"] != null)
                {
                    includeMetadata = data["includeMetadata"].Value<bool>();
                }

                // Subscribe the client
                var subscription = LibrarySubscriptionManager.Instance.Subscribe(
                    eEvent.ClientId,
                    eventTypes,
                    includeMetadata);

                if (subscription == null)
                {
                    Plugin.SendReply(
                        new SocketMessage(Constants.LibrarySubscribe,
                            new { success = false, error = "Failed to create subscription" }).ToJsonString(),
                        eEvent.ClientId);
                    return;
                }

                Logger.Info("Client {0} subscribed to library events: [{1}]",
                    eEvent.ClientId, string.Join(", ", subscription.EventTypes));

                Plugin.SendReply(
                    new SocketMessage(Constants.LibrarySubscribe,
                        new
                        {
                            success = true,
                            subscribedTo = subscription.EventTypes.ToArray(),
                            includeMetadata = subscription.IncludeMetadata
                        }).ToJsonString(),
                    eEvent.ClientId);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error in RequestLibrarySubscribe");
                Plugin.SendReply(
                    new SocketMessage(Constants.LibrarySubscribe,
                        new { success = false, error = ex.Message }).ToJsonString(),
                    eEvent.ClientId);
            }
        }
    }
}
