using MusicBeePlugin.AndroidRemote.Interfaces;
using MusicBeePlugin.AndroidRemote.Networking;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using Newtonsoft.Json.Linq;
using NLog;
using System;
using System.Globalization;

namespace MusicBeePlugin.AndroidRemote.Commands.Requests
{
    /// <summary>
    /// Sets the rating for a library track by file path.
    /// Request: { "path": "/path/to/file.mp3", "rating": "3.5" }
    /// Response: { "success": true } or { "error": "message" }
    /// </summary>
    internal class RequestLibrarySetRating : ICommand
    {
        private static readonly Logger Logger = LogManager.GetCurrentClassLogger();

        public void Execute(IEvent eEvent)
        {
            try
            {
                var data = eEvent.Data as JObject;
                if (data == null)
                {
                    Plugin.SendReply(
                        new SocketMessage(Constants.LibrarySetRating,
                            new { error = "Invalid request format" }).ToJsonString(),
                        eEvent.ClientId);
                    return;
                }

                var path = data["path"]?.ToString();
                var ratingStr = data["rating"]?.ToString();

                if (string.IsNullOrEmpty(path))
                {
                    Plugin.SendReply(
                        new SocketMessage(Constants.LibrarySetRating,
                            new { error = "path is required" }).ToJsonString(),
                        eEvent.ClientId);
                    return;
                }

                if (string.IsNullOrEmpty(ratingStr))
                {
                    Plugin.SendReply(
                        new SocketMessage(Constants.LibrarySetRating,
                            new { error = "rating is required" }).ToJsonString(),
                        eEvent.ClientId);
                    return;
                }

                // Parse rating (0-5 scale)
                var decimalSeparator = Convert.ToChar(CultureInfo.CurrentCulture.NumberFormat.NumberDecimalSeparator);
                ratingStr = ratingStr.Replace('.', decimalSeparator);

                if (!float.TryParse(ratingStr, out var rating) || rating < 0 || rating > 5)
                {
                    Plugin.SendReply(
                        new SocketMessage(Constants.LibrarySetRating,
                            new { error = "rating must be between 0 and 5" }).ToJsonString(),
                        eEvent.ClientId);
                    return;
                }

                // Set the rating via MusicBee API
                var success = Plugin.Instance.SetLibraryTrackRating(path, rating);

                Logger.Info("LibrarySetRating: path={0}, rating={1}, success={2}", path, rating, success);

                Plugin.SendReply(
                    new SocketMessage(Constants.LibrarySetRating,
                        new { success, path, rating }).ToJsonString(),
                    eEvent.ClientId);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error in RequestLibrarySetRating");
                Plugin.SendReply(
                    new SocketMessage(Constants.LibrarySetRating,
                        new { error = ex.Message }).ToJsonString(),
                    eEvent.ClientId);
            }
        }
    }
}
