using MusicBeePlugin.AndroidRemote.Interfaces;
using MusicBeePlugin.AndroidRemote.Networking;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using Newtonsoft.Json.Linq;
using NLog;
using System;

namespace MusicBeePlugin.AndroidRemote.Commands.Requests
{
    /// <summary>
    /// Sets the love/ban status for a library track by file path.
    /// Request: { "path": "/path/to/file.mp3", "status": "love" }
    /// Status values: "love", "ban", "normal" (clears love/ban)
    /// Response: { "success": true } or { "error": "message" }
    /// </summary>
    internal class RequestLibrarySetLove : ICommand
    {
        private static readonly Logger Logger = LogManager.GetCurrentClassLogger();

        public void Execute(IEvent eEvent)
        {
            try
            {
                var data = eEvent.Data as JObject;
                if (data == null)
                {
                    Plugin.SendReply(
                        new SocketMessage(Constants.LibrarySetLove,
                            new { error = "Invalid request format" }).ToJsonString(),
                        eEvent.ClientId);
                    return;
                }

                var path = data["path"]?.ToString();
                var status = data["status"]?.ToString()?.ToLowerInvariant();

                if (string.IsNullOrEmpty(path))
                {
                    Plugin.SendReply(
                        new SocketMessage(Constants.LibrarySetLove,
                            new { error = "path is required" }).ToJsonString(),
                        eEvent.ClientId);
                    return;
                }

                if (string.IsNullOrEmpty(status))
                {
                    Plugin.SendReply(
                        new SocketMessage(Constants.LibrarySetLove,
                            new { error = "status is required (love, ban, or normal)" }).ToJsonString(),
                        eEvent.ClientId);
                    return;
                }

                // Validate status value
                if (status != "love" && status != "ban" && status != "normal")
                {
                    Plugin.SendReply(
                        new SocketMessage(Constants.LibrarySetLove,
                            new { error = "status must be 'love', 'ban', or 'normal'" }).ToJsonString(),
                        eEvent.ClientId);
                    return;
                }

                // Set the love status via MusicBee API
                var success = Plugin.Instance.SetLibraryTrackLove(path, status);

                Logger.Info("LibrarySetLove: path={0}, status={1}, success={2}", path, status, success);

                Plugin.SendReply(
                    new SocketMessage(Constants.LibrarySetLove,
                        new { success, path, status }).ToJsonString(),
                    eEvent.ClientId);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error in RequestLibrarySetLove");
                Plugin.SendReply(
                    new SocketMessage(Constants.LibrarySetLove,
                        new { error = ex.Message }).ToJsonString(),
                    eEvent.ClientId);
            }
        }
    }
}
