using MusicBeePlugin.AndroidRemote.Interfaces;
using MusicBeePlugin.AndroidRemote.Events;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Networking;
using MusicBeePlugin.AndroidRemote.Commands;
using NLog;
using System.Threading.Tasks;

namespace MusicBeePlugin.AndroidRemote.Commands.Requests
{
    /// <summary>
    /// Handles initiator action requests from remote clients
    /// </summary>
    public class RequestInitiatorAction : ICommand
    {
        private static readonly Logger Logger = LogManager.GetCurrentClassLogger();

        public void Execute(IEvent @event)
        {
            Logger.Info($"RequestInitiatorAction.Execute called with event type: {@event?.Data?.GetType()?.FullName}");
            
            // The event.Data IS the action script directly, not a SocketMessage
            string actionScript = null;
            
            if (@event.Data is string scriptString)
            {
                actionScript = scriptString;
                Logger.Debug($"Action script received directly as string: {actionScript}");
            }
            else if (@event.Data != null)
            {
                // Try to convert to string
                actionScript = @event.Data.ToString();
                Logger.Debug($"Action script converted from {@event.Data.GetType().FullName}: {actionScript}");
            }

            if (string.IsNullOrWhiteSpace(actionScript))
            {
                Logger.Warn($"No action script provided. Event.Data: {@event?.Data}, Type: {@event?.Data?.GetType()?.FullName}");
                SendResponse(@event.ClientId, false, "No action script provided");
                return;
            }

            Logger.Info($"Processing initiator action: {actionScript}");

            // Get the handler and execute the action
            var handler = new InitiatorActionHandler(Plugin.Instance.MbApiInterface);
            bool success = handler.ExecuteActionScript(actionScript);

            // Send response
            SendResponse(@event.ClientId, success, success ? "Action executed" : "Action failed");
        }

        private void SendResponse(string clientId, bool success, string message)
        {
            var response = new SocketMessage(Constants.PlayerInitiatorAction, new 
            { 
                success = success, 
                message = message 
            });
            
            Plugin.SendReply(response.ToJsonString(), clientId);
        }
    }
}