using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Text.RegularExpressions;
using MusicBeePlugin.AndroidRemote.Interfaces;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using MusicBeePlugin.AndroidRemote.Networking;
using Newtonsoft.Json.Linq;
using NLog;

namespace MusicBeePlugin.AndroidRemote.Commands.Requests
{
    /// <summary>
    /// Configures ARiA hotkeys in MusicBee settings.
    /// Allows remote clients to set up Ctrl+Alt+A-L tab navigation hotkeys.
    /// Request: { "action": "check" } - returns current status without modifying
    /// Request: { "action": "configure" } or { } - configures missing hotkeys
    /// Response: { "success": true, "configured": 9, "total": 11, "hotkeys": [...], "settingsPath": "..." }
    /// </summary>
    internal class RequestConfigureAriaHotkeys : ICommand
    {
        private static readonly Logger Logger = LogManager.GetCurrentClassLogger();

        // Key code calculation: Ctrl (0x20000) + Alt (0x40000) = 0x60000 = 393216
        private const int CtrlAltModifier = 393216;

        // ARiA hotkey definitions: name -> (keyCode, command, expectedHotkey)
        private static readonly Dictionary<string, Tuple<int, string, string>> AriaHotkeys =
            new Dictionary<string, Tuple<int, string, string>>
        {
            { "Tab1", Tuple.Create(CtrlAltModifier + 65, "GeneralGotoTab1", "Ctrl+Alt+A") },
            { "Tab2", Tuple.Create(CtrlAltModifier + 83, "GeneralGotoTab2", "Ctrl+Alt+S") },
            { "Tab3", Tuple.Create(CtrlAltModifier + 68, "GeneralGotoTab3", "Ctrl+Alt+D") },
            { "Tab4", Tuple.Create(CtrlAltModifier + 70, "GeneralGotoTab4", "Ctrl+Alt+F") },
            { "Tab5", Tuple.Create(CtrlAltModifier + 71, "GeneralGotoTab5", "Ctrl+Alt+G") },
            { "Tab6", Tuple.Create(CtrlAltModifier + 72, "GeneralGotoTab6", "Ctrl+Alt+H") },
            { "Tab7", Tuple.Create(CtrlAltModifier + 74, "GeneralGotoTab7", "Ctrl+Alt+J") },
            { "Tab8", Tuple.Create(CtrlAltModifier + 75, "GeneralGotoTab8", "Ctrl+Alt+K") },
            { "Tab9", Tuple.Create(CtrlAltModifier + 76, "GeneralGotoTab9", "Ctrl+Alt+L") },
            { "VisualiserToggle", Tuple.Create(CtrlAltModifier + 86, "ViewToggleVisualiser", "Ctrl+Alt+V") },
        };

        public void Execute(IEvent eEvent)
        {
            try
            {
                // Parse action from request data
                var action = "configure"; // default
                if (eEvent.Data != null)
                {
                    try
                    {
                        var data = JObject.Parse(eEvent.Data.ToString());
                        action = data["action"]?.ToString() ?? "configure";
                    }
                    catch { /* ignore parse errors, use default */ }
                }

                // Get MusicBee's persistent storage path
                var storagePath = Plugin.Instance.GetPersistentStoragePath();
                if (string.IsNullOrEmpty(storagePath))
                {
                    SendError(eEvent.ClientId, "Could not determine MusicBee storage path");
                    return;
                }

                // Find the settings file
                var settingsPath = Path.Combine(storagePath, "MusicBee3Settings.ini");
                if (!File.Exists(settingsPath))
                {
                    var parentPath = Path.GetDirectoryName(storagePath);
                    if (parentPath != null)
                    {
                        settingsPath = Path.Combine(parentPath, "MusicBee3Settings.ini");
                    }
                }

                if (!File.Exists(settingsPath))
                {
                    SendError(eEvent.ClientId, $"Settings file not found at {settingsPath}");
                    return;
                }

                var content = File.ReadAllText(settingsPath);

                // Get current status
                var hotkeyStatus = GetHotkeyStatus(content);
                var configured = 0;
                foreach (var status in hotkeyStatus)
                {
                    if (status.Value.Item1) configured++;
                }

                if (action == "check")
                {
                    // Just return status, don't modify
                    SendStatusResponse(eEvent.ClientId, settingsPath, hotkeyStatus, configured, 0);
                    return;
                }

                // Configure missing hotkeys
                Logger.Info("Configuring ARiA hotkeys in: {0}", settingsPath);

                var hotkeysEndIndex = content.IndexOf("</HotKeys>");
                if (hotkeysEndIndex < 0)
                {
                    SendError(eEvent.ClientId, "Could not find <HotKeys> section in settings file");
                    return;
                }

                // Create backup before modifying
                var backupPath = settingsPath + string.Format(".backup_{0:yyyyMMdd_HHmmss}", DateTime.Now);
                File.Copy(settingsPath, backupPath);
                Logger.Info("Created backup: {0}", backupPath);

                var sb = new StringBuilder();
                var addedCount = 0;

                foreach (var kvp in AriaHotkeys)
                {
                    var command = kvp.Value.Item2;

                    if (content.Contains(string.Format("<Command>{0}</Command>", command)))
                        continue;

                    var keyCode = kvp.Value.Item1;
                    sb.AppendLine(" <KeyCommandPair>");
                    sb.AppendLine(string.Format("  <Key>{0}</Key>", keyCode));
                    sb.AppendLine(string.Format("  <Command>{0}</Command>", command));
                    sb.AppendLine("  <IsGlobal>true</IsGlobal>");
                    sb.AppendLine(" </KeyCommandPair>");
                    addedCount++;
                }

                if (addedCount > 0)
                {
                    var newContent = content.Insert(hotkeysEndIndex, sb.ToString());
                    File.WriteAllText(settingsPath, newContent);
                    Logger.Info("Added {0} ARiA hotkeys to MusicBee settings", addedCount);

                    // Re-read status after modification
                    content = File.ReadAllText(settingsPath);
                    hotkeyStatus = GetHotkeyStatus(content);
                    configured = 0;
                    foreach (var status in hotkeyStatus)
                    {
                        if (status.Value.Item1) configured++;
                    }
                }

                SendStatusResponse(eEvent.ClientId, settingsPath, hotkeyStatus, configured, addedCount, backupPath);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error in RequestConfigureAriaHotkeys");
                SendError(eEvent.ClientId, ex.Message);
            }
        }

        private Dictionary<string, Tuple<bool, string>> GetHotkeyStatus(string content)
        {
            var result = new Dictionary<string, Tuple<bool, string>>();

            foreach (var kvp in AriaHotkeys)
            {
                var name = kvp.Key;
                var command = kvp.Value.Item2;
                var expectedHotkey = kvp.Value.Item3;

                var isConfigured = content.Contains(string.Format("<Command>{0}</Command>", command));
                var actualHotkey = expectedHotkey;

                if (isConfigured)
                {
                    // Try to find the actual key binding
                    var pattern = string.Format(@"<Key>(\d+)</Key>\s*<Command>{0}</Command>",
                        Regex.Escape(command));
                    var match = Regex.Match(content, pattern);
                    if (match.Success)
                    {
                        actualHotkey = DecodeKeyCode(int.Parse(match.Groups[1].Value));
                    }
                }

                result[name] = Tuple.Create(isConfigured, isConfigured ? actualHotkey : "Not configured");
            }

            return result;
        }

        private string DecodeKeyCode(int keyCode)
        {
            var modifiers = new List<string>();
            if ((keyCode & 0x20000) != 0) modifiers.Add("Ctrl");
            if ((keyCode & 0x40000) != 0) modifiers.Add("Alt");
            if ((keyCode & 0x10000) != 0) modifiers.Add("Shift");

            var baseKey = keyCode & 0xFFFF;
            string keyName;

            if (baseKey >= 65 && baseKey <= 90)
                keyName = ((char)baseKey).ToString();
            else if (baseKey >= 112 && baseKey <= 123)
                keyName = "F" + (baseKey - 111);
            else
                keyName = baseKey.ToString();

            modifiers.Add(keyName);
            return string.Join("+", modifiers);
        }

        private void SendStatusResponse(string clientId, string settingsPath,
            Dictionary<string, Tuple<bool, string>> hotkeyStatus, int configured, int added,
            string backupPath = null)
        {
            var hotkeys = new List<object>();
            foreach (var kvp in hotkeyStatus)
            {
                hotkeys.Add(new
                {
                    name = kvp.Key,
                    isConfigured = kvp.Value.Item1,
                    hotkey = kvp.Value.Item2
                });
            }

            var message = added > 0
                ? string.Format("Added {0} ARiA hotkeys. Restart MusicBee to apply.", added)
                : configured >= AriaHotkeys.Count
                    ? "All ARiA hotkeys are configured."
                    : string.Format("{0} of {1} ARiA hotkeys are configured.", configured, AriaHotkeys.Count);

            Plugin.SendReply(
                new SocketMessage(Constants.ConfigureAriaHotkeys,
                    new
                    {
                        success = true,
                        configured = configured,
                        total = AriaHotkeys.Count,
                        added = added,
                        hotkeys = hotkeys,
                        settingsPath = settingsPath,
                        backupPath = backupPath,
                        message = message
                    }).ToJsonString(),
                clientId);
        }

        private void SendError(string clientId, string error)
        {
            Plugin.SendReply(
                new SocketMessage(Constants.ConfigureAriaHotkeys,
                    new { success = false, error = error }).ToJsonString(),
                clientId);
        }
    }
}
