using System;
using System.Windows.Forms;
using NLog;
using MusicBeePlugin;

namespace MusicBeePlugin.AndroidRemote.Commands
{
    /// <summary>
    /// Handles initiator action execution (SendKeys commands)
    /// </summary>
    public class InitiatorActionHandler
    {
        private static readonly Logger Logger = LogManager.GetCurrentClassLogger();
        private readonly Plugin.MusicBeeApiInterface _api;

        public InitiatorActionHandler(Plugin.MusicBeeApiInterface api)
        {
            _api = api;
        }

        public bool ExecuteActionScript(string actionScript)
        {
            if (string.IsNullOrWhiteSpace(actionScript))
            {
                Logger.Warn("Empty action script provided");
                return false;
            }
            
            // Check if ARiA functionality is enabled
            if (!Settings.UserSettings.Instance.AriaEnabled)
            {
                Logger.Warn("ARiA (Arbitrary Remote Initiated Actions) functionality is disabled");
                return false;
            }

            try
            {
                Logger.Info($"Executing action script: {actionScript}");

                // Parse and execute each command in the script
                var commands = actionScript.Split(';');
                foreach (var command in commands)
                {
                    var trimmedCommand = command.Trim();
                    if (string.IsNullOrWhiteSpace(trimmedCommand))
                        continue;

                    if (!ExecuteCommand(trimmedCommand))
                    {
                        Logger.Warn($"Failed to execute command: {trimmedCommand}");
                        return false;
                    }
                }

                return true;
            }
            catch (Exception ex)
            {
                Logger.Error($"Error executing action script: {ex.Message}");
                return false;
            }
        }

        private bool ExecuteCommand(string command)
        {
            try
            {
                // Parse command format: commandName(parameters)
                var openParen = command.IndexOf('(');
                var closeParen = command.LastIndexOf(')');

                if (openParen < 0 || closeParen < 0 || closeParen <= openParen)
                {
                    Logger.Warn($"Invalid command format: {command}");
                    return false;
                }

                var commandName = command.Substring(0, openParen).Trim().ToLower();
                var parameters = command.Substring(openParen + 1, closeParen - openParen - 1).Trim();

                switch (commandName)
                {
                    case "sndkeys":
                    case "sendkeys":
                        return ExecuteSendKeys(parameters);

                    case "delay":
                    case "wait":
                        return ExecuteDelay(parameters);

                    default:
                        Logger.Warn($"Unknown command: {commandName}");
                        return false;
                }
            }
            catch (Exception ex)
            {
                Logger.Error($"Error parsing command '{command}': {ex.Message}");
                return false;
            }
        }

        private bool ExecuteSendKeys(string keys)
        {
            try
            {
                // Check if we should focus MusicBee first
                // Use prefix to control focus behavior:
                // !mb! prefix = focus MusicBee (explicit, same as default)
                // !nofocus! prefix = don't change focus
                // No prefix = default to MusicBee focus (required for hotkeys to work)

                bool shouldFocus = true;  // Default: focus MusicBee so hotkeys work
                string actualKeys = keys;

                if (keys.StartsWith("!nofocus!"))
                {
                    shouldFocus = false;
                    actualKeys = keys.Substring(9); // Remove !nofocus! prefix
                    Logger.Debug($"No focus change requested, sending keys: {actualKeys}");
                }
                else if (keys.StartsWith("!mb!"))
                {
                    shouldFocus = true;
                    actualKeys = keys.Substring(4); // Remove !mb! prefix
                    Logger.Debug($"Explicit MusicBee focus requested, sending keys: {actualKeys}");
                }
                else
                {
                    // Default behavior - focus MusicBee for hotkeys to work
                    Logger.Debug($"Default behavior - focusing MusicBee, sending keys: {actualKeys}");
                }
                
                if (shouldFocus)
                {
                    IntPtr mbWindow = _api.MB_GetWindowHandle();
                    Logger.Info($"MusicBee window handle: 0x{mbWindow.ToInt64():X}");

                    if (mbWindow != IntPtr.Zero)
                    {
                        // Bring MusicBee to foreground
                        Logger.Info("Attempting to focus MusicBee window...");
                        bool focusResult = NativeMethods.SetForegroundWindow(mbWindow);
                        Logger.Info($"SetForegroundWindow returned: {focusResult}");

                        System.Threading.Thread.Sleep(50); // Small delay to ensure focus
                        Logger.Info("Focus delay complete (50ms)");

                        // Verify focus
                        IntPtr foregroundWindow = NativeMethods.GetForegroundWindow();
                        Logger.Info($"Current foreground window: 0x{foregroundWindow.ToInt64():X}, MB window: 0x{mbWindow.ToInt64():X}, Match: {foregroundWindow == mbWindow}");
                    }
                    else
                    {
                        Logger.Warn("MusicBee window handle is null/zero - cannot focus");
                    }
                }

                // Convert Ducky Script notation to SendKeys format
                var convertedKeys = ConvertDuckyToSendKeys(actualKeys);
                Logger.Info($"Converted '{actualKeys}' to SendKeys format: '{convertedKeys}'");

                // Send the keys
                Logger.Info($"Calling SendKeys.SendWait('{convertedKeys}')");
                SendKeys.SendWait(convertedKeys);

                Logger.Info($"SendKeys completed successfully");
                return true;
            }
            catch (Exception ex)
            {
                Logger.Error($"Error sending keys '{keys}': {ex.Message}");
                return false;
            }
        }

        private string ConvertDuckyToSendKeys(string duckyScript)
        {
            if (string.IsNullOrWhiteSpace(duckyScript))
                return "";
            
            // Already in SendKeys format (contains symbols)
            if (duckyScript.Contains("^") || duckyScript.Contains("%") || duckyScript.Contains("+"))
                return duckyScript;
            
            var result = duckyScript;
            
            // Replace modifier words with SendKeys symbols
            // Must be done in order to avoid conflicts
            result = System.Text.RegularExpressions.Regex.Replace(result, @"\bCTRL\b", "^", System.Text.RegularExpressions.RegexOptions.IgnoreCase);
            result = System.Text.RegularExpressions.Regex.Replace(result, @"\bCONTROL\b", "^", System.Text.RegularExpressions.RegexOptions.IgnoreCase);
            result = System.Text.RegularExpressions.Regex.Replace(result, @"\bALT\b", "%", System.Text.RegularExpressions.RegexOptions.IgnoreCase);
            result = System.Text.RegularExpressions.Regex.Replace(result, @"\bSHIFT\b", "+", System.Text.RegularExpressions.RegexOptions.IgnoreCase);
            result = System.Text.RegularExpressions.Regex.Replace(result, @"\bWIN\b", "#", System.Text.RegularExpressions.RegexOptions.IgnoreCase);
            result = System.Text.RegularExpressions.Regex.Replace(result, @"\bWINDOWS\b", "#", System.Text.RegularExpressions.RegexOptions.IgnoreCase);
            
            // Replace special key names with SendKeys format
            result = System.Text.RegularExpressions.Regex.Replace(result, @"\bENTER\b", "{ENTER}", System.Text.RegularExpressions.RegexOptions.IgnoreCase);
            result = System.Text.RegularExpressions.Regex.Replace(result, @"\bTAB\b", "{TAB}", System.Text.RegularExpressions.RegexOptions.IgnoreCase);
            result = System.Text.RegularExpressions.Regex.Replace(result, @"\bESC\b", "{ESC}", System.Text.RegularExpressions.RegexOptions.IgnoreCase);
            result = System.Text.RegularExpressions.Regex.Replace(result, @"\bESCAPE\b", "{ESC}", System.Text.RegularExpressions.RegexOptions.IgnoreCase);
            result = System.Text.RegularExpressions.Regex.Replace(result, @"\bSPACE\b", " ", System.Text.RegularExpressions.RegexOptions.IgnoreCase);
            
            // Handle function keys
            for (int i = 1; i <= 12; i++)
            {
                result = System.Text.RegularExpressions.Regex.Replace(result, $@"\bF{i}\b", $"{{F{i}}}", System.Text.RegularExpressions.RegexOptions.IgnoreCase);
            }
            
            // Remove extra spaces that might have been created
            result = System.Text.RegularExpressions.Regex.Replace(result, @"\s+", "");

            // Lowercase letter keys UNLESS SHIFT is explicitly used
            // This fixes case sensitivity: ^%S (CTRL+ALT+SHIFT+S) should be ^%s (CTRL+ALT+S)
            // But if user explicitly wants SHIFT: ^%+S should stay uppercase
            bool hasExplicitShift = result.Contains("+");

            var finalResult = "";
            bool inBraces = false;
            foreach (char c in result)
            {
                if (c == '{')
                {
                    inBraces = true;
                    finalResult += c;
                }
                else if (c == '}')
                {
                    inBraces = false;
                    finalResult += c;
                }
                else if (!inBraces && char.IsLetter(c) && !hasExplicitShift)
                {
                    // Only lowercase if SHIFT modifier is not explicitly present
                    finalResult += char.ToLower(c);
                }
                else
                {
                    finalResult += c;
                }
            }

            return finalResult;
        }

        private bool ExecuteDelay(string delayMs)
        {
            try
            {
                if (int.TryParse(delayMs, out int delay))
                {
                    Logger.Debug($"Delaying for {delay}ms");
                    System.Threading.Thread.Sleep(delay);
                    return true;
                }
                else
                {
                    Logger.Warn($"Invalid delay value: {delayMs}");
                    return false;
                }
            }
            catch (Exception ex)
            {
                Logger.Error($"Error executing delay: {ex.Message}");
                return false;
            }
        }
    }

    internal static class NativeMethods
    {
        [System.Runtime.InteropServices.DllImport("user32.dll")]
        public static extern bool SetForegroundWindow(IntPtr hWnd);

        [System.Runtime.InteropServices.DllImport("user32.dll")]
        public static extern IntPtr GetForegroundWindow();
    }
}