using System;
using System.Diagnostics;
using System.Windows.Forms;
using NLog;

namespace MusicBeePlugin.AndroidRemote.ARiA
{
    /// <summary>
    /// ARiA (Arbitrary Remote Initiated Actions) command executor.
    /// Parses and executes automation scripts on the host system.
    /// </summary>
    public class AriaCommandExecutor
    {
        private static readonly Logger Logger = LogManager.GetCurrentClassLogger();

        /// <summary>
        /// Delegate for getting the target window handle for SendKeys focus.
        /// </summary>
        public Func<IntPtr> GetWindowHandle { get; set; }

        /// <summary>
        /// Delegate for getting the application install path (for restart command).
        /// </summary>
        public Func<string> GetInstallPath { get; set; }

        /// <summary>
        /// Delegate for requesting application close (for restart command).
        /// </summary>
        public Action RequestApplicationClose { get; set; }

        /// <summary>
        /// Execute an ARiA script containing one or more commands.
        /// Commands are separated by semicolons.
        /// </summary>
        public bool ExecuteScript(string script)
        {
            if (string.IsNullOrWhiteSpace(script))
            {
                Logger.Warn("Empty action script provided");
                return false;
            }

            try
            {
                Logger.Info($"Executing ARiA script: {script}");

                var commands = script.Split(';');
                foreach (var command in commands)
                {
                    var trimmedCommand = command.Trim();
                    if (string.IsNullOrWhiteSpace(trimmedCommand))
                        continue;

                    if (!ExecuteCommand(trimmedCommand))
                    {
                        Logger.Warn($"Failed to execute command: {trimmedCommand}");
                        return false;
                    }
                }

                return true;
            }
            catch (Exception ex)
            {
                Logger.Error($"Error executing ARiA script: {ex.Message}");
                return false;
            }
        }

        private bool ExecuteCommand(string command)
        {
            try
            {
                // Parse command format: commandName(parameters)
                var openParen = command.IndexOf('(');
                var closeParen = command.LastIndexOf(')');

                if (openParen < 0 || closeParen < 0 || closeParen <= openParen)
                {
                    Logger.Warn($"Invalid command format: {command}");
                    return false;
                }

                var commandName = command.Substring(0, openParen).Trim().ToLower();
                var parameters = command.Substring(openParen + 1, closeParen - openParen - 1).Trim();

                switch (commandName)
                {
                    case "sndkeys":
                    case "sendkeys":
                        return ExecuteSendKeys(parameters);

                    case "delay":
                    case "wait":
                        return ExecuteDelay(parameters);

                    case "mouseclick":
                    case "click":
                        return ExecuteMouseClick(parameters);

                    case "volume":
                    case "vol":
                        return ExecuteVolume(parameters);

                    case "restart":
                        return ExecuteRestart(parameters);

                    case "run":
                    case "exec":
                    case "launch":
                        return ExecuteRun(parameters);

                    default:
                        Logger.Warn($"Unknown command: {commandName}");
                        return false;
                }
            }
            catch (Exception ex)
            {
                Logger.Error($"Error parsing command '{command}': {ex.Message}");
                return false;
            }
        }

        private bool ExecuteSendKeys(string keys)
        {
            try
            {
                Logger.Debug($"Sending keys: {keys}");

                // Focus target window first if handler is set
                if (GetWindowHandle != null)
                {
                    IntPtr window = GetWindowHandle();
                    if (window != IntPtr.Zero)
                    {
                        AriaNativeMethods.SetForegroundWindow(window);
                        System.Threading.Thread.Sleep(50);
                    }
                }

                SendKeys.SendWait(keys);

                Logger.Debug($"Keys sent successfully: {keys}");
                return true;
            }
            catch (Exception ex)
            {
                Logger.Error($"Error sending keys '{keys}': {ex.Message}");
                return false;
            }
        }

        private bool ExecuteDelay(string delayMs)
        {
            try
            {
                if (int.TryParse(delayMs, out int delay))
                {
                    // Cap delay at 30 seconds to prevent abuse
                    delay = Math.Min(delay, 30000);
                    Logger.Debug($"Delaying for {delay}ms");
                    System.Threading.Thread.Sleep(delay);
                    return true;
                }
                else
                {
                    Logger.Warn($"Invalid delay value: {delayMs}");
                    return false;
                }
            }
            catch (Exception ex)
            {
                Logger.Error($"Error executing delay: {ex.Message}");
                return false;
            }
        }

        private bool ExecuteMouseClick(string parameters)
        {
            try
            {
                // Parse parameters: x,y or x,y,button (button: left/right/middle)
                var parts = parameters.Split(',');
                if (parts.Length < 2)
                {
                    Logger.Warn($"Invalid mouseClick parameters: {parameters}");
                    return false;
                }

                if (!int.TryParse(parts[0].Trim(), out int x) || !int.TryParse(parts[1].Trim(), out int y))
                {
                    Logger.Warn($"Invalid coordinates in mouseClick: {parameters}");
                    return false;
                }

                string button = parts.Length > 2 ? parts[2].Trim().ToLower() : "left";

                Logger.Debug($"Mouse click at ({x}, {y}) button={button}");

                // Move cursor to position
                AriaNativeMethods.SetCursorPos(x, y);
                System.Threading.Thread.Sleep(10);

                // Perform click based on button type
                uint downFlag, upFlag;
                switch (button)
                {
                    case "right":
                        downFlag = AriaNativeMethods.MOUSEEVENTF_RIGHTDOWN;
                        upFlag = AriaNativeMethods.MOUSEEVENTF_RIGHTUP;
                        break;
                    case "middle":
                        downFlag = AriaNativeMethods.MOUSEEVENTF_MIDDLEDOWN;
                        upFlag = AriaNativeMethods.MOUSEEVENTF_MIDDLEUP;
                        break;
                    default: // left
                        downFlag = AriaNativeMethods.MOUSEEVENTF_LEFTDOWN;
                        upFlag = AriaNativeMethods.MOUSEEVENTF_LEFTUP;
                        break;
                }

                AriaNativeMethods.mouse_event(downFlag, 0, 0, 0, UIntPtr.Zero);
                System.Threading.Thread.Sleep(10);
                AriaNativeMethods.mouse_event(upFlag, 0, 0, 0, UIntPtr.Zero);

                Logger.Debug($"Mouse click completed at ({x}, {y})");
                return true;
            }
            catch (Exception ex)
            {
                Logger.Error($"Error executing mouseClick: {ex.Message}");
                return false;
            }
        }

        private bool ExecuteVolume(string parameters)
        {
            try
            {
                var param = parameters.Trim().ToLower();
                Logger.Debug($"Volume command: {param}");

                switch (param)
                {
                    case "up":
                        AriaNativeMethods.keybd_event(AriaNativeMethods.VK_VOLUME_UP, 0, 0, UIntPtr.Zero);
                        AriaNativeMethods.keybd_event(AriaNativeMethods.VK_VOLUME_UP, 0, AriaNativeMethods.KEYEVENTF_KEYUP, UIntPtr.Zero);
                        break;

                    case "down":
                        AriaNativeMethods.keybd_event(AriaNativeMethods.VK_VOLUME_DOWN, 0, 0, UIntPtr.Zero);
                        AriaNativeMethods.keybd_event(AriaNativeMethods.VK_VOLUME_DOWN, 0, AriaNativeMethods.KEYEVENTF_KEYUP, UIntPtr.Zero);
                        break;

                    case "mute":
                    case "unmute":
                    case "toggle":
                        AriaNativeMethods.keybd_event(AriaNativeMethods.VK_VOLUME_MUTE, 0, 0, UIntPtr.Zero);
                        AriaNativeMethods.keybd_event(AriaNativeMethods.VK_VOLUME_MUTE, 0, AriaNativeMethods.KEYEVENTF_KEYUP, UIntPtr.Zero);
                        break;

                    default:
                        // Try to parse as number of volume steps
                        if (int.TryParse(param, out int steps))
                        {
                            byte vk = steps > 0 ? AriaNativeMethods.VK_VOLUME_UP : AriaNativeMethods.VK_VOLUME_DOWN;
                            int count = Math.Abs(steps);
                            for (int i = 0; i < count; i++)
                            {
                                AriaNativeMethods.keybd_event(vk, 0, 0, UIntPtr.Zero);
                                AriaNativeMethods.keybd_event(vk, 0, AriaNativeMethods.KEYEVENTF_KEYUP, UIntPtr.Zero);
                                System.Threading.Thread.Sleep(20);
                            }
                        }
                        else
                        {
                            Logger.Warn($"Unknown volume parameter: {param}");
                            return false;
                        }
                        break;
                }

                Logger.Debug($"Volume command completed: {param}");
                return true;
            }
            catch (Exception ex)
            {
                Logger.Error($"Error executing volume command: {ex.Message}");
                return false;
            }
        }

        private bool ExecuteRestart(string parameters)
        {
            try
            {
                var param = parameters.Trim().ToLower();
                Logger.Info($"Restart command: {param}");

                switch (param)
                {
                    case "musicbee":
                    case "mb":
                    case "app":
                        // Restart the host application
                        if (GetInstallPath == null || RequestApplicationClose == null)
                        {
                            Logger.Error("Application restart handlers not configured");
                            return false;
                        }

                        string appPath = GetInstallPath();
                        string appExe = System.IO.Path.Combine(appPath, "MusicBee.exe");

                        Logger.Info($"Restarting application from: {appExe}");

                        using (Process.Start(new ProcessStartInfo
                        {
                            FileName = appExe,
                            UseShellExecute = true
                        })) { }

                        System.Threading.Thread.Sleep(500);
                        RequestApplicationClose();
                        break;

                    case "system":
                    case "host":
                    case "pc":
                    case "computer":
                        Logger.Warn("System restart requested");
                        using (Process.Start(new ProcessStartInfo
                        {
                            FileName = "shutdown",
                            Arguments = "/r /t 5 /c \"MBXRemote requested system restart\"",
                            UseShellExecute = true,
                            CreateNoWindow = true
                        })) { }
                        break;

                    case "shutdown":
                        Logger.Warn("System shutdown requested");
                        using (Process.Start(new ProcessStartInfo
                        {
                            FileName = "shutdown",
                            Arguments = "/s /t 5 /c \"MBXRemote requested system shutdown\"",
                            UseShellExecute = true,
                            CreateNoWindow = true
                        })) { }
                        break;

                    default:
                        Logger.Warn($"Unknown restart target: {param}");
                        return false;
                }

                return true;
            }
            catch (Exception ex)
            {
                Logger.Error($"Error executing restart: {ex.Message}");
                return false;
            }
        }

        private bool ExecuteRun(string parameters)
        {
            try
            {
                if (string.IsNullOrWhiteSpace(parameters))
                {
                    Logger.Warn("Empty run command");
                    return false;
                }

                Logger.Info($"Run command: {parameters}");

                // Parse command and optional arguments
                // Format: "path" or "path,args"
                string path, args = "";

                if (parameters.Contains(","))
                {
                    int commaIdx = parameters.IndexOf(',');
                    path = parameters.Substring(0, commaIdx).Trim().Trim('"');
                    args = parameters.Substring(commaIdx + 1).Trim().Trim('"');
                }
                else
                {
                    path = parameters.Trim().Trim('"');
                }

                Logger.Debug($"Launching: {path} with args: {args}");

                var psi = new ProcessStartInfo
                {
                    FileName = path,
                    Arguments = args,
                    UseShellExecute = true
                };

                using (Process.Start(psi)) { }

                Logger.Info($"Launched successfully: {path}");
                return true;
            }
            catch (Exception ex)
            {
                Logger.Error($"Error executing run command: {ex.Message}");
                return false;
            }
        }
    }
}
