# MusicBee Remote Plugin Packager
# Creates a ZIP file that can be installed via MusicBee's "Add Plugin" feature

$ErrorActionPreference = "Stop"

$OutputDir = ".\deploy"

# Extract version from AssemblyInfo.cs
$AssemblyInfoPath = ".\plugin\Properties\AssemblyInfo.cs"
if (-not (Test-Path $AssemblyInfoPath)) {
    Write-Host "ERROR: $AssemblyInfoPath not found." -ForegroundColor Red
    exit 1
}

$AssemblyContent = Get-Content $AssemblyInfoPath -Raw
if ($AssemblyContent -match 'AssemblyFileVersion\("([^"]+)"\)') {
    $Version = $Matches[1]
} else {
    Write-Host "ERROR: Could not find AssemblyFileVersion in $AssemblyInfoPath" -ForegroundColor Red
    exit 1
}

# Paths
$DeployDir = ".\deploy"
$PluginDll = "$DeployDir\mb_remote.dll"

# Get BuildEpoch from AssemblyInfo.cs (embedded during build, matches what InfoWindow displays)
$hexTime = "00000000"
if ($AssemblyContent -match 'AssemblyMetadata\("BuildEpoch",\s*"([^"]+)"\)') {
    $hexTime = $Matches[1]
}
$FirewallUtil = "$DeployDir\firewall-utility.exe"
$ZipName = "MBRC-Plugin-$Version-$hexTime.zip"
$ZipPath = "$OutputDir\$ZipName"

Write-Host "MusicBee Remote Plugin Packager" -ForegroundColor Cyan
Write-Host "Version: $Version-$hexTime" -ForegroundColor Gray
Write-Host ""

# Check if deploy files exist
if (-not (Test-Path $PluginDll)) {
    Write-Host "ERROR: $PluginDll not found. Run build first." -ForegroundColor Red
    exit 1
}

if (-not (Test-Path $FirewallUtil)) {
    Write-Host "WARNING: $FirewallUtil not found. Continuing without it." -ForegroundColor Yellow
}

# Create output directory
if (-not (Test-Path $OutputDir)) {
    New-Item -ItemType Directory -Path $OutputDir | Out-Null
    Write-Host "Created output directory: $OutputDir" -ForegroundColor Green
}

# Move existing MBRC-Plugin ZIPs to archive before creating new one
$ArchiveDir = Join-Path $OutputDir "archive"
if (-not (Test-Path $ArchiveDir)) {
    New-Item -ItemType Directory -Path $ArchiveDir -Force | Out-Null
}
$existingZips = Get-ChildItem -Path $OutputDir -Filter "MBRC-Plugin-*.zip" -File -ErrorAction SilentlyContinue
foreach ($zip in $existingZips) {
    $archiveDest = Join-Path $ArchiveDir $zip.Name
    if (-not (Test-Path $archiveDest)) {
        Move-Item $zip.FullName $archiveDest -Force
        Write-Host "Archived: $($zip.Name)" -ForegroundColor Gray
    } else {
        Remove-Item $zip.FullName -Force
    }
}

# Create ZIP
Write-Host "Creating package..." -ForegroundColor Cyan

$filesToZip = @()
if (Test-Path $PluginDll) { $filesToZip += $PluginDll }
if (Test-Path $FirewallUtil) { $filesToZip += $FirewallUtil }

# Include logo files for plugin settings dialog
$MbrcLogo = ".\mbrc.png"
$HalradLogo = ".\halrad.png"
if (Test-Path $MbrcLogo) { $filesToZip += $MbrcLogo }
if (Test-Path $HalradLogo) { $filesToZip += $HalradLogo }

Compress-Archive -Path $filesToZip -DestinationPath $ZipPath -CompressionLevel Optimal

# Verify
if (Test-Path $ZipPath) {
    $zipInfo = Get-Item $ZipPath
    $sizeMB = [math]::Round($zipInfo.Length / 1MB, 2)
    Write-Host ""
    Write-Host "SUCCESS: Created $ZipName ($sizeMB MB)" -ForegroundColor Green
    Write-Host "Location: $((Resolve-Path $ZipPath).Path)" -ForegroundColor Gray
    Write-Host ""
    Write-Host "Installation:" -ForegroundColor Cyan
    Write-Host "  1. Open MusicBee" -ForegroundColor White
    Write-Host "  2. Edit > Preferences > Plugins" -ForegroundColor White
    Write-Host "  3. Click 'Add Plugin'" -ForegroundColor White
    Write-Host "  4. Select $ZipName" -ForegroundColor White
    Write-Host "  5. Restart MusicBee" -ForegroundColor White
} else {
    Write-Host "ERROR: Failed to create ZIP" -ForegroundColor Red
    exit 1
}
