# Package MBRC Plugin source files for distribution
# Excludes: .claude, .github, .vs, bin, obj, packages, build, deploy, TestResults
# Also excludes: _startclaudehere.cmd, msbuild.log, .gitignore, *.user, *.suo, *.cache

$scriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
$sourceRoot = $scriptDir

# Extract version from AssemblyInfo.cs
$assemblyInfoPath = Join-Path $sourceRoot "plugin\Properties\AssemblyInfo.cs"
$version = "0.0.0.0"
if (Test-Path $assemblyInfoPath) {
    $assemblyInfo = Get-Content $assemblyInfoPath -Raw
    if ($assemblyInfo -match 'AssemblyFileVersion\("([^"]+)"\)') {
        $version = $matches[1]
    }
}

# Get BuildEpoch from AssemblyInfo.cs (embedded during build, matches what InfoWindow displays)
$hexTime = "00000000"
if ($assemblyInfo -match 'AssemblyMetadata\("BuildEpoch",\s*"([^"]+)"\)') {
    $hexTime = $matches[1]
}

$zipName = "Halrad-fork-mbrc-plugin-main-server-${version}-${hexTime}_src.zip"
$outputDir = Join-Path $scriptDir "deploy"

# Ensure output directory exists
if (-not (Test-Path $outputDir)) {
    New-Item -ItemType Directory -Path $outputDir -Force | Out-Null
}

$zipPath = Join-Path $outputDir $zipName

# Move existing source ZIPs to archive before creating new one
$archiveDir = Join-Path $outputDir "archive"
if (-not (Test-Path $archiveDir)) {
    New-Item -ItemType Directory -Path $archiveDir -Force | Out-Null
}
$existingZips = Get-ChildItem -Path $outputDir -Filter "Halrad-fork-*_src.zip" -File -ErrorAction SilentlyContinue
foreach ($zip in $existingZips) {
    $archiveDest = Join-Path $archiveDir $zip.Name
    if (-not (Test-Path $archiveDest)) {
        Move-Item $zip.FullName $archiveDest -Force
        Write-Host "Archived: $($zip.Name)" -ForegroundColor Gray
    } else {
        Remove-Item $zip.FullName -Force
    }
}

# Create temp directory for staging
$tempDir = Join-Path $env:TEMP "mbrc-plugin-src-$(Get-Date -Format 'yyyyMMddHHmmss')"
New-Item -ItemType Directory -Path $tempDir -Force | Out-Null

# Directories and files to exclude
$excludeDirs = @('.claude', '.github', '.vs', 'bin', 'obj', 'packages', 'build', 'deploy', 'TestResults')
$excludeFiles = @('*claude*.cmd', 'msbuild.log', '.gitignore', '*.user', '*.suo', '*.cache', 'nul')

# Copy files, excluding specified items
Get-ChildItem -Path $sourceRoot -Force | Where-Object {
    $item = $_
    $exclude = $false

    # Check directory exclusions
    if ($item.PSIsContainer) {
        $exclude = $excludeDirs -contains $item.Name
    }

    # Check file exclusions
    if (-not $item.PSIsContainer) {
        foreach ($pattern in $excludeFiles) {
            if ($item.Name -like $pattern) {
                $exclude = $true
                break
            }
        }
    }

    -not $exclude
} | ForEach-Object {
    $dest = Join-Path $tempDir $_.Name
    if ($_.PSIsContainer) {
        # Copy directory, excluding bin/obj subdirectories
        Copy-Item -Path $_.FullName -Destination $dest -Recurse -Force -Exclude @('bin', 'obj', '*claude*.cmd', 'msbuild.log', '.gitignore', '*.user', '*.suo', '*.cache')
        # Remove any nested bin/obj/.vs folders that got copied
        Get-ChildItem -Path $dest -Recurse -Directory -Force -ErrorAction SilentlyContinue | Where-Object {
            $_.Name -in @('bin', 'obj', '.vs')
        } | Remove-Item -Recurse -Force -ErrorAction SilentlyContinue
        # Remove excluded files from nested directories
        Get-ChildItem -Path $dest -Recurse -File -Force -ErrorAction SilentlyContinue | Where-Object {
            $_.Name -like '*claude*.cmd' -or $_.Name -eq 'msbuild.log' -or $_.Name -eq '.gitignore' -or $_.Name -like '*.user' -or $_.Name -like '*.suo' -or $_.Name -like '*.cache' -or $_.Name -eq 'nul'
        } | Remove-Item -Force -ErrorAction SilentlyContinue
    } else {
        Copy-Item -Path $_.FullName -Destination $dest -Force
    }
}

# Final cleanup - forcibly remove any claude cmd files that slipped through
Get-ChildItem -Path $tempDir -Recurse -File -Filter '*claude*.cmd' -Force -ErrorAction SilentlyContinue | Remove-Item -Force -ErrorAction SilentlyContinue
Get-ChildItem -Path $tempDir -File -Filter '*claude*.cmd' -Force -ErrorAction SilentlyContinue | Remove-Item -Force -ErrorAction SilentlyContinue

# Create zip
Compress-Archive -Path "$tempDir\*" -DestinationPath $zipPath -Force

# Cleanup temp directory
Remove-Item -Path $tempDir -Recurse -Force

# Report
if (Test-Path $zipPath) {
    $zipSize = (Get-Item $zipPath).Length / 1KB
    Write-Host "Created: $zipPath" -ForegroundColor Green
    Write-Host "Size: $([math]::Round($zipSize, 2)) KB" -ForegroundColor Cyan
} else {
    Write-Host "ERROR: Failed to create $zipName" -ForegroundColor Red
}
