# Package MBRC Protocol SDK for distribution
# Creates a clean SDK package for protocol implementers
#
# Excludes: .claude, .github, .vs, build, deploy, Documentation, packages
#           .gitignore, _startclaudehere.cmd, *.user, *.suo, *.cache, bin, obj
#
# Includes: plugin source, mbrcpval, firewall-utility, and core project files

$ErrorActionPreference = "Stop"

$scriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
$sourceRoot = $scriptDir

Write-Host "MBRC Protocol SDK Packager" -ForegroundColor Cyan
Write-Host ""

# Extract version from AssemblyInfo.cs
$assemblyInfoPath = Join-Path $sourceRoot "plugin\Properties\AssemblyInfo.cs"
$version = "0.0.0.0"
$hexTime = "00000000"

if (Test-Path $assemblyInfoPath) {
    $assemblyInfo = Get-Content $assemblyInfoPath -Raw
    if ($assemblyInfo -match 'AssemblyFileVersion\("([^"]+)"\)') {
        $version = $matches[1]
    }
    if ($assemblyInfo -match 'AssemblyMetadata\("BuildEpoch",\s*"([^"]+)"\)') {
        $hexTime = $matches[1]
    }
}

Write-Host "Version: $version-$hexTime" -ForegroundColor Gray

$zipName = "MBRC-Protocol-SDK-${version}-${hexTime}.zip"
$outputDir = Join-Path $scriptDir "deploy"

# Ensure output directory exists
if (-not (Test-Path $outputDir)) {
    New-Item -ItemType Directory -Path $outputDir -Force | Out-Null
}

$zipPath = Join-Path $outputDir $zipName

# Archive existing SDK ZIPs
$archiveDir = Join-Path $outputDir "archive"
if (-not (Test-Path $archiveDir)) {
    New-Item -ItemType Directory -Path $archiveDir -Force | Out-Null
}

$existingZips = Get-ChildItem -Path $outputDir -Filter "MBRC-Protocol-SDK-*.zip" -File -ErrorAction SilentlyContinue
foreach ($zip in $existingZips) {
    $archiveDest = Join-Path $archiveDir $zip.Name
    if (-not (Test-Path $archiveDest)) {
        Move-Item $zip.FullName $archiveDest -Force
        Write-Host "Archived: $($zip.Name)" -ForegroundColor Gray
    } else {
        Remove-Item $zip.FullName -Force
    }
}

# Create temp directory for staging
$tempDir = Join-Path $env:TEMP "mbrc-sdk-$(Get-Date -Format 'yyyyMMddHHmmss')"
$sdkRoot = Join-Path $tempDir "MBRC-Protocol-SDK"
New-Item -ItemType Directory -Path $sdkRoot -Force | Out-Null

Write-Host "Staging SDK files..." -ForegroundColor Cyan

# =============================================================================
# EXCLUSION LISTS
# =============================================================================

# Top-level directories to EXCLUDE entirely
$excludeTopDirs = @(
    '.claude',
    '.github',
    '.vs',
    'build',
    'deploy',
    'Documentation',
    'packages',
    'TestResults'
)

# Files to exclude (patterns)
$excludeFilePatterns = @(
    '*claude*.cmd',
    '*claude*.md',
    '_startclaudehere*',
    'msbuild.log',
    '.gitignore',
    '.gitattributes',
    '*.user',
    '*.suo',
    '*.cache',
    '*.DotSettings*',
    'nul'
)

# Nested directories to exclude (in any subdirectory)
$excludeNestedDirs = @('bin', 'obj', '.vs', 'TestResults')

# =============================================================================
# COPY LOGIC
# =============================================================================

# Get top-level items, filtering exclusions
$topLevelItems = Get-ChildItem -Path $sourceRoot -Force | Where-Object {
    $item = $_
    $exclude = $false

    # Check top-level directory exclusions
    if ($item.PSIsContainer) {
        $exclude = $excludeTopDirs -contains $item.Name
    }

    # Check file pattern exclusions
    if (-not $item.PSIsContainer) {
        foreach ($pattern in $excludeFilePatterns) {
            if ($item.Name -like $pattern) {
                $exclude = $true
                break
            }
        }
    }

    -not $exclude
}

foreach ($item in $topLevelItems) {
    $dest = Join-Path $sdkRoot $item.Name

    if ($item.PSIsContainer) {
        # Copy directory recursively
        Copy-Item -Path $item.FullName -Destination $dest -Recurse -Force

        # Remove excluded nested directories
        Get-ChildItem -Path $dest -Recurse -Directory -Force -ErrorAction SilentlyContinue |
            Where-Object { $excludeNestedDirs -contains $_.Name } |
            Remove-Item -Recurse -Force -ErrorAction SilentlyContinue

        # Remove excluded files from all subdirectories
        Get-ChildItem -Path $dest -Recurse -File -Force -ErrorAction SilentlyContinue |
            Where-Object {
                $file = $_
                $shouldExclude = $false
                foreach ($pattern in $excludeFilePatterns) {
                    if ($file.Name -like $pattern) {
                        $shouldExclude = $true
                        break
                    }
                }
                $shouldExclude
            } | Remove-Item -Force -ErrorAction SilentlyContinue
    } else {
        # Copy single file
        Copy-Item -Path $item.FullName -Destination $dest -Force
    }
}

# =============================================================================
# CREATE SDK README
# =============================================================================

$sdkReadme = @"
# MBRC Protocol SDK v$version

MusicBee Remote Control Protocol SDK for building compatible clients.

## Contents

| Folder | Purpose |
|--------|---------|
| `plugin/` | MBRC Server Plugin source code |
| `mbrcpval/` | Protocol Validator CLI toolkit (52+ tests) |
| `firewall-utility/` | Windows Firewall configuration helper |
| `mbrc-plugin.Tests/` | Unit tests |

## Documentation

- `README.md` - Project overview
- `plugin_CHANGELOG.md` - Version history
- `mbrcpval/README.md` - Validator documentation
- `mbrcpval/Docs/PROTOCOL-SDK-MIRROR.md` - Protocol specification (markdown)

## Quick Start

### Build the Plugin
``````powershell
# Using MSBuild
msbuild plugin/mbrc-plugin.csproj /p:Configuration=Release
``````

### Build MBRCPVAL (Validator)
``````powershell
cd mbrcpval
dotnet build -c Release
``````

### Run Protocol Tests
``````powershell
mbrcpval server --host localhost --port 3000 --tags mandatory
``````

## Protocol Overview

- **Transport:** TCP with JSON messages, CRLF terminated
- **Default Port:** 3000 (configurable)
- **Discovery:** UDP Multicast 239.1.5.10:45345
- **Versions:** v4.0 (standard), v4.5 (extended features)

See `mbrcpval/Docs/PROTOCOL-SDK-MIRROR.md` for full specification.

## License

Licensed under GPLv3. See LICENSE file.

---

*MBRC Protocol SDK v$version | Generated $(Get-Date -Format 'yyyy-MM-dd')*
"@

$sdkReadme | Out-File -FilePath (Join-Path $sdkRoot "SDK-README.md") -Encoding UTF8

# =============================================================================
# CREATE ZIP
# =============================================================================

Write-Host "Creating ZIP archive..." -ForegroundColor Cyan

Compress-Archive -Path "$sdkRoot" -DestinationPath $zipPath -Force

# Cleanup temp directory
Remove-Item -Path $tempDir -Recurse -Force

# =============================================================================
# REPORT
# =============================================================================

if (Test-Path $zipPath) {
    $zipInfo = Get-Item $zipPath
    $sizeMB = [math]::Round($zipInfo.Length / 1MB, 2)
    $sizeKB = [math]::Round($zipInfo.Length / 1KB, 2)

    Write-Host ""
    Write-Host "SUCCESS: Created $zipName" -ForegroundColor Green
    if ($sizeMB -ge 1) {
        Write-Host "Size: $sizeMB MB" -ForegroundColor Cyan
    } else {
        Write-Host "Size: $sizeKB KB" -ForegroundColor Cyan
    }
    Write-Host "Location: $((Resolve-Path $zipPath).Path)" -ForegroundColor Gray
    Write-Host ""
    Write-Host "SDK Contents:" -ForegroundColor Cyan
    Write-Host "  - Plugin source code" -ForegroundColor White
    Write-Host "  - MBRCPVAL validator toolkit" -ForegroundColor White
    Write-Host "  - Protocol documentation" -ForegroundColor White
    Write-Host "  - Unit tests" -ForegroundColor White
} else {
    Write-Host "ERROR: Failed to create ZIP" -ForegroundColor Red
    exit 1
}
