# MBRC Plugin - Master Packaging Script
# Creates all distribution packages:
#   1. Plugin Binary   - MBRC-Plugin-{version}.zip (for end users)
#   2. Plugin Source   - Halrad-fork-*_src.zip (GPL compliance)
#   3. Protocol SDK    - MBRC-Protocol-SDK-{version}.zip (for developers)

param(
    [switch]$Binary,
    [switch]$Source,
    [switch]$SDK,
    [switch]$All
)

$ErrorActionPreference = "Stop"

Write-Host ""
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "  MBRC Plugin Distribution Packager" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host ""

# If no flags specified, default to -All
if (-not ($Binary -or $Source -or $SDK -or $All)) {
    $All = $true
}

if ($All) {
    $Binary = $true
    $Source = $true
    $SDK = $true
}

$scriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path

# Extract version for summary
$assemblyInfoPath = Join-Path $scriptDir "plugin\Properties\AssemblyInfo.cs"
$version = "unknown"
$hexTime = ""
if (Test-Path $assemblyInfoPath) {
    $assemblyInfo = Get-Content $assemblyInfoPath -Raw
    if ($assemblyInfo -match 'AssemblyFileVersion\("([^"]+)"\)') {
        $version = $matches[1]
    }
    if ($assemblyInfo -match 'AssemblyMetadata\("BuildEpoch",\s*"([^"]+)"\)') {
        $hexTime = $matches[1]
    }
}

Write-Host "Version: $version" -ForegroundColor Gray
if ($hexTime) { Write-Host "Epoch: $hexTime" -ForegroundColor Gray }
Write-Host ""

$results = @()

# =============================================================================
# 1. BINARY PACKAGE
# =============================================================================
if ($Binary) {
    Write-Host "----------------------------------------" -ForegroundColor DarkGray
    Write-Host "1. Creating Binary Package..." -ForegroundColor Yellow
    Write-Host "----------------------------------------" -ForegroundColor DarkGray

    $binaryScript = Join-Path $scriptDir "package.ps1"
    if (Test-Path $binaryScript) {
        try {
            & $binaryScript
            $results += @{ Name = "Binary"; Status = "OK"; File = "MBRC-Plugin-$version-$hexTime.zip" }
        } catch {
            Write-Host "ERROR: Binary package failed - $_" -ForegroundColor Red
            $results += @{ Name = "Binary"; Status = "FAILED"; File = "" }
        }
    } else {
        Write-Host "ERROR: package.ps1 not found" -ForegroundColor Red
        $results += @{ Name = "Binary"; Status = "MISSING"; File = "" }
    }
    Write-Host ""
}

# =============================================================================
# 2. SOURCE PACKAGE
# =============================================================================
if ($Source) {
    Write-Host "----------------------------------------" -ForegroundColor DarkGray
    Write-Host "2. Creating Source Package..." -ForegroundColor Yellow
    Write-Host "----------------------------------------" -ForegroundColor DarkGray

    $sourceScript = Join-Path $scriptDir "package-source.ps1"
    if (Test-Path $sourceScript) {
        try {
            & $sourceScript
            $results += @{ Name = "Source"; Status = "OK"; File = "Halrad-fork-mbrc-plugin-main-server-$version-${hexTime}_src.zip" }
        } catch {
            Write-Host "ERROR: Source package failed - $_" -ForegroundColor Red
            $results += @{ Name = "Source"; Status = "FAILED"; File = "" }
        }
    } else {
        Write-Host "ERROR: package-source.ps1 not found" -ForegroundColor Red
        $results += @{ Name = "Source"; Status = "MISSING"; File = "" }
    }
    Write-Host ""
}

# =============================================================================
# 3. SDK PACKAGE
# =============================================================================
if ($SDK) {
    Write-Host "----------------------------------------" -ForegroundColor DarkGray
    Write-Host "3. Creating SDK Package..." -ForegroundColor Yellow
    Write-Host "----------------------------------------" -ForegroundColor DarkGray

    $sdkScript = Join-Path $scriptDir "package-sdk.ps1"
    if (Test-Path $sdkScript) {
        try {
            & $sdkScript
            $results += @{ Name = "SDK"; Status = "OK"; File = "MBRC-Protocol-SDK-$version-$hexTime.zip" }
        } catch {
            Write-Host "ERROR: SDK package failed - $_" -ForegroundColor Red
            $results += @{ Name = "SDK"; Status = "FAILED"; File = "" }
        }
    } else {
        Write-Host "ERROR: package-sdk.ps1 not found" -ForegroundColor Red
        $results += @{ Name = "SDK"; Status = "MISSING"; File = "" }
    }
    Write-Host ""
}

# =============================================================================
# SUMMARY
# =============================================================================
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "  Packaging Summary" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host ""

$deployDir = Join-Path $scriptDir "deploy"

foreach ($result in $results) {
    $statusColor = switch ($result.Status) {
        "OK" { "Green" }
        "FAILED" { "Red" }
        default { "Yellow" }
    }

    $icon = switch ($result.Status) {
        "OK" { "[OK]" }
        "FAILED" { "[X]" }
        default { "[?]" }
    }

    Write-Host "$icon $($result.Name): " -NoNewline -ForegroundColor $statusColor
    if ($result.File) {
        Write-Host $result.File -ForegroundColor Gray
    } else {
        Write-Host $result.Status -ForegroundColor $statusColor
    }
}

Write-Host ""
Write-Host "Output: $deployDir" -ForegroundColor Gray
Write-Host ""

# =============================================================================
# DISTRIBUTION GUIDE
# =============================================================================
Write-Host "Distribution Guide:" -ForegroundColor Cyan
Write-Host "  Binary  - For end users (MusicBee Add Plugin)" -ForegroundColor White
Write-Host "  Source  - GPL compliance (full source code)" -ForegroundColor White
Write-Host "  SDK     - For developers (protocol implementation)" -ForegroundColor White
Write-Host ""
