using System.Reflection;
using YamlDotNet.Serialization;
using YamlDotNet.Serialization.NamingConventions;

namespace Mbrcpval.Testing;

/// <summary>
/// Loads test cases and test suites from YAML files.
/// </summary>
public class YamlTestLoader
{
    private readonly IDeserializer _deserializer;

    public YamlTestLoader()
    {
        _deserializer = new DeserializerBuilder()
            .WithNamingConvention(CamelCaseNamingConvention.Instance)
            .IgnoreUnmatchedProperties()
            .Build();
    }

    /// <summary>
    /// Loads a test suite from a YAML file path.
    /// </summary>
    /// <param name="yamlPath">Path to the YAML file.</param>
    /// <returns>The loaded test suite.</returns>
    public TestSuite LoadTestSuite(string yamlPath)
    {
        if (!File.Exists(yamlPath))
            throw new FileNotFoundException($"Test suite file not found: {yamlPath}");

        var yaml = File.ReadAllText(yamlPath);
        var suite = LoadTestSuiteFromString(yaml);
        suite.SourceFile = yamlPath;

        // Set source file on each test
        foreach (var test in suite.Tests)
        {
            test.SourceFile = yamlPath;
        }

        return suite;
    }

    /// <summary>
    /// Loads a test suite from a YAML string.
    /// </summary>
    /// <param name="yaml">YAML content.</param>
    /// <returns>The loaded test suite.</returns>
    public TestSuite LoadTestSuiteFromString(string yaml)
    {
        var rawSuite = _deserializer.Deserialize<YamlTestSuite>(yaml);
        return ConvertToTestSuite(rawSuite);
    }

    /// <summary>
    /// Loads all test suites from a directory.
    /// </summary>
    /// <param name="directory">Directory containing YAML files.</param>
    /// <param name="pattern">File pattern to match (default: *.yaml).</param>
    /// <returns>List of loaded test suites.</returns>
    public List<TestSuite> LoadAllSuites(string directory, string pattern = "*.yaml")
    {
        if (!Directory.Exists(directory))
            throw new DirectoryNotFoundException($"Test suite directory not found: {directory}");

        var suites = new List<TestSuite>();
        var files = Directory.GetFiles(directory, pattern, SearchOption.AllDirectories);

        foreach (var file in files)
        {
            try
            {
                var suite = LoadTestSuite(file);
                suites.Add(suite);
            }
            catch (Exception ex)
            {
                // Log error but continue loading other files
                Console.Error.WriteLine($"Warning: Failed to load {file}: {ex.Message}");
            }
        }

        return suites;
    }

    /// <summary>
    /// Loads test suites from embedded resources.
    /// </summary>
    /// <param name="assembly">Assembly containing embedded resources (default: calling assembly).</param>
    /// <param name="resourcePrefix">Resource name prefix to filter by.</param>
    /// <returns>List of loaded test suites.</returns>
    public List<TestSuite> LoadFromEmbeddedResources(Assembly? assembly = null, string? resourcePrefix = null)
    {
        assembly ??= Assembly.GetCallingAssembly();
        var suites = new List<TestSuite>();

        var resourceNames = assembly.GetManifestResourceNames()
            .Where(name => name.EndsWith(".yaml", StringComparison.OrdinalIgnoreCase));

        if (!string.IsNullOrEmpty(resourcePrefix))
        {
            resourceNames = resourceNames.Where(name =>
                name.StartsWith(resourcePrefix, StringComparison.OrdinalIgnoreCase));
        }

        foreach (var resourceName in resourceNames)
        {
            try
            {
                using var stream = assembly.GetManifestResourceStream(resourceName);
                if (stream == null) continue;

                using var reader = new StreamReader(stream);
                var yaml = reader.ReadToEnd();
                var suite = LoadTestSuiteFromString(yaml);
                suite.SourceFile = resourceName;

                foreach (var test in suite.Tests)
                {
                    test.SourceFile = resourceName;
                }

                suites.Add(suite);
            }
            catch (Exception ex)
            {
                Console.Error.WriteLine($"Warning: Failed to load embedded resource {resourceName}: {ex.Message}");
            }
        }

        return suites;
    }

    /// <summary>
    /// Loads all tests from a directory and returns a flattened list.
    /// </summary>
    public List<TestCase> LoadAllTests(string directory, string pattern = "*.yaml")
    {
        var suites = LoadAllSuites(directory, pattern);
        return suites.SelectMany(s => s.Tests).ToList();
    }

    private TestSuite ConvertToTestSuite(YamlTestSuite raw)
    {
        var suite = new TestSuite
        {
            Name = raw.Name ?? "Unnamed Suite",
            Description = raw.Description ?? string.Empty,
            Version = raw.Version ?? "1.0"
        };

        if (raw.Defaults != null)
        {
            suite.Defaults = new TestSuiteDefaults
            {
                Timeout = ParseTimeSpan(raw.Defaults.Timeout, TimeSpan.FromSeconds(30)),
                Tags = raw.Defaults.Tags ?? new List<string>(),
                Prerequisites = raw.Defaults.Prerequisites ?? new List<string>()
            };
        }

        foreach (var rawTest in raw.Tests ?? new List<YamlTestCase>())
        {
            var testCase = ConvertToTestCase(rawTest, suite.Defaults);
            suite.Tests.Add(testCase);
        }

        return suite;
    }

    private TestCase ConvertToTestCase(YamlTestCase raw, TestSuiteDefaults defaults)
    {
        var testCase = new TestCase
        {
            Id = raw.Id ?? GenerateId(),
            Name = raw.Name ?? "Unnamed Test",
            Description = raw.Description ?? string.Empty,
            Category = raw.Category ?? "general",
            Prerequisites = MergeList(defaults.Prerequisites, raw.Prerequisites),
            Tags = MergeList(defaults.Tags, raw.Tags),
            Timeout = ParseTimeSpan(raw.Timeout, defaults.Timeout),
            Skip = raw.Skip ?? false,
            SkipReason = raw.SkipReason,
            Priority = raw.Priority ?? 100
        };

        foreach (var rawStep in raw.Steps ?? new List<YamlTestStep>())
        {
            var step = ConvertToTestStep(rawStep);
            testCase.Steps.Add(step);
        }

        foreach (var rawStep in raw.Cleanup ?? new List<YamlTestStep>())
        {
            var step = ConvertToTestStep(rawStep);
            testCase.Cleanup.Add(step);
        }

        return testCase;
    }

    private TestStep ConvertToTestStep(YamlTestStep raw)
    {
        var step = new TestStep
        {
            Action = ParseStepAction(raw.Action),
            Context = raw.Context,
            Data = raw.Data,
            ExpectedContext = raw.ExpectedContext ?? raw.Expect,
            Description = raw.Description,
            Timeout = ParseTimeSpan(raw.Timeout, null),
            WaitDuration = ParseTimeSpan(raw.Wait, null),
            VariableName = raw.Variable,
            VariableValue = raw.Value,
            LogMessage = raw.Message,
            LogLevel = raw.Level ?? "Info"
        };

        foreach (var rawAssertion in raw.Assertions ?? new List<YamlAssertion>())
        {
            var assertion = ConvertToAssertion(rawAssertion);
            step.Assertions.Add(assertion);
        }

        // Handle shorthand assertion syntax
        if (raw.Assert != null)
        {
            foreach (var kvp in raw.Assert)
            {
                step.Assertions.Add(new Assertion
                {
                    Type = AssertionType.Equals,
                    Path = kvp.Key,
                    Expected = kvp.Value
                });
            }
        }

        return step;
    }

    private Assertion ConvertToAssertion(YamlAssertion raw)
    {
        return new Assertion
        {
            Type = ParseAssertionType(raw.Type),
            Path = raw.Path ?? string.Empty,
            Expected = raw.Expected ?? raw.Value,
            Message = raw.Message
        };
    }

    private StepAction ParseStepAction(string? action)
    {
        if (string.IsNullOrEmpty(action))
            return StepAction.Send;

        return action.ToLowerInvariant() switch
        {
            "connect" => StepAction.Connect,
            "disconnect" => StepAction.Disconnect,
            "send" => StepAction.Send,
            "receive" => StepAction.Receive,
            "receiveany" or "receive_any" or "receive-any" => StepAction.ReceiveAny,
            "wait" => StepAction.Wait,
            "assert" => StepAction.Assert,
            "setvariable" or "set_variable" or "set-variable" or "set" => StepAction.SetVariable,
            "log" => StepAction.Log,
            _ => throw new ArgumentException($"Unknown step action: {action}")
        };
    }

    private AssertionType ParseAssertionType(string? type)
    {
        if (string.IsNullOrEmpty(type))
            return AssertionType.Equals;

        return type.ToLowerInvariant() switch
        {
            "equals" or "eq" or "=" or "==" => AssertionType.Equals,
            "notequals" or "not_equals" or "neq" or "!=" or "<>" => AssertionType.NotEquals,
            "contains" => AssertionType.Contains,
            "notcontains" or "not_contains" => AssertionType.NotContains,
            "matches" or "regex" => AssertionType.Matches,
            "exists" => AssertionType.Exists,
            "notexists" or "not_exists" => AssertionType.NotExists,
            "greaterthan" or "greater_than" or "gt" or ">" => AssertionType.GreaterThan,
            "lessthan" or "less_than" or "lt" or "<" => AssertionType.LessThan,
            "greaterorequal" or "greater_or_equal" or "gte" or ">=" => AssertionType.GreaterOrEqual,
            "lessorequal" or "less_or_equal" or "lte" or "<=" => AssertionType.LessOrEqual,
            "istype" or "is_type" or "type" => AssertionType.IsType,
            "schemavalid" or "schema_valid" or "schema" => AssertionType.SchemaValid,
            _ => throw new ArgumentException($"Unknown assertion type: {type}")
        };
    }

    private TimeSpan ParseTimeSpan(string? value, TimeSpan? defaultValue)
    {
        if (string.IsNullOrEmpty(value))
            return defaultValue ?? TimeSpan.Zero;

        // Try parsing as milliseconds number
        if (int.TryParse(value, out var ms))
            return TimeSpan.FromMilliseconds(ms);

        // Try parsing duration strings like "5s", "100ms", "1m"
        value = value.Trim().ToLowerInvariant();

        if (value.EndsWith("ms"))
        {
            if (int.TryParse(value[..^2], out var milliseconds))
                return TimeSpan.FromMilliseconds(milliseconds);
        }
        else if (value.EndsWith("s"))
        {
            if (double.TryParse(value[..^1], out var seconds))
                return TimeSpan.FromSeconds(seconds);
        }
        else if (value.EndsWith("m"))
        {
            if (double.TryParse(value[..^1], out var minutes))
                return TimeSpan.FromMinutes(minutes);
        }

        // Try parsing as TimeSpan
        if (TimeSpan.TryParse(value, out var result))
            return result;

        return defaultValue ?? TimeSpan.Zero;
    }

    private List<string> MergeList(List<string> defaults, List<string>? overrides)
    {
        var result = new List<string>(defaults);
        if (overrides != null)
        {
            foreach (var item in overrides)
            {
                if (!result.Contains(item, StringComparer.OrdinalIgnoreCase))
                    result.Add(item);
            }
        }
        return result;
    }

    private static int _idCounter;
    private static string GenerateId() => $"TEST-{Interlocked.Increment(ref _idCounter):D3}";

    #region YAML DTOs

    private class YamlTestSuite
    {
        public string? Name { get; set; }
        public string? Description { get; set; }
        public string? Version { get; set; }
        public YamlDefaults? Defaults { get; set; }
        public List<YamlTestCase>? Tests { get; set; }
    }

    private class YamlDefaults
    {
        public string? Timeout { get; set; }
        public List<string>? Tags { get; set; }
        public List<string>? Prerequisites { get; set; }
    }

    private class YamlTestCase
    {
        public string? Id { get; set; }
        public string? Name { get; set; }
        public string? Description { get; set; }
        public string? Category { get; set; }
        public List<string>? Prerequisites { get; set; }
        public List<YamlTestStep>? Steps { get; set; }
        public List<YamlTestStep>? Cleanup { get; set; }
        public List<string>? Tags { get; set; }
        public string? Timeout { get; set; }
        public bool? Skip { get; set; }
        public string? SkipReason { get; set; }
        public int? Priority { get; set; }
    }

    private class YamlTestStep
    {
        public string? Action { get; set; }
        public string? Context { get; set; }
        public object? Data { get; set; }
        public string? ExpectedContext { get; set; }
        public string? Expect { get; set; } // Alias for ExpectedContext
        public List<YamlAssertion>? Assertions { get; set; }
        public Dictionary<string, object>? Assert { get; set; } // Shorthand assertions
        public string? Timeout { get; set; }
        public string? Description { get; set; }
        public string? Wait { get; set; } // For Wait action
        public string? Variable { get; set; } // For SetVariable action
        public string? Value { get; set; } // For SetVariable action
        public string? Message { get; set; } // For Log action
        public string? Level { get; set; } // For Log action
    }

    private class YamlAssertion
    {
        public string? Type { get; set; }
        public string? Path { get; set; }
        public object? Expected { get; set; }
        public object? Value { get; set; } // Alias for Expected
        public string? Message { get; set; }
    }

    #endregion
}
