namespace Mbrcpval.Testing;

/// <summary>
/// Factory for creating protocol test suites.
/// </summary>
public static class TestSuiteFactory
{
    /// <summary>
    /// Creates all test cases for the specified protocol version.
    /// </summary>
    /// <param name="protocolVersion">Protocol version ("4.0" or "4.5")</param>
    /// <returns>List of test cases appropriate for the protocol version.</returns>
    public static List<TestCase> CreateTestSuite(string protocolVersion)
    {
        var tests = new List<TestCase>();

        // Core tests - required for all protocol versions
        tests.AddRange(CreateCoreTests());

        // Protocol 4.0 tests
        tests.AddRange(CreateProtocol40Tests());

        // Protocol 4.5 additional tests
        if (protocolVersion == "4.5")
        {
            tests.AddRange(CreateProtocol45Tests());
        }

        return tests;
    }

    /// <summary>
    /// Creates tests filtered by suite name.
    /// </summary>
    public static List<TestCase> CreateTestSuite(string protocolVersion, string suite)
    {
        if (string.Equals(suite, "all", StringComparison.OrdinalIgnoreCase))
        {
            return CreateTestSuite(protocolVersion);
        }

        var allTests = CreateTestSuite(protocolVersion);
        return allTests.Where(t => t.Category.Equals(suite, StringComparison.OrdinalIgnoreCase)).ToList();
    }

    /// <summary>
    /// Core tests that verify fundamental protocol functionality.
    /// These tests must pass for any compliant MBRC implementation.
    /// </summary>
    private static List<TestCase> CreateCoreTests()
    {
        return new List<TestCase>
        {
            // CORE-001: Protocol Handshake
            new TestCase
            {
                Id = "CORE-001",
                Name = "Protocol Handshake",
                Description = "Verify server responds to protocol version request",
                Category = "core",
                Priority = 1,
                Tags = new List<string> { "smoke", "handshake" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("player", description: "Request protocol info"),
                    TestStep.Receive("player", "Receive player response",
                        Assertion.Exists("$.context", "Response has context"),
                        Assertion.Equals("$.context", "player", "Context is 'player'"))
                }
            },

            // CORE-002: Ping/Pong
            new TestCase
            {
                Id = "CORE-002",
                Name = "Ping Pong",
                Description = "Verify server responds to ping with pong",
                Category = "core",
                Priority = 1,
                Tags = new List<string> { "smoke", "keepalive" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("ping", description: "Send ping"),
                    TestStep.Receive("pong", "Receive pong response",
                        Assertion.Equals("$.context", "pong", "Context is 'pong'"))
                }
            },

            // CORE-003: Player State Request
            // Note: Old plugin uses 'playerstatus' context, not 'playerstate'
            new TestCase
            {
                Id = "CORE-003",
                Name = "Player State Request",
                Description = "Verify server returns complete player state",
                Category = "core",
                Priority = 2,
                Tags = new List<string> { "smoke", "player" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("playerstatus", description: "Request player status"),
                    TestStep.Receive("playerstatus", "Receive player status",
                        Assertion.Exists("$.data", "Response has data"),
                        Assertion.Exists("$.data.playerstate", "Has playerstate field"),
                        Assertion.Exists("$.data.playervolume", "Has playervolume field"),
                        Assertion.Exists("$.data.playermute", "Has playermute field"),
                        Assertion.Exists("$.data.playershuffle", "Has playershuffle field"),
                        Assertion.Exists("$.data.playerrepeat", "Has playerrepeat field"))
                }
            },

            // CORE-004: Invalid Context Handling
            new TestCase
            {
                Id = "CORE-004",
                Name = "Invalid Context Handling",
                Description = "Verify server handles unknown context gracefully",
                Category = "core",
                Priority = 3,
                Tags = new List<string> { "error-handling" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("invalid_context_xyz", description: "Send invalid context"),
                    TestStep.ReceiveAny("Receive response (error or empty)",
                        Assertion.Exists("$.context", "Response has context"))
                }
            },

            // CORE-005: Empty Data Handling
            new TestCase
            {
                Id = "CORE-005",
                Name = "Empty Data Request",
                Description = "Verify player context with empty data returns state",
                Category = "core",
                Priority = 2,
                Tags = new List<string> { "player" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("player", "", description: "Send player with empty data"),
                    TestStep.Receive("player", "Receive player response",
                        Assertion.Exists("$.context", "Response has context"))
                }
            }
        };
    }

    /// <summary>
    /// Protocol 4.0 tests - standard MBRC functionality.
    /// </summary>
    private static List<TestCase> CreateProtocol40Tests()
    {
        return new List<TestCase>
        {
            // Playback Tests
            // PLY-001: Get Player Status
            new TestCase
            {
                Id = "PLY-001",
                Name = "Get Player Status",
                Description = "Request current playback status",
                Category = "playback",
                Priority = 10,
                Tags = new List<string> { "playback", "status" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("playerstatus", description: "Request player status"),
                    TestStep.Receive("playerstatus", "Receive status",
                        Assertion.Exists("$.data", "Has data"))
                }
            },

            // PLY-002: Get Volume Level
            // Note: Volume is available in playerstatus response
            new TestCase
            {
                Id = "PLY-002",
                Name = "Get Volume Level",
                Description = "Request current volume level via playerstatus",
                Category = "playback",
                Priority = 10,
                Tags = new List<string> { "playback", "volume" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("playerstatus", description: "Request player status for volume"),
                    TestStep.Receive("playerstatus", "Receive status with volume",
                        Assertion.Exists("$.data.playervolume", "Has volume data"))
                }
            },

            // PLY-003: Get Shuffle State
            new TestCase
            {
                Id = "PLY-003",
                Name = "Get Shuffle State",
                Description = "Request current shuffle setting",
                Category = "playback",
                Priority = 10,
                Tags = new List<string> { "playback", "shuffle" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("playershuffle", description: "Request shuffle"),
                    TestStep.Receive("playershuffle", "Receive shuffle state",
                        Assertion.Exists("$.data", "Has shuffle data"))
                }
            },

            // PLY-004: Get Repeat State
            new TestCase
            {
                Id = "PLY-004",
                Name = "Get Repeat State",
                Description = "Request current repeat setting",
                Category = "playback",
                Priority = 10,
                Tags = new List<string> { "playback", "repeat" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("playerrepeat", description: "Request repeat"),
                    TestStep.Receive("playerrepeat", "Receive repeat state",
                        Assertion.Exists("$.data", "Has repeat data"))
                }
            },

            // PLY-005: Get Mute State
            new TestCase
            {
                Id = "PLY-005",
                Name = "Get Mute State",
                Description = "Request current mute setting",
                Category = "playback",
                Priority = 10,
                Tags = new List<string> { "playback", "mute" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("playermute", description: "Request mute"),
                    TestStep.Receive("playermute", "Receive mute state",
                        Assertion.Exists("$.data", "Has mute data"))
                }
            },

            // PLY-006: Get Scrobble State
            new TestCase
            {
                Id = "PLY-006",
                Name = "Get Scrobble State",
                Description = "Request current scrobble setting",
                Category = "playback",
                Priority = 10,
                Tags = new List<string> { "playback", "scrobble" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("scrobbler", description: "Request scrobble"),
                    TestStep.Receive("scrobbler", "Receive scrobble state",
                        Assertion.Exists("$.data", "Has scrobble data"))
                }
            },

            // Now Playing Tests
            // NP-001: Get Now Playing Track
            new TestCase
            {
                Id = "NP-001",
                Name = "Get Now Playing Track",
                Description = "Request current track information",
                Category = "nowplaying",
                Priority = 20,
                Tags = new List<string> { "nowplaying", "track" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("nowplayingtrack", description: "Request now playing track"),
                    TestStep.Receive("nowplayingtrack", "Receive track info",
                        Assertion.Exists("$.context", "Has context"),
                        Assertion.Equals("$.context", "nowplayingtrack", "Context matches"))
                }
            },

            // NP-002: Get Now Playing Position
            new TestCase
            {
                Id = "NP-002",
                Name = "Get Now Playing Position",
                Description = "Request current playback position",
                Category = "nowplaying",
                Priority = 20,
                Tags = new List<string> { "nowplaying", "position" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("nowplayingposition", description: "Request position"),
                    TestStep.Receive("nowplayingposition", "Receive position",
                        Assertion.Exists("$.data", "Has position data"))
                }
            },

            // NP-003: Get Now Playing Cover
            new TestCase
            {
                Id = "NP-003",
                Name = "Get Now Playing Cover",
                Description = "Request album cover art",
                Category = "nowplaying",
                Priority = 20,
                Tags = new List<string> { "nowplaying", "cover" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("nowplayingcover", description: "Request cover"),
                    TestStep.Receive("nowplayingcover", "Receive cover response",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // NP-004: Get Now Playing Lyrics
            new TestCase
            {
                Id = "NP-004",
                Name = "Get Now Playing Lyrics",
                Description = "Request track lyrics",
                Category = "nowplaying",
                Priority = 20,
                Tags = new List<string> { "nowplaying", "lyrics" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("nowplayinglyrics", description: "Request lyrics"),
                    TestStep.Receive("nowplayinglyrics", "Receive lyrics response",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // NP-005: Get Now Playing Rating
            new TestCase
            {
                Id = "NP-005",
                Name = "Get Now Playing Rating",
                Description = "Request track rating",
                Category = "nowplaying",
                Priority = 20,
                Tags = new List<string> { "nowplaying", "rating" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("nowplayingrating", description: "Request rating"),
                    TestStep.Receive("nowplayingrating", "Receive rating response",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // NP-006: Get Now Playing List (Queue)
            new TestCase
            {
                Id = "NP-006",
                Name = "Get Now Playing List",
                Description = "Request the play queue",
                Category = "nowplaying",
                Priority = 20,
                Tags = new List<string> { "nowplaying", "queue" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("nowplayinglist", description: "Request queue"),
                    TestStep.Receive("nowplayinglist", "Receive queue response",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // Library Tests
            // LIB-001: Browse Artists
            new TestCase
            {
                Id = "LIB-001",
                Name = "Browse Library Artists",
                Description = "Request list of artists",
                Category = "library",
                Priority = 30,
                Tags = new List<string> { "library", "artists" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("browseartists", description: "Request artists"),
                    TestStep.Receive("browseartists", "Receive artists",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // LIB-002: Browse Albums
            new TestCase
            {
                Id = "LIB-002",
                Name = "Browse Library Albums",
                Description = "Request list of albums",
                Category = "library",
                Priority = 30,
                Tags = new List<string> { "library", "albums" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("browsealbums", description: "Request albums"),
                    TestStep.Receive("browsealbums", "Receive albums",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // LIB-003: Browse Tracks
            new TestCase
            {
                Id = "LIB-003",
                Name = "Browse Library Tracks",
                Description = "Request list of tracks",
                Category = "library",
                Priority = 30,
                Tags = new List<string> { "library", "tracks" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("browsetracks", description: "Request tracks"),
                    TestStep.Receive("browsetracks", "Receive tracks",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // LIB-004: Browse Genres
            new TestCase
            {
                Id = "LIB-004",
                Name = "Browse Library Genres",
                Description = "Request list of genres",
                Category = "library",
                Priority = 30,
                Tags = new List<string> { "library", "genres" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("browsegenres", description: "Request genres"),
                    TestStep.Receive("browsegenres", "Receive genres",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // Playlist Tests
            // PLS-001: Get Playlist List
            new TestCase
            {
                Id = "PLS-001",
                Name = "Get Playlist List",
                Description = "Request list of playlists",
                Category = "playlist",
                Priority = 40,
                Tags = new List<string> { "playlist" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("playlistlist", description: "Request playlists"),
                    TestStep.Receive("playlistlist", "Receive playlists",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // Output Tests
            // OUT-001: Get Output Devices
            new TestCase
            {
                Id = "OUT-001",
                Name = "Get Output Devices",
                Description = "Request list of audio output devices",
                Category = "output",
                Priority = 50,
                Tags = new List<string> { "output" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("playeroutput", description: "Request outputs"),
                    TestStep.Receive("playeroutput", "Receive outputs",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // Protocol Tests
            // PROTO-001: Protocol Version Request
            // Note: Old plugin returns integer (e.g., data:4), not object
            new TestCase
            {
                Id = "PROTO-001",
                Name = "Protocol Version",
                Description = "Request protocol version information",
                Category = "protocol",
                Priority = 5,
                Tags = new List<string> { "protocol", "version" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("protocol", description: "Request protocol version"),
                    TestStep.Receive("protocol", "Receive protocol info",
                        Assertion.Exists("$.data", "Has protocol version data"))
                }
            }
        };
    }

    /// <summary>
    /// Protocol 4.5 additional tests - ARiA and enhanced functionality.
    /// </summary>
    private static List<TestCase> CreateProtocol45Tests()
    {
        return new List<TestCase>
        {
            // 4.5-001: Protocol 4.5 Version Check
            // Note: Plugin returns {"context":"protocol","data":4.5} - data is a number, not object
            new TestCase
            {
                Id = "V45-001",
                Name = "Protocol 4.5 Version",
                Description = "Verify server reports protocol version 4.5",
                Category = "protocol",
                Priority = 1,
                Tags = new List<string> { "protocol", "v4.5" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("protocol", new { protocol_version = 4.5 }, "Request protocol v4.5"),
                    TestStep.Receive("protocol", "Receive protocol info",
                        Assertion.Exists("$.data", "Has protocol version data"))
                }
            },

            // 4.5-002: Enhanced Player State
            new TestCase
            {
                Id = "V45-002",
                Name = "Enhanced Player State",
                Description = "Verify 4.5 player state has all required fields",
                Category = "playback",
                Priority = 10,
                Tags = new List<string> { "playback", "v4.5", "enhanced" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("playerstatus", description: "Request player status"),
                    TestStep.Receive("playerstatus", "Receive enhanced state",
                        Assertion.Exists("$.data.playerstate", "Has playerstate"),
                        Assertion.Exists("$.data.playervolume", "Has volume"),
                        Assertion.Exists("$.data.playermute", "Has mute"),
                        Assertion.Exists("$.data.playershuffle", "Has shuffle"),
                        Assertion.Exists("$.data.playerrepeat", "Has repeat"),
                        Assertion.Exists("$.data.scrobbler", "Has scrobbler"))
                }
            },

            // 4.5-003: Enhanced Track Info
            new TestCase
            {
                Id = "V45-003",
                Name = "Enhanced Track Info",
                Description = "Verify 4.5 track info has extended metadata",
                Category = "nowplaying",
                Priority = 20,
                Tags = new List<string> { "nowplaying", "v4.5", "metadata" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("nowplayingtrack", description: "Request track info"),
                    TestStep.Receive("nowplayingtrack", "Receive track metadata",
                        Assertion.Exists("$.context", "Has context"),
                        Assertion.Equals("$.context", "nowplayingtrack", "Context is nowplayingtrack"))
                }
            },

            // 4.5-004: Position with Duration
            new TestCase
            {
                Id = "V45-004",
                Name = "Position with Duration",
                Description = "Verify position response includes total duration",
                Category = "nowplaying",
                Priority = 20,
                Tags = new List<string> { "nowplaying", "v4.5", "position" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("nowplayingposition", description: "Request position"),
                    TestStep.Receive("nowplayingposition", "Receive position with duration",
                        Assertion.Exists("$.data", "Has data"))
                }
            },

            // 4.5-005: Queue with Extended Info
            new TestCase
            {
                Id = "V45-005",
                Name = "Queue Extended Info",
                Description = "Verify queue includes extended track information",
                Category = "nowplaying",
                Priority = 20,
                Tags = new List<string> { "nowplaying", "v4.5", "queue" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("nowplayinglist", description: "Request queue"),
                    TestStep.Receive("nowplayinglist", "Receive queue",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // 4.5-006: Library Pagination
            new TestCase
            {
                Id = "V45-006",
                Name = "Library Pagination",
                Description = "Verify library supports offset/limit pagination",
                Category = "library",
                Priority = 30,
                Tags = new List<string> { "library", "v4.5", "pagination" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("browseartists", new { offset = 0, limit = 10 }, "Request artists with pagination"),
                    TestStep.Receive("browseartists", "Receive paginated artists",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // 4.5-007: Search Artist
            new TestCase
            {
                Id = "V45-007",
                Name = "Search Artist",
                Description = "Search for artists in library",
                Category = "library",
                Priority = 30,
                Tags = new List<string> { "library", "v4.5", "search" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("librarysearchartist", new { query = "a" }, "Search for artist"),
                    TestStep.Receive("librarysearchartist", "Receive search results",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // 4.5-008: Playlist Details
            new TestCase
            {
                Id = "V45-008",
                Name = "Playlist Details",
                Description = "Verify playlist info includes track count",
                Category = "playlist",
                Priority = 40,
                Tags = new List<string> { "playlist", "v4.5" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("playlistlist", description: "Request playlists"),
                    TestStep.Receive("playlistlist", "Receive playlist list",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // 4.5-009: Output with Active State
            new TestCase
            {
                Id = "V45-009",
                Name = "Output Active State",
                Description = "Verify output list includes active device indicator",
                Category = "output",
                Priority = 50,
                Tags = new List<string> { "output", "v4.5" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("playeroutput", description: "Request outputs"),
                    TestStep.Receive("playeroutput", "Receive outputs with active state",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // 4.5-010: Multiple Sequential Requests
            new TestCase
            {
                Id = "V45-010",
                Name = "Sequential Requests",
                Description = "Verify server handles rapid sequential requests",
                Category = "protocol",
                Priority = 60,
                Tags = new List<string> { "protocol", "v4.5", "stress" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("ping", description: "First ping"),
                    TestStep.Receive("pong", "First pong"),
                    TestStep.Send("ping", description: "Second ping"),
                    TestStep.Receive("pong", "Second pong"),
                    TestStep.Send("ping", description: "Third ping"),
                    TestStep.Receive("pong", "Third pong"),
                    TestStep.Send("playerstatus", description: "Player status after pings"),
                    TestStep.Receive("playerstatus", "Receive player status",
                        Assertion.Exists("$.data", "Has data"))
                }
            },

            // ============================================
            // ARiA (Arbitrary Remote Initiated Actions) Tests
            // ============================================

            // V45-011: ARiA Action Test (Delay - Safe)
            // Tests ARiA command framework with a non-destructive delay command
            // Note: ARiA must be enabled in plugin settings for this to work
            new TestCase
            {
                Id = "V45-011",
                Name = "ARiA Delay Action",
                Description = "Test ARiA with safe delay command (100ms)",
                Category = "aria",
                Priority = 50,
                Tags = new List<string> { "aria", "v4.5", "action" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("playerinitiatoraction", "delay(100)", "Execute delay command"),
                    TestStep.Receive("playerinitiatoraction", "Receive action response",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // V45-012: Visualizer List Query
            // Tests experimental visualizer feature
            new TestCase
            {
                Id = "V45-012",
                Name = "Visualizer List",
                Description = "Query available visualizers",
                Category = "experimental",
                Priority = 50,
                Tags = new List<string> { "experimental", "v4.5", "visualizer" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("playervisualizerlist", description: "Request visualizer list"),
                    TestStep.Receive("playervisualizerlist", "Receive visualizer list",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // V45-013: Library Subscribe (Two-Way Sync)
            new TestCase
            {
                Id = "V45-013",
                Name = "Library Subscribe",
                Description = "Subscribe to library change events",
                Category = "library",
                Priority = 40,
                Tags = new List<string> { "library", "v4.5", "sync" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("librarysubscribe", description: "Subscribe to library events"),
                    TestStep.Receive("librarysubscribe", "Receive subscription confirmation",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // V45-014: Library Unsubscribe
            new TestCase
            {
                Id = "V45-014",
                Name = "Library Unsubscribe",
                Description = "Unsubscribe from library change events",
                Category = "library",
                Priority = 41,
                Tags = new List<string> { "library", "v4.5", "sync" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("libraryunsubscribe", description: "Unsubscribe from library events"),
                    TestStep.Receive("libraryunsubscribe", "Receive unsubscription confirmation",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // V45-015: Plugin Storage Path
            new TestCase
            {
                Id = "V45-015",
                Name = "Plugin Storage Path",
                Description = "Query plugin storage path",
                Category = "aria",
                Priority = 50,
                Tags = new List<string> { "aria", "v4.5", "storage" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("pluginstoragepath", description: "Request storage path"),
                    TestStep.Receive("pluginstoragepath", "Receive storage path",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // V45-016: ARiA Hotkey Configuration
            new TestCase
            {
                Id = "V45-016",
                Name = "ARiA Hotkey Configuration",
                Description = "Query configured ARiA hotkeys",
                Category = "aria",
                Priority = 50,
                Tags = new List<string> { "aria", "v4.5", "hotkeys" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("configureariahotkeys", description: "Request ARiA hotkey config"),
                    TestStep.Receive("configureariahotkeys", "Receive hotkey configuration",
                        Assertion.Exists("$.context", "Has context"))
                }
            },

            // V45-017: ARiA Invalid Command (Error Handling)
            new TestCase
            {
                Id = "V45-017",
                Name = "ARiA Invalid Command",
                Description = "Verify ARiA handles invalid commands gracefully",
                Category = "aria",
                Priority = 80,
                Tags = new List<string> { "aria", "v4.5", "error" },
                Steps = new List<TestStep>
                {
                    TestStep.Send("playerinitiatoraction", "invalidcommand()", "Send invalid ARiA command"),
                    TestStep.Receive("playerinitiatoraction", "Receive error response",
                        Assertion.Exists("$.context", "Has context"))
                }
            }

            // ============================================
            // MANUAL ONLY - Destructive ARiA Commands
            // ============================================
            // These are NOT included in automated tests:
            // - restart(musicbee) - Restarts MusicBee
            // - restart(system) - Restarts the computer
            // - restart(shutdown) - Shuts down the computer
            // - sendkeys(...) - Sends keystrokes to the system
            // - mouseclick(...) - Clicks at screen coordinates
            // - run(...) - Executes arbitrary programs
            //
            // Test these manually using the REPL:
            //   mbrcpval repl -H localhost -p 3000
            //   > .send playerinitiatoraction restart(musicbee)
        };
    }

    /// <summary>
    /// Gets test count summary for a protocol version.
    /// </summary>
    public static (int core, int v40, int v45, int total) GetTestCounts(string protocolVersion)
    {
        int core = CreateCoreTests().Count;
        int v40 = CreateProtocol40Tests().Count;
        int v45 = protocolVersion == "4.5" ? CreateProtocol45Tests().Count : 0;

        return (core, v40, v45, core + v40 + v45);
    }
}
