namespace Mbrcpval.Testing;

/// <summary>
/// Represents an individual step within a test case.
/// </summary>
public class TestStep
{
    /// <summary>
    /// Gets or sets the action to perform in this step.
    /// </summary>
    public StepAction Action { get; set; }

    /// <summary>
    /// Gets or sets the message context for Send actions.
    /// This corresponds to the MBRC protocol context field (e.g., "player", "nowplayinglist").
    /// </summary>
    public string? Context { get; set; }

    /// <summary>
    /// Gets or sets the message data for Send actions.
    /// Can be a string, number, boolean, or complex object depending on the message type.
    /// </summary>
    public object? Data { get; set; }

    /// <summary>
    /// Gets or sets the expected context to match when receiving a message.
    /// Used with Receive action to wait for a specific message type.
    /// </summary>
    public string? ExpectedContext { get; set; }

    /// <summary>
    /// Gets or sets the assertions to evaluate for Receive/Assert actions.
    /// </summary>
    public List<Assertion> Assertions { get; set; } = new();

    /// <summary>
    /// Gets or sets the step-specific timeout.
    /// If null, the test case default timeout is used.
    /// </summary>
    public TimeSpan? Timeout { get; set; }

    /// <summary>
    /// Gets or sets a human-readable description of what this step does.
    /// </summary>
    public string? Description { get; set; }

    /// <summary>
    /// Gets or sets the duration to wait for Wait actions.
    /// </summary>
    public TimeSpan? WaitDuration { get; set; }

    /// <summary>
    /// Gets or sets the variable name for SetVariable actions.
    /// </summary>
    public string? VariableName { get; set; }

    /// <summary>
    /// Gets or sets the variable value or JSONPath expression for SetVariable actions.
    /// </summary>
    public string? VariableValue { get; set; }

    /// <summary>
    /// Gets or sets the log message for Log actions.
    /// </summary>
    public string? LogMessage { get; set; }

    /// <summary>
    /// Gets or sets the log level for Log actions.
    /// </summary>
    public string LogLevel { get; set; } = "Info";

    /// <summary>
    /// Creates a Connect step.
    /// </summary>
    public static TestStep Connect(string? description = null) =>
        new() { Action = StepAction.Connect, Description = description ?? "Connect to server" };

    /// <summary>
    /// Creates a Disconnect step.
    /// </summary>
    public static TestStep Disconnect(string? description = null) =>
        new() { Action = StepAction.Disconnect, Description = description ?? "Disconnect from server" };

    /// <summary>
    /// Creates a Send step.
    /// </summary>
    public static TestStep Send(string context, object? data = null, string? description = null) =>
        new() { Action = StepAction.Send, Context = context, Data = data, Description = description };

    /// <summary>
    /// Creates a Receive step that waits for a specific context.
    /// </summary>
    public static TestStep Receive(string expectedContext, string? description = null, params Assertion[] assertions) =>
        new()
        {
            Action = StepAction.Receive,
            ExpectedContext = expectedContext,
            Description = description,
            Assertions = assertions.ToList()
        };

    /// <summary>
    /// Creates a ReceiveAny step.
    /// </summary>
    public static TestStep ReceiveAny(string? description = null, params Assertion[] assertions) =>
        new()
        {
            Action = StepAction.ReceiveAny,
            Description = description,
            Assertions = assertions.ToList()
        };

    /// <summary>
    /// Creates a Wait step.
    /// </summary>
    public static TestStep Wait(TimeSpan duration, string? description = null) =>
        new() { Action = StepAction.Wait, WaitDuration = duration, Description = description };

    /// <summary>
    /// Creates a Wait step with milliseconds.
    /// </summary>
    public static TestStep WaitMs(int milliseconds, string? description = null) =>
        Wait(TimeSpan.FromMilliseconds(milliseconds), description);

    /// <summary>
    /// Creates an Assert step.
    /// </summary>
    public static TestStep Assert(string? description = null, params Assertion[] assertions) =>
        new() { Action = StepAction.Assert, Description = description, Assertions = assertions.ToList() };

    /// <summary>
    /// Creates a SetVariable step.
    /// </summary>
    public static TestStep SetVariable(string name, string value, string? description = null) =>
        new() { Action = StepAction.SetVariable, VariableName = name, VariableValue = value, Description = description };

    /// <summary>
    /// Creates a Log step.
    /// </summary>
    public static TestStep Log(string message, string level = "Info", string? description = null) =>
        new() { Action = StepAction.Log, LogMessage = message, LogLevel = level, Description = description };

    public override string ToString()
    {
        if (!string.IsNullOrEmpty(Description))
            return Description;

        return Action switch
        {
            StepAction.Connect => "Connect to server",
            StepAction.Disconnect => "Disconnect from server",
            StepAction.Send => $"Send {Context}",
            StepAction.Receive => $"Receive {ExpectedContext}",
            StepAction.ReceiveAny => "Receive any message",
            StepAction.Wait => $"Wait {WaitDuration?.TotalMilliseconds ?? 0}ms",
            StepAction.Assert => $"Assert ({Assertions.Count} assertions)",
            StepAction.SetVariable => $"Set {VariableName}",
            StepAction.Log => $"Log: {LogMessage}",
            _ => Action.ToString()
        };
    }
}
