using System.Collections.Concurrent;
using System.Diagnostics;
using System.Net.Sockets;
using System.Text;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;

namespace Mbrcpval.Testing;

/// <summary>
/// Executes test cases against an MBRC server.
/// </summary>
public class TestRunner
{
    private readonly AssertionEvaluator _evaluator = new();
    private readonly ConcurrentDictionary<string, object> _variables = new();

    /// <summary>
    /// Gets or sets the default timeout for operations.
    /// </summary>
    public TimeSpan DefaultTimeout { get; set; } = TimeSpan.FromSeconds(30);

    /// <summary>
    /// Gets or sets whether to stop on first failure.
    /// </summary>
    public bool FailFast { get; set; }

    /// <summary>
    /// Gets or sets the maximum degree of parallelism for parallel test execution.
    /// WARNING: Parallel execution with a shared client is not supported for MBRC protocol
    /// as it is stateful. Use MaxParallelism > 1 only with independent client instances.
    /// </summary>
    public int MaxParallelism { get; set; } = 1;

    /// <summary>
    /// Raised when a test starts.
    /// </summary>
    public event EventHandler<TestStartedEventArgs>? TestStarted;

    /// <summary>
    /// Raised when a test completes.
    /// </summary>
    public event EventHandler<TestCompletedEventArgs>? TestCompleted;

    /// <summary>
    /// Raised when a test step is executed.
    /// </summary>
    public event EventHandler<StepExecutedEventArgs>? StepExecuted;

    /// <summary>
    /// Raised when a message is sent or received.
    /// </summary>
    public event EventHandler<MessageEventArgs>? MessageEvent;

    /// <summary>
    /// Runs a single test case.
    /// </summary>
    /// <param name="test">The test case to run.</param>
    /// <param name="client">The MBRC socket client to use.</param>
    /// <param name="cancellationToken">Cancellation token.</param>
    /// <returns>The test result.</returns>
    public async Task<TestResult> RunTestAsync(TestCase test, IMbrcSocketClient client,
        CancellationToken cancellationToken = default)
    {
        var stopwatch = Stopwatch.StartNew();
        var stepResults = new List<StepResult>();
        var capturedMessages = new List<CapturedMessage>();

        OnTestStarted(test);

        // Check if test should be skipped
        if (test.Skip)
        {
            var skipResult = TestResult.Skip(test, test.SkipReason ?? "Test marked as skip");
            OnTestCompleted(test, skipResult);
            return skipResult;
        }

        try
        {
            // Execute test steps
            for (var i = 0; i < test.Steps.Count; i++)
            {
                cancellationToken.ThrowIfCancellationRequested();

                var step = test.Steps[i];
                var stepResult = await ExecuteStepAsync(step, i, client, test.Timeout, capturedMessages, cancellationToken);
                stepResults.Add(stepResult);

                OnStepExecuted(test, step, i, stepResult);

                if (!stepResult.Passed)
                {
                    stopwatch.Stop();
                    var failResult = TestResult.Fail(
                        test, stopwatch.Elapsed, i,
                        stepResult.ErrorMessage ?? "Step failed",
                        stepResult.AssertionResults.FirstOrDefault(a => !a.Passed)?.ActualValue,
                        stepResult.AssertionResults.FirstOrDefault(a => !a.Passed)?.ExpectedValue);
                    failResult.StepResults = stepResults;
                    failResult.CapturedMessages = capturedMessages;
                    failResult.AssertionResults = stepResult.AssertionResults;

                    // Run cleanup steps even on failure
                    await RunCleanupAsync(test, client, cancellationToken);

                    OnTestCompleted(test, failResult);
                    return failResult;
                }
            }

            stopwatch.Stop();

            // Run cleanup steps
            await RunCleanupAsync(test, client, cancellationToken);

            var result = TestResult.Pass(test, stopwatch.Elapsed, stepResults);
            result.CapturedMessages = capturedMessages;

            OnTestCompleted(test, result);
            return result;
        }
        catch (OperationCanceledException)
        {
            stopwatch.Stop();
            await RunCleanupAsync(test, client, CancellationToken.None);

            var result = TestResult.Skip(test, "Test cancelled");
            OnTestCompleted(test, result);
            return result;
        }
        catch (Exception ex)
        {
            stopwatch.Stop();
            await RunCleanupAsync(test, client, CancellationToken.None);

            var result = TestResult.Error(test, stopwatch.Elapsed, ex.Message, ex);
            result.StepResults = stepResults;
            result.CapturedMessages = capturedMessages;

            OnTestCompleted(test, result);
            return result;
        }
    }

    /// <summary>
    /// Runs a suite of test cases.
    /// </summary>
    /// <param name="tests">The test cases to run.</param>
    /// <param name="client">The MBRC socket client to use.</param>
    /// <param name="cancellationToken">Cancellation token.</param>
    /// <returns>List of test results.</returns>
    public async Task<List<TestResult>> RunSuiteAsync(IEnumerable<TestCase> tests, IMbrcSocketClient client,
        CancellationToken cancellationToken = default)
    {
        var testList = tests.OrderBy(t => t.Priority).ToList();
        var results = new List<TestResult>();

        if (MaxParallelism <= 1)
        {
            // Sequential execution
            foreach (var test in testList)
            {
                cancellationToken.ThrowIfCancellationRequested();

                var result = await RunTestAsync(test, client, cancellationToken);
                results.Add(result);

                if (FailFast && result.Failed)
                    break;
            }
        }
        else
        {
            // Parallel execution
            var semaphore = new SemaphoreSlim(MaxParallelism);
            var tasks = new List<Task<TestResult>>();

            foreach (var test in testList)
            {
                await semaphore.WaitAsync(cancellationToken);

                if (FailFast && results.Any(r => r.Failed))
                {
                    semaphore.Release();
                    break;
                }

                tasks.Add(Task.Run(async () =>
                {
                    try
                    {
                        return await RunTestAsync(test, client, cancellationToken);
                    }
                    finally
                    {
                        semaphore.Release();
                    }
                }, cancellationToken));
            }

            var completedResults = await Task.WhenAll(tasks);
            results.AddRange(completedResults);
        }

        return results;
    }

    /// <summary>
    /// Connects to a server and runs all tests.
    /// </summary>
    /// <param name="host">Server hostname.</param>
    /// <param name="port">Server port.</param>
    /// <param name="tests">Test cases to run.</param>
    /// <param name="cancellationToken">Cancellation token.</param>
    /// <returns>List of test results.</returns>
    public async Task<List<TestResult>> RunAllAsync(string host, int port, IEnumerable<TestCase> tests,
        CancellationToken cancellationToken = default)
    {
        return await RunAllAsync(host, port, tests, null, null, cancellationToken);
    }

    /// <summary>
    /// Connects to a server and runs all tests with optional logging callbacks.
    /// </summary>
    /// <param name="host">Server hostname.</param>
    /// <param name="port">Server port.</param>
    /// <param name="tests">Test cases to run.</param>
    /// <param name="onSend">Callback when a message is sent.</param>
    /// <param name="onReceive">Callback when a message is received.</param>
    /// <param name="cancellationToken">Cancellation token.</param>
    /// <returns>List of test results.</returns>
    public async Task<List<TestResult>> RunAllAsync(string host, int port, IEnumerable<TestCase> tests,
        Action<string>? onSend, Action<string>? onReceive,
        CancellationToken cancellationToken = default)
    {
        using var client = new TcpMbrcSocketClient();

        // Wire up logging callbacks
        if (onSend != null)
            client.RawMessageSent += onSend;
        if (onReceive != null)
            client.RawMessageReceived += onReceive;

        await client.ConnectAsync(host, port, cancellationToken);

        try
        {
            // CRITICAL: Old MBRC plugin requires strict handshake sequence
            // Packet 0 MUST be 'player' context
            // Packet 1 MUST be 'protocol' context
            // Otherwise the server disconnects the client!
            await PerformHandshakeAsync(client, cancellationToken);

            return await RunSuiteAsync(tests, client, cancellationToken);
        }
        finally
        {
            await client.DisconnectAsync();
        }
    }

    /// <summary>
    /// Performs the required MBRC handshake sequence.
    /// Old plugin requires: packet 0 = player, packet 1 = protocol
    /// </summary>
    private async Task PerformHandshakeAsync(IMbrcSocketClient client, CancellationToken cancellationToken)
    {
        using var cts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
        cts.CancelAfter(TimeSpan.FromSeconds(10));

        // Packet 0: Send player request (required as first message)
        await client.SendAsync("{\"context\":\"player\",\"data\":\"\"}", cts.Token);

        // Wait for player response
        await client.ReceiveAsync("player", cts.Token);

        // Packet 1: Send protocol request with version info (required as second message)
        // The data object contains the client's protocol version
        await client.SendAsync("{\"context\":\"protocol\",\"data\":{\"protocol_version\":5}}", cts.Token);

        // Wait for protocol response
        await client.ReceiveAsync("protocol", cts.Token);

        // Give server a moment to process handshake
        await Task.Delay(50, cancellationToken);
    }

    private async Task<StepResult> ExecuteStepAsync(TestStep step, int index, IMbrcSocketClient client,
        TimeSpan testTimeout, List<CapturedMessage> capturedMessages, CancellationToken cancellationToken)
    {
        var timeout = step.Timeout ?? testTimeout;
        var stepResult = new StepResult { Step = step, StepIndex = index };
        var stopwatch = Stopwatch.StartNew();

        try
        {
            using var cts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
            cts.CancelAfter(timeout);

            switch (step.Action)
            {
                case StepAction.Connect:
                    // Connection is handled by client creation
                    stepResult.Passed = client.IsConnected;
                    if (!stepResult.Passed)
                        stepResult.ErrorMessage = "Client is not connected";
                    break;

                case StepAction.Disconnect:
                    await client.DisconnectAsync();
                    stepResult.Passed = true;
                    break;

                case StepAction.Send:
                    var sendMessage = CreateMessage(step.Context, step.Data);
                    await client.SendAsync(sendMessage, cts.Token);

                    var sentCapture = new CapturedMessage
                    {
                        Context = step.Context ?? string.Empty,
                        Data = step.Data,
                        RawJson = sendMessage,
                        Direction = CapturedMessageDirection.Sent
                    };
                    capturedMessages.Add(sentCapture);
                    stepResult.SentMessage = sentCapture;
                    stepResult.Passed = true;

                    OnMessageEvent(sentCapture);
                    break;

                case StepAction.Receive:
                    var received = await client.ReceiveAsync(step.ExpectedContext, cts.Token);

                    if (received == null)
                    {
                        stepResult.Passed = false;
                        stepResult.ErrorMessage = $"Timeout waiting for message with context '{step.ExpectedContext}'";
                    }
                    else
                    {
                        var receivedCapture = new CapturedMessage
                        {
                            Context = received.Context,
                            Data = received.Data,
                            RawJson = received.RawJson,
                            Direction = CapturedMessageDirection.Received
                        };
                        capturedMessages.Add(receivedCapture);
                        stepResult.ReceivedMessage = receivedCapture;

                        OnMessageEvent(receivedCapture);

                        // Evaluate assertions
                        if (step.Assertions.Count > 0)
                        {
                            stepResult.AssertionResults = _evaluator.EvaluateAll(step.Assertions, receivedCapture);
                            stepResult.Passed = stepResult.AssertionResults.All(r => r.Passed);
                            stepResult.ErrorMessage = stepResult.AssertionResults
                                .FirstOrDefault(r => !r.Passed)?.ErrorMessage;
                        }
                        else
                        {
                            stepResult.Passed = true;
                        }
                    }
                    break;

                case StepAction.ReceiveAny:
                    var anyReceived = await client.ReceiveAnyAsync(cts.Token);

                    if (anyReceived == null)
                    {
                        stepResult.Passed = false;
                        stepResult.ErrorMessage = "Timeout waiting for any message";
                    }
                    else
                    {
                        var anyCapture = new CapturedMessage
                        {
                            Context = anyReceived.Context,
                            Data = anyReceived.Data,
                            RawJson = anyReceived.RawJson,
                            Direction = CapturedMessageDirection.Received
                        };
                        capturedMessages.Add(anyCapture);
                        stepResult.ReceivedMessage = anyCapture;

                        OnMessageEvent(anyCapture);

                        // Evaluate assertions
                        if (step.Assertions.Count > 0)
                        {
                            stepResult.AssertionResults = _evaluator.EvaluateAll(step.Assertions, anyCapture);
                            stepResult.Passed = stepResult.AssertionResults.All(r => r.Passed);
                            stepResult.ErrorMessage = stepResult.AssertionResults
                                .FirstOrDefault(r => !r.Passed)?.ErrorMessage;
                        }
                        else
                        {
                            stepResult.Passed = true;
                        }
                    }
                    break;

                case StepAction.Wait:
                    var waitDuration = step.WaitDuration ?? TimeSpan.FromSeconds(1);
                    await Task.Delay(waitDuration, cts.Token);
                    stepResult.Passed = true;
                    break;

                case StepAction.Assert:
                    // Assert against the last received message
                    var lastReceived = capturedMessages
                        .LastOrDefault(m => m.Direction == CapturedMessageDirection.Received);

                    if (lastReceived == null)
                    {
                        stepResult.Passed = false;
                        stepResult.ErrorMessage = "No received message to assert against";
                    }
                    else
                    {
                        stepResult.AssertionResults = _evaluator.EvaluateAll(step.Assertions, lastReceived);
                        stepResult.Passed = stepResult.AssertionResults.All(r => r.Passed);
                        stepResult.ErrorMessage = stepResult.AssertionResults
                            .FirstOrDefault(r => !r.Passed)?.ErrorMessage;
                    }
                    break;

                case StepAction.SetVariable:
                    if (!string.IsNullOrEmpty(step.VariableName))
                    {
                        _variables[step.VariableName] = step.VariableValue ?? string.Empty;
                    }
                    stepResult.Passed = true;
                    break;

                case StepAction.Log:
                    // Log step always passes - just for debugging
                    stepResult.Passed = true;
                    break;

                default:
                    stepResult.Passed = false;
                    stepResult.ErrorMessage = $"Unknown step action: {step.Action}";
                    break;
            }
        }
        catch (OperationCanceledException) when (!cancellationToken.IsCancellationRequested)
        {
            stepResult.Passed = false;
            stepResult.ErrorMessage = $"Step timed out after {timeout.TotalSeconds:F1}s";
        }
        catch (Exception ex)
        {
            stepResult.Passed = false;
            stepResult.ErrorMessage = $"Step error: {ex.Message}";
        }

        stopwatch.Stop();
        stepResult.Duration = stopwatch.Elapsed;
        return stepResult;
    }

    private async Task RunCleanupAsync(TestCase test, IMbrcSocketClient client, CancellationToken cancellationToken)
    {
        foreach (var cleanupStep in test.Cleanup)
        {
            try
            {
                await ExecuteStepAsync(cleanupStep, -1, client, TimeSpan.FromSeconds(5),
                    new List<CapturedMessage>(), cancellationToken);
            }
            catch
            {
                // Ignore cleanup errors
            }
        }
    }

    private string CreateMessage(string? context, object? data)
    {
        var message = new { context, data };
        return JsonConvert.SerializeObject(message);
    }

    private void OnTestStarted(TestCase test)
    {
        SafeInvokeEvent(() => TestStarted?.Invoke(this, new TestStartedEventArgs(test)));
    }

    private void OnTestCompleted(TestCase test, TestResult result)
    {
        SafeInvokeEvent(() => TestCompleted?.Invoke(this, new TestCompletedEventArgs(test, result)));
    }

    private void OnStepExecuted(TestCase test, TestStep step, int index, StepResult result)
    {
        SafeInvokeEvent(() => StepExecuted?.Invoke(this, new StepExecutedEventArgs(test, step, index, result)));
    }

    private void OnMessageEvent(CapturedMessage message)
    {
        SafeInvokeEvent(() => MessageEvent?.Invoke(this, new MessageEventArgs(message)));
    }

    /// <summary>
    /// Safely invokes an event handler, catching and logging any exceptions.
    /// Event handler exceptions should never crash the test runner.
    /// </summary>
    private static void SafeInvokeEvent(Action eventInvocation)
    {
        try
        {
            eventInvocation();
        }
        catch (Exception ex)
        {
            // Log but don't rethrow - event handlers should never crash test execution
            Console.Error.WriteLine($"[TestRunner] Event handler exception (ignored): {ex.Message}");
        }
    }
}

#region Event Args

/// <summary>
/// Event args for test started event.
/// </summary>
public class TestStartedEventArgs : EventArgs
{
    public TestCase TestCase { get; }

    public TestStartedEventArgs(TestCase testCase)
    {
        TestCase = testCase;
    }
}

/// <summary>
/// Event args for test completed event.
/// </summary>
public class TestCompletedEventArgs : EventArgs
{
    public TestCase TestCase { get; }
    public TestResult Result { get; }

    public TestCompletedEventArgs(TestCase testCase, TestResult result)
    {
        TestCase = testCase;
        Result = result;
    }
}

/// <summary>
/// Event args for step executed event.
/// </summary>
public class StepExecutedEventArgs : EventArgs
{
    public TestCase TestCase { get; }
    public TestStep Step { get; }
    public int StepIndex { get; }
    public StepResult Result { get; }

    public StepExecutedEventArgs(TestCase testCase, TestStep step, int stepIndex, StepResult result)
    {
        TestCase = testCase;
        Step = step;
        StepIndex = stepIndex;
        Result = result;
    }
}

/// <summary>
/// Event args for message events.
/// </summary>
public class MessageEventArgs : EventArgs
{
    public CapturedMessage Message { get; }

    public MessageEventArgs(CapturedMessage message)
    {
        Message = message;
    }
}

#endregion

#region Socket Client Interface

/// <summary>
/// Interface for MBRC socket client implementations.
/// </summary>
public interface IMbrcSocketClient : IDisposable, IAsyncDisposable
{
    /// <summary>
    /// Gets whether the client is connected.
    /// </summary>
    bool IsConnected { get; }

    /// <summary>
    /// Connects to the MBRC server.
    /// </summary>
    Task ConnectAsync(string host, int port, CancellationToken cancellationToken = default);

    /// <summary>
    /// Disconnects from the MBRC server.
    /// </summary>
    Task DisconnectAsync();

    /// <summary>
    /// Sends a message to the server.
    /// </summary>
    Task SendAsync(string message, CancellationToken cancellationToken = default);

    /// <summary>
    /// Receives a message with a specific context.
    /// </summary>
    Task<TestMessage?> ReceiveAsync(string? expectedContext, CancellationToken cancellationToken = default);

    /// <summary>
    /// Receives any message from the server.
    /// </summary>
    Task<TestMessage?> ReceiveAnyAsync(CancellationToken cancellationToken = default);
}

/// <summary>
/// Lightweight message wrapper for test framework (avoids coupling to Core.MbrcMessage).
/// </summary>
public class TestMessage
{
    public string Context { get; set; } = string.Empty;
    public object? Data { get; set; }
    public string? RawJson { get; set; }

    public static TestMessage? Parse(string json)
    {
        try
        {
            var token = JObject.Parse(json);
            return new TestMessage
            {
                Context = token["context"]?.Value<string>() ?? string.Empty,
                Data = token["data"],
                RawJson = json
            };
        }
        catch
        {
            return null;
        }
    }
}

/// <summary>
/// TCP-based implementation of the MBRC socket client.
/// Thread-safe implementation with proper synchronization for concurrent access.
/// </summary>
public class TcpMbrcSocketClient : IMbrcSocketClient
{
    private const int MaxQueueSize = 1000;
    private const int MaxRetries = 3;
    private const int RetryDelayMs = 100;
    private const int ConnectionTimeoutMs = 30000;
    private const int OperationTimeoutMs = 5000;

    private TcpClient? _client;
    private NetworkStream? _stream;
    private StreamReader? _reader;
    private StreamWriter? _writer;
    private readonly List<TestMessage> _messageList = new();
    private readonly object _messageLock = new();
    private readonly object _connectionLock = new();
    private Task? _receiveTask;
    private CancellationTokenSource? _receiveCts;
    private volatile bool _disposed;  // Volatile for cross-thread visibility

    public bool IsConnected => _client?.Connected ?? false;

    public async Task ConnectAsync(string host, int port, CancellationToken cancellationToken = default)
    {
        ObjectDisposedException.ThrowIf(_disposed, this);

        _client = new TcpClient();
        await _client.ConnectAsync(host, port, cancellationToken);

        _stream = _client.GetStream();
        // Use UTF8 without BOM for protocol compliance
        _reader = new StreamReader(_stream, new UTF8Encoding(false), leaveOpen: true);
        _writer = new StreamWriter(_stream, new UTF8Encoding(false), leaveOpen: true) { AutoFlush = true };

        // Start background message receiver
        _receiveCts = new CancellationTokenSource();
        _receiveTask = ReceiveLoopAsync(_receiveCts.Token);
    }

    public async Task DisconnectAsync()
    {
        if (_receiveCts != null)
        {
            await _receiveCts.CancelAsync();
        }

        if (_receiveTask != null)
        {
            try
            {
                await _receiveTask.WaitAsync(TimeSpan.FromSeconds(2));
            }
            catch (OperationCanceledException)
            {
                // Expected
            }
            catch (TimeoutException)
            {
                // Timeout waiting for receive task
            }
        }

        _writer?.Dispose();
        _reader?.Dispose();
        _stream?.Dispose();
        _client?.Dispose();

        _writer = null;
        _reader = null;
        _stream = null;
        _client = null;

        lock (_messageLock)
        {
            _messageList.Clear();
        }
    }

    public async Task SendAsync(string message, CancellationToken cancellationToken = default)
    {
        ObjectDisposedException.ThrowIf(_disposed, this);

        if (_writer == null)
            throw new InvalidOperationException("Not connected");

        // Log raw message being sent
        RawMessageSent?.Invoke(message);

        // MBRC protocol uses CRLF-terminated JSON
        await _writer.WriteLineAsync(message.AsMemory(), cancellationToken);
    }

    public async Task<TestMessage?> ReceiveAsync(string? expectedContext, CancellationToken cancellationToken = default)
    {
        ObjectDisposedException.ThrowIf(_disposed, this);

        while (!cancellationToken.IsCancellationRequested)
        {
            lock (_messageLock)
            {
                for (int i = 0; i < _messageList.Count; i++)
                {
                    var msg = _messageList[i];
                    if (expectedContext == null ||
                        string.Equals(msg.Context, expectedContext, StringComparison.OrdinalIgnoreCase))
                    {
                        _messageList.RemoveAt(i);
                        return msg;
                    }
                }
            }

            // Wait a bit and try again
            await Task.Delay(10, cancellationToken);
        }

        return null;
    }

    public async Task<TestMessage?> ReceiveAnyAsync(CancellationToken cancellationToken = default)
    {
        ObjectDisposedException.ThrowIf(_disposed, this);

        while (!cancellationToken.IsCancellationRequested)
        {
            lock (_messageLock)
            {
                if (_messageList.Count > 0)
                {
                    var message = _messageList[0];
                    _messageList.RemoveAt(0);
                    return message;
                }
            }

            await Task.Delay(10, cancellationToken);
        }

        return null;
    }

    /// <summary>
    /// Raised when a raw message is received from the server.
    /// </summary>
    public event Action<string>? RawMessageReceived;

    /// <summary>
    /// Raised when a raw message is sent to the server.
    /// </summary>
    public event Action<string>? RawMessageSent;

    private async Task ReceiveLoopAsync(CancellationToken cancellationToken)
    {
        while (!cancellationToken.IsCancellationRequested && _reader != null)
        {
            try
            {
                var line = await _reader.ReadLineAsync(cancellationToken);
                if (line == null)
                    break; // Connection closed - exit immediately

                // Log raw message
                RawMessageReceived?.Invoke(line);

                var message = TestMessage.Parse(line);
                if (message != null)
                {
                    lock (_messageLock)
                    {
                        // Enforce queue size limit to prevent memory exhaustion
                        if (_messageList.Count >= MaxQueueSize)
                        {
                            _messageList.RemoveAt(0); // Remove oldest
                        }
                        _messageList.Add(message);
                    }
                }
            }
            catch (OperationCanceledException)
            {
                break;
            }
            catch (IOException)
            {
                break;
            }
            catch (ObjectDisposedException)
            {
                break; // Reader was disposed - exit gracefully
            }
        }
    }

    public void Dispose()
    {
        if (_disposed) return;
        _disposed = true;

        // Synchronous cleanup - avoid deadlock by not calling async methods
        _receiveCts?.Cancel();

        // Give receive task a moment to complete, but don't block indefinitely
        if (_receiveTask != null)
        {
            try
            {
                _receiveTask.Wait(TimeSpan.FromMilliseconds(500));
            }
            catch (AggregateException)
            {
                // Ignore cancellation/timeout exceptions
            }
        }

        _receiveCts?.Dispose();
        _writer?.Dispose();
        _reader?.Dispose();
        _stream?.Dispose();
        _client?.Dispose();

        lock (_messageLock)
        {
            _messageList.Clear();
        }

        GC.SuppressFinalize(this);
    }

    public async ValueTask DisposeAsync()
    {
        if (_disposed) return;
        _disposed = true;

        await DisconnectAsync();
        _receiveCts?.Dispose();
        GC.SuppressFinalize(this);
    }
}

#endregion
