namespace Mbrcpval.Testing;

/// <summary>
/// Represents the complete result of executing a test case.
/// </summary>
public class TestResult
{
    /// <summary>
    /// Gets or sets the test case that was executed.
    /// </summary>
    public TestCase TestCase { get; set; } = new();

    /// <summary>
    /// Gets or sets the final status of the test.
    /// </summary>
    public TestStatus Status { get; set; }

    /// <summary>
    /// Gets or sets the total duration of the test execution.
    /// </summary>
    public TimeSpan Duration { get; set; }

    /// <summary>
    /// Gets or sets the index of the step that failed (0-based), if applicable.
    /// </summary>
    public int? FailedStepIndex { get; set; }

    /// <summary>
    /// Gets or sets the step that failed, if applicable.
    /// </summary>
    public TestStep? FailedStep { get; set; }

    /// <summary>
    /// Gets or sets the error message if the test failed or errored.
    /// </summary>
    public string? ErrorMessage { get; set; }

    /// <summary>
    /// Gets or sets the actual value that caused an assertion failure.
    /// </summary>
    public object? ActualValue { get; set; }

    /// <summary>
    /// Gets or sets the expected value from the failed assertion.
    /// </summary>
    public object? ExpectedValue { get; set; }

    /// <summary>
    /// Gets or sets all messages captured during test execution.
    /// </summary>
    public List<CapturedMessage> CapturedMessages { get; set; } = new();

    /// <summary>
    /// Gets or sets the results of individual step executions.
    /// </summary>
    public List<StepResult> StepResults { get; set; } = new();

    /// <summary>
    /// Gets or sets the assertion results for the failed step.
    /// </summary>
    public List<AssertionResult> AssertionResults { get; set; } = new();

    /// <summary>
    /// Gets or sets the timestamp when the test started.
    /// </summary>
    public DateTime StartTime { get; set; }

    /// <summary>
    /// Gets or sets the timestamp when the test ended.
    /// </summary>
    public DateTime EndTime { get; set; }

    /// <summary>
    /// Gets or sets the exception that caused an error, if any.
    /// </summary>
    public Exception? Exception { get; set; }

    /// <summary>
    /// Gets or sets the reason for skipping, if the test was skipped.
    /// </summary>
    public string? SkipReason { get; set; }

    /// <summary>
    /// Gets whether this test passed.
    /// </summary>
    public bool Passed => Status == TestStatus.Pass;

    /// <summary>
    /// Gets whether this test failed.
    /// </summary>
    public bool Failed => Status == TestStatus.Fail;

    /// <summary>
    /// Gets whether this test errored.
    /// </summary>
    public bool Errored => Status == TestStatus.Error;

    /// <summary>
    /// Gets whether this test was skipped.
    /// </summary>
    public bool Skipped => Status == TestStatus.Skip;

    /// <summary>
    /// Gets the number of steps that were executed.
    /// </summary>
    public int StepsExecuted => StepResults.Count;

    /// <summary>
    /// Gets the total number of steps in the test.
    /// </summary>
    public int TotalSteps => TestCase.Steps.Count;

    /// <summary>
    /// Creates a passing test result.
    /// </summary>
    public static TestResult Pass(TestCase testCase, TimeSpan duration, List<StepResult>? stepResults = null) =>
        new()
        {
            TestCase = testCase,
            Status = TestStatus.Pass,
            Duration = duration,
            StepResults = stepResults ?? new(),
            StartTime = DateTime.UtcNow - duration,
            EndTime = DateTime.UtcNow
        };

    /// <summary>
    /// Creates a failing test result.
    /// </summary>
    public static TestResult Fail(TestCase testCase, TimeSpan duration, int failedStepIndex,
        string errorMessage, object? actualValue = null, object? expectedValue = null) =>
        new()
        {
            TestCase = testCase,
            Status = TestStatus.Fail,
            Duration = duration,
            FailedStepIndex = failedStepIndex,
            FailedStep = failedStepIndex < testCase.Steps.Count ? testCase.Steps[failedStepIndex] : null,
            ErrorMessage = errorMessage,
            ActualValue = actualValue,
            ExpectedValue = expectedValue,
            StartTime = DateTime.UtcNow - duration,
            EndTime = DateTime.UtcNow
        };

    /// <summary>
    /// Creates an error test result.
    /// </summary>
    public static TestResult Error(TestCase testCase, TimeSpan duration, string errorMessage, Exception? exception = null) =>
        new()
        {
            TestCase = testCase,
            Status = TestStatus.Error,
            Duration = duration,
            ErrorMessage = errorMessage,
            Exception = exception,
            StartTime = DateTime.UtcNow - duration,
            EndTime = DateTime.UtcNow
        };

    /// <summary>
    /// Creates a skipped test result.
    /// </summary>
    public static TestResult Skip(TestCase testCase, string reason) =>
        new()
        {
            TestCase = testCase,
            Status = TestStatus.Skip,
            Duration = TimeSpan.Zero,
            SkipReason = reason,
            StartTime = DateTime.UtcNow,
            EndTime = DateTime.UtcNow
        };

    public override string ToString()
    {
        var status = Status switch
        {
            TestStatus.Pass => "PASS",
            TestStatus.Fail => "FAIL",
            TestStatus.Skip => "SKIP",
            TestStatus.Error => "ERROR",
            _ => "UNKNOWN"
        };

        var result = $"[{status}] {TestCase.DisplayName} ({Duration.TotalMilliseconds:F0}ms)";

        if (!string.IsNullOrEmpty(ErrorMessage))
            result += $"\n  {ErrorMessage}";

        if (!string.IsNullOrEmpty(SkipReason))
            result += $"\n  Reason: {SkipReason}";

        return result;
    }
}

/// <summary>
/// Represents the result of executing a single test step.
/// </summary>
public class StepResult
{
    /// <summary>
    /// Gets or sets the step that was executed.
    /// </summary>
    public TestStep Step { get; set; } = new();

    /// <summary>
    /// Gets or sets the index of this step (0-based).
    /// </summary>
    public int StepIndex { get; set; }

    /// <summary>
    /// Gets or sets whether this step passed.
    /// </summary>
    public bool Passed { get; set; }

    /// <summary>
    /// Gets or sets the duration of this step.
    /// </summary>
    public TimeSpan Duration { get; set; }

    /// <summary>
    /// Gets or sets the error message if this step failed.
    /// </summary>
    public string? ErrorMessage { get; set; }

    /// <summary>
    /// Gets or sets the assertion results for this step.
    /// </summary>
    public List<AssertionResult> AssertionResults { get; set; } = new();

    /// <summary>
    /// Gets or sets the message received during this step (for Receive steps).
    /// </summary>
    public CapturedMessage? ReceivedMessage { get; set; }

    /// <summary>
    /// Gets or sets the message sent during this step (for Send steps).
    /// </summary>
    public CapturedMessage? SentMessage { get; set; }
}

/// <summary>
/// Represents a captured MBRC message during test execution.
/// </summary>
public class CapturedMessage
{
    /// <summary>
    /// Gets or sets the message context.
    /// </summary>
    public string Context { get; set; } = string.Empty;

    /// <summary>
    /// Gets or sets the message data.
    /// </summary>
    public object? Data { get; set; }

    /// <summary>
    /// Gets or sets the raw JSON representation.
    /// </summary>
    public string? RawJson { get; set; }

    /// <summary>
    /// Gets or sets the timestamp when the message was captured.
    /// </summary>
    public DateTime Timestamp { get; set; } = DateTime.UtcNow;

    /// <summary>
    /// Gets or sets whether this was a sent or received message.
    /// </summary>
    public CapturedMessageDirection Direction { get; set; }
}

/// <summary>
/// Indicates whether a message was sent or received.
/// </summary>
public enum CapturedMessageDirection
{
    /// <summary>
    /// Message was sent to the server.
    /// </summary>
    Sent,

    /// <summary>
    /// Message was received from the server.
    /// </summary>
    Received
}
