namespace Mbrcpval.Testing;

/// <summary>
/// Represents a complete test case with steps, assertions, and metadata.
/// </summary>
public class TestCase
{
    /// <summary>
    /// Gets or sets the unique identifier for this test case.
    /// Convention: CATEGORY-NNN (e.g., "HSK-001", "PLY-005", "LIB-012")
    /// </summary>
    public string Id { get; set; } = string.Empty;

    /// <summary>
    /// Gets or sets the descriptive name of the test case.
    /// </summary>
    public string Name { get; set; } = string.Empty;

    /// <summary>
    /// Gets or sets a detailed description of what this test validates.
    /// </summary>
    public string Description { get; set; } = string.Empty;

    /// <summary>
    /// Gets or sets the category for grouping related tests.
    /// Examples: "handshake", "player", "library", "nowplaying", "playlist", "output"
    /// </summary>
    public string Category { get; set; } = string.Empty;

    /// <summary>
    /// Gets or sets the list of prerequisites that must be met before running this test.
    /// Examples: "connected", "playing", "has_library", "has_playlists"
    /// </summary>
    public List<string> Prerequisites { get; set; } = new();

    /// <summary>
    /// Gets or sets the list of test steps to execute.
    /// </summary>
    public List<TestStep> Steps { get; set; } = new();

    /// <summary>
    /// Gets or sets the cleanup steps to run after the test (success or failure).
    /// </summary>
    public List<TestStep> Cleanup { get; set; } = new();

    /// <summary>
    /// Gets or sets the tags for filtering and categorization.
    /// Examples: "smoke", "regression", "slow", "requires-playback"
    /// </summary>
    public List<string> Tags { get; set; } = new();

    /// <summary>
    /// Gets or sets the maximum time allowed for this test to complete.
    /// </summary>
    public TimeSpan Timeout { get; set; } = TimeSpan.FromSeconds(30);

    /// <summary>
    /// Gets or sets whether this test should be skipped.
    /// </summary>
    public bool Skip { get; set; }

    /// <summary>
    /// Gets or sets the reason for skipping this test.
    /// </summary>
    public string? SkipReason { get; set; }

    /// <summary>
    /// Gets or sets the priority of this test (lower numbers run first).
    /// </summary>
    public int Priority { get; set; } = 100;

    /// <summary>
    /// Gets or sets the source file this test was loaded from.
    /// </summary>
    public string? SourceFile { get; set; }

    /// <summary>
    /// Checks if this test has a specific tag.
    /// </summary>
    public bool HasTag(string tag) =>
        Tags.Contains(tag, StringComparer.OrdinalIgnoreCase);

    /// <summary>
    /// Checks if this test belongs to a specific category.
    /// </summary>
    public bool IsCategory(string category) =>
        string.Equals(Category, category, StringComparison.OrdinalIgnoreCase);

    /// <summary>
    /// Gets the full display name including ID.
    /// </summary>
    public string DisplayName => $"[{Id}] {Name}";

    public override string ToString() => DisplayName;
}

/// <summary>
/// Represents a collection of test cases loaded from a single YAML file.
/// </summary>
public class TestSuite
{
    /// <summary>
    /// Gets or sets the name of this test suite.
    /// </summary>
    public string Name { get; set; } = string.Empty;

    /// <summary>
    /// Gets or sets the description of this test suite.
    /// </summary>
    public string Description { get; set; } = string.Empty;

    /// <summary>
    /// Gets or sets the version of this test suite.
    /// </summary>
    public string Version { get; set; } = "1.0";

    /// <summary>
    /// Gets or sets the source file this suite was loaded from.
    /// </summary>
    public string? SourceFile { get; set; }

    /// <summary>
    /// Gets or sets the test cases in this suite.
    /// </summary>
    public List<TestCase> Tests { get; set; } = new();

    /// <summary>
    /// Gets or sets default settings for all tests in this suite.
    /// </summary>
    public TestSuiteDefaults Defaults { get; set; } = new();

    /// <summary>
    /// Gets the total number of tests in this suite.
    /// </summary>
    public int TestCount => Tests.Count;

    /// <summary>
    /// Gets tests filtered by category.
    /// </summary>
    public IEnumerable<TestCase> GetByCategory(string category) =>
        Tests.Where(t => t.IsCategory(category));

    /// <summary>
    /// Gets tests filtered by tag.
    /// </summary>
    public IEnumerable<TestCase> GetByTag(string tag) =>
        Tests.Where(t => t.HasTag(tag));
}

/// <summary>
/// Default settings that apply to all tests in a suite.
/// </summary>
public class TestSuiteDefaults
{
    /// <summary>
    /// Gets or sets the default timeout for all tests.
    /// </summary>
    public TimeSpan Timeout { get; set; } = TimeSpan.FromSeconds(30);

    /// <summary>
    /// Gets or sets default tags applied to all tests.
    /// </summary>
    public List<string> Tags { get; set; } = new();

    /// <summary>
    /// Gets or sets default prerequisites for all tests.
    /// </summary>
    public List<string> Prerequisites { get; set; } = new();
}
