using System.Text;
using System.Text.Json;
using Mbrcpval.Core;

namespace Mbrcpval.Testing;

/// <summary>
/// Protocol fuzzer for generating malformed and edge-case messages.
/// Used to test the robustness of MBRC protocol implementations.
/// </summary>
public class ProtocolFuzzer
{
    private readonly Random _random;
    private readonly FuzzerOptions _options;

    /// <summary>
    /// Creates a new protocol fuzzer.
    /// </summary>
    /// <param name="seed">Random seed for reproducibility. Use null for random seed.</param>
    /// <param name="options">Fuzzing options.</param>
    public ProtocolFuzzer(int? seed = null, FuzzerOptions? options = null)
    {
        _random = seed.HasValue ? new Random(seed.Value) : new Random();
        _options = options ?? new FuzzerOptions();
    }

    /// <summary>
    /// Generates a batch of fuzzed messages.
    /// </summary>
    /// <param name="count">Number of messages to generate.</param>
    /// <returns>Enumerable of fuzzed messages with their category.</returns>
    public IEnumerable<FuzzedMessage> GenerateFuzzedMessages(int count = 100)
    {
        var generators = new List<Func<FuzzedMessage>>
        {
            GenerateMalformedJson,
            GenerateMissingContext,
            GenerateInvalidContext,
            GenerateExtremelyLongContext,
            GenerateUnicodeContext,
            GenerateControlCharacterContext,
            GenerateNullBytesMessage,
            GenerateDeepNesting,
            GenerateMassiveData,
            GenerateBinaryData,
            GenerateValidButWeird,
            GenerateEmptyVariations,
            GenerateNumericOverflow,
            GenerateBoundaryData,
            GenerateInjectionAttempts,
        };

        for (int i = 0; i < count; i++)
        {
            var generator = generators[_random.Next(generators.Count)];
            yield return generator();
        }
    }

    /// <summary>
    /// Generates specifically malformed JSON.
    /// </summary>
    public FuzzedMessage GenerateMalformedJson()
    {
        var variations = new[]
        {
            "{",
            "}",
            "{{",
            "}}",
            "{\"context\":",
            "{\"context\": \"test\"",
            "{context: \"test\"}",
            "{'context': 'test'}",
            "{\"context\": \"test\", }",
            "{\"context\": \"test\",, \"data\": null}",
            "[{\"context\": \"test\"}]",
            "null",
            "undefined",
            "NaN",
            "Infinity",
            "{\"context\": \"test\", \"data\": undefined}",
            "{\"context\": \"test\", \"data\": NaN}",
            $"{{{new string(' ', 1000)}\"context\": \"test\"}}",
            "{\"\": \"\"}",
            "{\"\\u0000\": \"test\"}",
        };

        return new FuzzedMessage
        {
            Category = FuzzCategory.MalformedJson,
            Message = variations[_random.Next(variations.Length)],
            ExpectedBehavior = "Should reject with parse error"
        };
    }

    /// <summary>
    /// Generates messages missing the required context field.
    /// </summary>
    public FuzzedMessage GenerateMissingContext()
    {
        var variations = new[]
        {
            "{}",
            "{\"data\": \"test\"}",
            "{\"Context\": \"test\"}",  // Wrong case
            "{\"CONTEXT\": \"test\"}",  // All caps
            "{\"conte xt\": \"test\"}", // Space in key
            "{\"context \": \"test\"}", // Trailing space
            "{ \"data\": {} }",
            "{\"ctx\": \"test\"}",      // Abbreviation
        };

        return new FuzzedMessage
        {
            Category = FuzzCategory.MissingContext,
            Message = variations[_random.Next(variations.Length)],
            ExpectedBehavior = "Should reject with missing context error"
        };
    }

    /// <summary>
    /// Generates messages with invalid context values.
    /// </summary>
    public FuzzedMessage GenerateInvalidContext()
    {
        var variations = new object[]
        {
            ("null_context", "{\"context\": null}"),
            ("empty_context", "{\"context\": \"\"}"),
            ("numeric_context", "{\"context\": 123}"),
            ("boolean_context", "{\"context\": true}"),
            ("array_context", "{\"context\": [\"test\"]}"),
            ("object_context", "{\"context\": {\"name\": \"test\"}}"),
            ("whitespace_context", "{\"context\": \"   \"}"),
            ("newline_context", "{\"context\": \"test\\ninjection\"}"),
        };

        var (name, message) = ((string, string))variations[_random.Next(variations.Length)];
        return new FuzzedMessage
        {
            Category = FuzzCategory.InvalidContext,
            Message = message,
            ExpectedBehavior = $"Should reject invalid context type: {name}"
        };
    }

    /// <summary>
    /// Generates extremely long context values.
    /// </summary>
    public FuzzedMessage GenerateExtremelyLongContext()
    {
        var lengths = new[] { 256, 1024, 4096, 65536, 1024 * 1024 };
        var length = lengths[_random.Next(lengths.Length)];

        if (length > _options.MaxMessageSize)
            length = _options.MaxMessageSize / 2;

        var context = new string('A', Math.Min(length, 10000));

        return new FuzzedMessage
        {
            Category = FuzzCategory.LengthBoundary,
            Message = $"{{\"context\": \"{context}\"}}",
            ExpectedBehavior = "Should reject or truncate extremely long context"
        };
    }

    /// <summary>
    /// Generates messages with unicode edge cases.
    /// </summary>
    public FuzzedMessage GenerateUnicodeContext()
    {
        var unicodeCases = new[]
        {
            "\u0000",           // Null
            "\u001F",           // Control
            "\u007F",           // DEL
            "\u0080",           // Continuation
            "\uFEFF",           // BOM
            "\uFFFD",           // Replacement
            "\uFFFF",           // Not a character
            "\uD800",           // High surrogate (unpaired)
            "\uDFFF",           // Low surrogate (unpaired)
            "A\u0300",          // Combining character
            "\u202E",           // Right-to-left override
            "\u200B",           // Zero-width space
            "\u00A0",           // Non-breaking space
            "日本語",            // CJK
            "العربية",           // Arabic
            "🔥💀🎉",            // Emoji
            new string('界', 100), // Repeated multibyte
        };

        var unicode = unicodeCases[_random.Next(unicodeCases.Length)];

        return new FuzzedMessage
        {
            Category = FuzzCategory.UnicodeEdgeCases,
            Message = $"{{\"context\": \"test{unicode}\"}}",
            ExpectedBehavior = "Should handle unicode properly or reject cleanly"
        };
    }

    /// <summary>
    /// Generates messages with control characters.
    /// </summary>
    public FuzzedMessage GenerateControlCharacterContext()
    {
        var controlChars = Enumerable.Range(0, 32)
            .Where(i => i != 9 && i != 10 && i != 13) // Exclude tab, LF, CR
            .Select(i => (char)i)
            .ToArray();

        var sb = new StringBuilder("test");
        for (int i = 0; i < 5; i++)
        {
            sb.Append(controlChars[_random.Next(controlChars.Length)]);
        }

        return new FuzzedMessage
        {
            Category = FuzzCategory.ControlCharacters,
            Message = $"{{\"context\": \"{sb}\"}}",
            ExpectedBehavior = "Should reject or sanitize control characters"
        };
    }

    /// <summary>
    /// Generates messages with embedded null bytes.
    /// </summary>
    public FuzzedMessage GenerateNullBytesMessage()
    {
        var positions = new[]
        {
            "\0{\"context\": \"test\"}",
            "{\"context\": \"test\"}\0",
            "{\"context\": \"te\0st\"}",
            "{\"context\": \"\0\"}",
            $"{{\"\0context\": \"test\"}}",
        };

        return new FuzzedMessage
        {
            Category = FuzzCategory.NullBytes,
            Message = positions[_random.Next(positions.Length)],
            ExpectedBehavior = "Should reject or handle null bytes safely"
        };
    }

    /// <summary>
    /// Generates deeply nested JSON structures.
    /// </summary>
    public FuzzedMessage GenerateDeepNesting()
    {
        var depths = new[] { 10, 50, 100, 500, 1000 };
        var depth = depths[_random.Next(depths.Length)];

        if (depth > _options.MaxNestingDepth)
            depth = _options.MaxNestingDepth;

        var opening = string.Concat(Enumerable.Repeat("{\"data\":", depth));
        var closing = string.Concat(Enumerable.Repeat("}", depth));

        return new FuzzedMessage
        {
            Category = FuzzCategory.DeepNesting,
            Message = $"{{\"context\": \"test\", \"data\": {opening}null{closing}}}",
            ExpectedBehavior = "Should limit nesting depth"
        };
    }

    /// <summary>
    /// Generates messages with massive data payloads.
    /// </summary>
    public FuzzedMessage GenerateMassiveData()
    {
        var sizes = new[] { 1024, 10 * 1024, 100 * 1024, 1024 * 1024 };
        var size = sizes[_random.Next(sizes.Length)];

        if (size > _options.MaxMessageSize)
            size = _options.MaxMessageSize;

        var data = new string('X', size);

        return new FuzzedMessage
        {
            Category = FuzzCategory.MassivePayload,
            Message = $"{{\"context\": \"test\", \"data\": \"{data}\"}}",
            ExpectedBehavior = "Should reject or truncate massive payloads"
        };
    }

    /// <summary>
    /// Generates messages with binary/non-text data.
    /// </summary>
    public FuzzedMessage GenerateBinaryData()
    {
        var bytes = new byte[100];
        _random.NextBytes(bytes);
        var binary = Encoding.Latin1.GetString(bytes);

        return new FuzzedMessage
        {
            Category = FuzzCategory.BinaryData,
            Message = $"{{\"context\": \"test\", \"data\": \"{binary}\"}}",
            ExpectedBehavior = "Should handle or reject binary data"
        };
    }

    /// <summary>
    /// Generates syntactically valid but semantically weird messages.
    /// </summary>
    public FuzzedMessage GenerateValidButWeird()
    {
        var variations = new[]
        {
            "{\"context\": \"test\", \"context\": \"duplicate\"}",  // Duplicate key
            "{\"context\": \"test\", \"data\": -0}",                // Negative zero
            "{\"context\": \"test\", \"data\": 1e308}",             // Near max double
            "{\"context\": \"test\", \"data\": 1e-308}",            // Near min double
            "{\"context\": \"test\", \"data\": 9999999999999999999999999999}",  // Big integer
            "{\"context\": \"test\", \"extra1\": 1, \"extra2\": 2, \"extra3\": 3}",  // Extra fields
            "{\"context\":\"test\",\"data\":null,\"data\":\"override\"}", // Duplicate data
        };

        return new FuzzedMessage
        {
            Category = FuzzCategory.ValidButWeird,
            Message = variations[_random.Next(variations.Length)],
            ExpectedBehavior = "Should handle edge cases in valid JSON"
        };
    }

    /// <summary>
    /// Generates empty/minimal variations.
    /// </summary>
    public FuzzedMessage GenerateEmptyVariations()
    {
        var variations = new[]
        {
            "",
            " ",
            "\t",
            "\n",
            "\r\n",
            "  \t\n\r  ",
            "\n\n\n",
            "\0",
        };

        return new FuzzedMessage
        {
            Category = FuzzCategory.EmptyInput,
            Message = variations[_random.Next(variations.Length)],
            ExpectedBehavior = "Should reject empty/whitespace input"
        };
    }

    /// <summary>
    /// Generates numeric overflow/underflow cases.
    /// </summary>
    public FuzzedMessage GenerateNumericOverflow()
    {
        var variations = new[]
        {
            "{\"context\": \"test\", \"data\": 999999999999999999999999999999999999999999}",
            "{\"context\": \"test\", \"data\": -999999999999999999999999999999999999999999}",
            "{\"context\": \"test\", \"data\": 0.00000000000000000000000000000000000001}",
            "{\"context\": \"test\", \"data\": 1.7976931348623157e+308}",
            "{\"context\": \"test\", \"data\": -1.7976931348623157e+308}",
            "{\"context\": \"test\", \"data\": 2147483647}",   // Int32.MaxValue
            "{\"context\": \"test\", \"data\": -2147483648}",  // Int32.MinValue
            "{\"context\": \"test\", \"data\": 9223372036854775807}",  // Int64.MaxValue
        };

        return new FuzzedMessage
        {
            Category = FuzzCategory.NumericOverflow,
            Message = variations[_random.Next(variations.Length)],
            ExpectedBehavior = "Should handle numeric edge cases"
        };
    }

    /// <summary>
    /// Generates boundary condition data.
    /// </summary>
    public FuzzedMessage GenerateBoundaryData()
    {
        var variations = new[]
        {
            "{\"context\": \"A\"}",                             // Minimum valid
            "{\"context\": \"playervolume\", \"data\": 0}",     // Min volume
            "{\"context\": \"playervolume\", \"data\": 100}",   // Max volume
            "{\"context\": \"playervolume\", \"data\": -1}",    // Below min
            "{\"context\": \"playervolume\", \"data\": 101}",   // Above max
            "{\"context\": \"playerseek\", \"data\": 0}",       // Start position
            "{\"context\": \"playerseek\", \"data\": -1}",      // Invalid position
            "{\"context\": \"nowplayinglistplay\", \"data\": 0}",  // First index
            "{\"context\": \"nowplayinglistplay\", \"data\": -1}", // Invalid index
        };

        return new FuzzedMessage
        {
            Category = FuzzCategory.BoundaryConditions,
            Message = variations[_random.Next(variations.Length)],
            ExpectedBehavior = "Should validate boundary conditions"
        };
    }

    /// <summary>
    /// Generates potential injection attempts.
    /// </summary>
    public FuzzedMessage GenerateInjectionAttempts()
    {
        var variations = new[]
        {
            // Path traversal
            "{\"context\": \"../../../etc/passwd\"}",
            "{\"context\": \"..\\\\..\\\\windows\\\\system32\"}",

            // Command injection
            "{\"context\": \"; rm -rf /\"}",
            "{\"context\": \"| cat /etc/passwd\"}",
            "{\"context\": \"`whoami`\"}",
            "{\"context\": \"$(whoami)\"}",

            // SQL injection patterns (shouldn't matter but test anyway)
            "{\"context\": \"' OR '1'='1\"}",
            "{\"context\": \"'; DROP TABLE users; --\"}",

            // LDAP injection
            "{\"context\": \"*)(uid=*\"}",

            // XSS (shouldn't matter for CLI but test)
            "{\"context\": \"<script>alert('xss')</script>\"}",
            "{\"context\": \"javascript:alert('xss')\"}",

            // Format string
            "{\"context\": \"%s%s%s%s%s%s%s%s%s%s\"}",
            "{\"context\": \"%n%n%n%n%n\"}",
            "{\"context\": \"{0}{1}{2}\"}",
        };

        return new FuzzedMessage
        {
            Category = FuzzCategory.InjectionAttempt,
            Message = variations[_random.Next(variations.Length)],
            ExpectedBehavior = "Should not be vulnerable to injection"
        };
    }

    /// <summary>
    /// Tests a message handler with fuzzed inputs.
    /// </summary>
    /// <param name="handler">Function that processes a message and returns true if it handled it without crashing.</param>
    /// <param name="count">Number of fuzz tests to run.</param>
    /// <returns>Fuzz test results.</returns>
    public async Task<FuzzTestResults> RunFuzzTestAsync(
        Func<string, Task<FuzzHandlerResult>> handler,
        int count = 1000)
    {
        var results = new FuzzTestResults { TotalTests = count };
        var fuzzedMessages = GenerateFuzzedMessages(count).ToList();

        foreach (var fuzz in fuzzedMessages)
        {
            try
            {
                var result = await handler(fuzz.Message);

                if (result.Crashed)
                {
                    results.Crashes.Add(new FuzzCrash
                    {
                        Message = fuzz,
                        Exception = result.Exception?.ToString()
                    });
                }
                else if (result.WasAccepted && fuzz.Category != FuzzCategory.ValidButWeird)
                {
                    // Malformed input was accepted - potential vulnerability
                    results.FalseAccepts.Add(fuzz);
                }
                else
                {
                    results.PassedTests++;
                }
            }
            catch (Exception ex)
            {
                results.Crashes.Add(new FuzzCrash
                {
                    Message = fuzz,
                    Exception = ex.ToString()
                });
            }
        }

        return results;
    }
}

/// <summary>
/// Options for the protocol fuzzer.
/// </summary>
public class FuzzerOptions
{
    /// <summary>Maximum message size to generate.</summary>
    public int MaxMessageSize { get; set; } = 10 * 1024 * 1024; // 10 MB

    /// <summary>Maximum nesting depth to generate.</summary>
    public int MaxNestingDepth { get; set; } = 100;
}

/// <summary>
/// A fuzzed message with metadata.
/// </summary>
public class FuzzedMessage
{
    public FuzzCategory Category { get; set; }
    public string Message { get; set; } = string.Empty;
    public string ExpectedBehavior { get; set; } = string.Empty;
}

/// <summary>
/// Categories of fuzz testing.
/// </summary>
public enum FuzzCategory
{
    MalformedJson,
    MissingContext,
    InvalidContext,
    LengthBoundary,
    UnicodeEdgeCases,
    ControlCharacters,
    NullBytes,
    DeepNesting,
    MassivePayload,
    BinaryData,
    ValidButWeird,
    EmptyInput,
    NumericOverflow,
    BoundaryConditions,
    InjectionAttempt
}

/// <summary>
/// Result from a fuzz handler.
/// </summary>
public class FuzzHandlerResult
{
    public bool Crashed { get; set; }
    public bool WasAccepted { get; set; }
    public Exception? Exception { get; set; }

    public static FuzzHandlerResult Ok() => new() { WasAccepted = false };
    public static FuzzHandlerResult AcceptedResult() => new() { WasAccepted = true };
    public static FuzzHandlerResult Crash(Exception ex) => new() { Crashed = true, Exception = ex };
}

/// <summary>
/// Results from a fuzz test run.
/// </summary>
public class FuzzTestResults
{
    public int TotalTests { get; set; }
    public int PassedTests { get; set; }
    public List<FuzzCrash> Crashes { get; set; } = new();
    public List<FuzzedMessage> FalseAccepts { get; set; } = new();

    public bool Success => Crashes.Count == 0 && FalseAccepts.Count == 0;
    public double PassRate => TotalTests > 0 ? (double)PassedTests / TotalTests : 0;

    public override string ToString()
    {
        var sb = new StringBuilder();
        sb.AppendLine($"Fuzz Test Results: {PassedTests}/{TotalTests} passed ({PassRate:P1})");

        if (Crashes.Count > 0)
        {
            sb.AppendLine($"  CRASHES: {Crashes.Count}");
            foreach (var crash in Crashes.Take(5))
            {
                sb.AppendLine($"    - {crash.Message.Category}: {(crash.Exception != null ? crash.Exception[..Math.Min(100, crash.Exception.Length)] : "N/A")}...");
            }
        }

        if (FalseAccepts.Count > 0)
        {
            sb.AppendLine($"  FALSE ACCEPTS: {FalseAccepts.Count}");
            foreach (var fa in FalseAccepts.Take(5))
            {
                sb.AppendLine($"    - {fa.Category}");
            }
        }

        return sb.ToString();
    }
}

/// <summary>
/// Record of a crash during fuzzing.
/// </summary>
public class FuzzCrash
{
    public FuzzedMessage Message { get; set; } = new();
    public string? Exception { get; set; }
}
