namespace Mbrcpval.Testing;

/// <summary>
/// Represents the result of evaluating a single assertion.
/// </summary>
public record AssertionResult
{
    /// <summary>
    /// Gets or sets whether the assertion passed.
    /// </summary>
    public bool Passed { get; init; }

    /// <summary>
    /// Gets or sets the assertion that was evaluated.
    /// </summary>
    public Assertion Assertion { get; init; } = new();

    /// <summary>
    /// Gets or sets the actual value that was found.
    /// </summary>
    public object? ActualValue { get; init; }

    /// <summary>
    /// Gets or sets the expected value from the assertion.
    /// </summary>
    public object? ExpectedValue { get; init; }

    /// <summary>
    /// Gets or sets the error message if the assertion failed.
    /// </summary>
    public string? ErrorMessage { get; init; }

    /// <summary>
    /// Gets or sets the JSONPath that was evaluated.
    /// </summary>
    public string? EvaluatedPath { get; init; }

    /// <summary>
    /// Gets or sets the time taken to evaluate this assertion.
    /// </summary>
    public TimeSpan Duration { get; init; }

    /// <summary>
    /// Creates a successful assertion result.
    /// </summary>
    public static AssertionResult Success(Assertion assertion, object? actualValue = null) =>
        new()
        {
            Passed = true,
            Assertion = assertion,
            ActualValue = actualValue,
            ExpectedValue = assertion.Expected,
            EvaluatedPath = assertion.Path
        };

    /// <summary>
    /// Creates a failed assertion result.
    /// </summary>
    public static AssertionResult Failure(Assertion assertion, object? actualValue, string? errorMessage = null) =>
        new()
        {
            Passed = false,
            Assertion = assertion,
            ActualValue = actualValue,
            ExpectedValue = assertion.Expected,
            EvaluatedPath = assertion.Path,
            ErrorMessage = errorMessage ?? assertion.Message ?? GenerateDefaultErrorMessage(assertion, actualValue)
        };

    /// <summary>
    /// Creates an error result for assertion evaluation failures.
    /// </summary>
    public static AssertionResult Error(Assertion assertion, string errorMessage) =>
        new()
        {
            Passed = false,
            Assertion = assertion,
            ExpectedValue = assertion.Expected,
            EvaluatedPath = assertion.Path,
            ErrorMessage = errorMessage
        };

    private static string GenerateDefaultErrorMessage(Assertion assertion, object? actualValue)
    {
        var actualStr = FormatValue(actualValue);
        var expectedStr = FormatValue(assertion.Expected);

        return assertion.Type switch
        {
            AssertionType.Equals => $"Expected {assertion.Path} to equal {expectedStr}, but got {actualStr}",
            AssertionType.NotEquals => $"Expected {assertion.Path} to not equal {expectedStr}, but it did",
            AssertionType.Contains => $"Expected {assertion.Path} to contain {expectedStr}, but got {actualStr}",
            AssertionType.NotContains => $"Expected {assertion.Path} to not contain {expectedStr}, but it did",
            AssertionType.Matches => $"Expected {assertion.Path} to match pattern {expectedStr}, but got {actualStr}",
            AssertionType.Exists => $"Expected {assertion.Path} to exist, but it was not found",
            AssertionType.NotExists => $"Expected {assertion.Path} to not exist, but it was found with value {actualStr}",
            AssertionType.GreaterThan => $"Expected {assertion.Path} to be greater than {expectedStr}, but got {actualStr}",
            AssertionType.LessThan => $"Expected {assertion.Path} to be less than {expectedStr}, but got {actualStr}",
            AssertionType.GreaterOrEqual => $"Expected {assertion.Path} to be >= {expectedStr}, but got {actualStr}",
            AssertionType.LessOrEqual => $"Expected {assertion.Path} to be <= {expectedStr}, but got {actualStr}",
            AssertionType.IsType => $"Expected {assertion.Path} to be of type {expectedStr}, but got {actualValue?.GetType().Name ?? "null"}",
            AssertionType.SchemaValid => $"Value at {assertion.Path} did not conform to schema {expectedStr}",
            _ => $"Assertion failed: {assertion}"
        };
    }

    private static string FormatValue(object? value)
    {
        if (value == null) return "null";
        if (value is string s) return $"\"{s}\"";
        if (value is bool b) return b ? "true" : "false";
        return value.ToString() ?? "null";
    }

    public override string ToString()
    {
        if (Passed)
            return $"PASS: {Assertion}";
        return $"FAIL: {ErrorMessage}";
    }
}
