using System.Diagnostics;
using System.Text.RegularExpressions;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;

namespace Mbrcpval.Testing;

/// <summary>
/// Evaluates assertions against MBRC messages using JSONPath queries.
/// </summary>
public class AssertionEvaluator
{
    private readonly Core.SchemaValidator? _schemaValidator;

    /// <summary>
    /// Creates a new assertion evaluator without schema validation.
    /// </summary>
    public AssertionEvaluator()
    {
    }

    /// <summary>
    /// Creates a new assertion evaluator with schema validation support.
    /// </summary>
    /// <param name="schemaValidator">The schema validator to use for SchemaValid assertions.</param>
    public AssertionEvaluator(Core.SchemaValidator schemaValidator)
    {
        _schemaValidator = schemaValidator ?? throw new ArgumentNullException(nameof(schemaValidator));
    }

    /// <summary>
    /// Evaluates an assertion against a captured message.
    /// </summary>
    /// <param name="assertion">The assertion to evaluate.</param>
    /// <param name="message">The captured message to evaluate against.</param>
    /// <returns>The assertion result.</returns>
    public AssertionResult Evaluate(Assertion assertion, CapturedMessage message)
    {
        var stopwatch = Stopwatch.StartNew();

        try
        {
            // Convert message data to JToken for JSONPath queries
            var json = message.RawJson ?? JsonConvert.SerializeObject(new
            {
                context = message.Context,
                data = message.Data
            });

            var token = JToken.Parse(json);
            return Evaluate(assertion, token, stopwatch);
        }
        catch (Exception ex)
        {
            stopwatch.Stop();
            return new AssertionResult
            {
                Passed = false,
                Assertion = assertion,
                ErrorMessage = $"Failed to evaluate assertion: {ex.Message}",
                Duration = stopwatch.Elapsed
            };
        }
    }

    /// <summary>
    /// Evaluates an assertion against a JToken (parsed JSON).
    /// </summary>
    /// <param name="assertion">The assertion to evaluate.</param>
    /// <param name="token">The JSON token to evaluate against.</param>
    /// <param name="stopwatch">Optional stopwatch for timing.</param>
    /// <returns>The assertion result.</returns>
    public AssertionResult Evaluate(Assertion assertion, JToken token, Stopwatch? stopwatch = null)
    {
        stopwatch ??= Stopwatch.StartNew();

        try
        {
            // Handle Exists and NotExists first (they don't need a value comparison)
            if (assertion.Type == AssertionType.Exists)
            {
                var exists = EvaluateJsonPath(assertion.Path, token, out _);
                stopwatch.Stop();

                return exists
                    ? AssertionResult.Success(assertion) with { Duration = stopwatch.Elapsed }
                    : AssertionResult.Failure(assertion, null, $"Path '{assertion.Path}' does not exist") with { Duration = stopwatch.Elapsed };
            }

            if (assertion.Type == AssertionType.NotExists)
            {
                var exists = EvaluateJsonPath(assertion.Path, token, out var value);
                stopwatch.Stop();

                return !exists
                    ? AssertionResult.Success(assertion) with { Duration = stopwatch.Elapsed }
                    : AssertionResult.Failure(assertion, value, $"Path '{assertion.Path}' exists with value: {value}") with { Duration = stopwatch.Elapsed };
            }

            // For other assertion types, we need to extract the value
            if (!EvaluateJsonPath(assertion.Path, token, out var actualValue))
            {
                stopwatch.Stop();
                return AssertionResult.Failure(assertion, null,
                    $"Path '{assertion.Path}' not found in message") with { Duration = stopwatch.Elapsed };
            }

            // Compare values based on assertion type
            var (passed, errorMessage) = CompareValues(actualValue, assertion.Expected, assertion.Type);
            stopwatch.Stop();

            if (passed)
            {
                return AssertionResult.Success(assertion, actualValue) with { Duration = stopwatch.Elapsed };
            }

            return AssertionResult.Failure(assertion, actualValue, errorMessage ?? assertion.Message)
                with { Duration = stopwatch.Elapsed };
        }
        catch (Exception ex)
        {
            stopwatch.Stop();
            return AssertionResult.Error(assertion, $"Assertion evaluation error: {ex.Message}")
                with { Duration = stopwatch.Elapsed };
        }
    }

    /// <summary>
    /// Evaluates a JSONPath expression against JSON data.
    /// </summary>
    /// <param name="path">JSONPath expression (e.g., "$.data.playerstate").</param>
    /// <param name="token">The JSON token to query.</param>
    /// <param name="value">The extracted value, if found.</param>
    /// <returns>True if the path matched, false otherwise.</returns>
    public bool EvaluateJsonPath(string path, JToken token, out object? value)
    {
        value = null;

        if (string.IsNullOrEmpty(path))
        {
            value = ConvertJTokenToObject(token);
            return true;
        }

        try
        {
            var results = token.SelectTokens(path).ToList();

            if (results.Count == 0)
                return false;

            if (results.Count == 1)
            {
                value = ConvertJTokenToObject(results[0]);
            }
            else
            {
                value = results.Select(ConvertJTokenToObject).ToList();
            }

            return true;
        }
        catch (JsonException)
        {
            return false;
        }
    }

    /// <summary>
    /// Compares actual and expected values according to the assertion type.
    /// </summary>
    /// <param name="actual">The actual value from the message.</param>
    /// <param name="expected">The expected value from the assertion.</param>
    /// <param name="type">The type of comparison to perform.</param>
    /// <returns>A tuple of (passed, errorMessage).</returns>
    public (bool Passed, string? ErrorMessage) CompareValues(object? actual, object? expected, AssertionType type)
    {
        return type switch
        {
            AssertionType.Equals => CompareEquals(actual, expected),
            AssertionType.NotEquals => CompareNotEquals(actual, expected),
            AssertionType.Contains => CompareContains(actual, expected),
            AssertionType.NotContains => CompareNotContains(actual, expected),
            AssertionType.Matches => CompareMatches(actual, expected),
            AssertionType.GreaterThan => CompareNumeric(actual, expected, (a, e) => a > e, ">"),
            AssertionType.LessThan => CompareNumeric(actual, expected, (a, e) => a < e, "<"),
            AssertionType.GreaterOrEqual => CompareNumeric(actual, expected, (a, e) => a >= e, ">="),
            AssertionType.LessOrEqual => CompareNumeric(actual, expected, (a, e) => a <= e, "<="),
            AssertionType.IsType => CompareType(actual, expected),
            AssertionType.SchemaValid => CompareSchema(actual, expected),
            _ => (false, $"Unsupported assertion type: {type}")
        };
    }

    private (bool, string?) CompareEquals(object? actual, object? expected)
    {
        if (actual == null && expected == null)
            return (true, null);

        if (actual == null || expected == null)
            return (false, $"Expected {FormatValue(expected)}, but got {FormatValue(actual)}");

        // Handle JToken comparisons
        if (actual is JToken actualToken && expected is JToken expectedToken)
        {
            return JToken.DeepEquals(actualToken, expectedToken)
                ? (true, null)
                : (false, $"Expected {FormatValue(expected)}, but got {FormatValue(actual)}");
        }

        // Convert to comparable types
        var actualStr = actual.ToString();
        var expectedStr = expected.ToString();

        // Try numeric comparison first
        if (double.TryParse(actualStr, out var actualNum) && double.TryParse(expectedStr, out var expectedNum))
        {
            return Math.Abs(actualNum - expectedNum) < 0.0001
                ? (true, null)
                : (false, $"Expected {expectedNum}, but got {actualNum}");
        }

        // Try boolean comparison
        if (bool.TryParse(actualStr, out var actualBool) && bool.TryParse(expectedStr, out var expectedBool))
        {
            return actualBool == expectedBool
                ? (true, null)
                : (false, $"Expected {expectedBool}, but got {actualBool}");
        }

        // String comparison
        return string.Equals(actualStr, expectedStr, StringComparison.Ordinal)
            ? (true, null)
            : (false, $"Expected \"{expectedStr}\", but got \"{actualStr}\"");
    }

    private (bool, string?) CompareNotEquals(object? actual, object? expected)
    {
        var (equals, _) = CompareEquals(actual, expected);
        return !equals
            ? (true, null)
            : (false, $"Expected value to not equal {FormatValue(expected)}");
    }

    private (bool, string?) CompareContains(object? actual, object? expected)
    {
        if (actual == null)
            return (false, "Actual value is null");

        var actualStr = actual.ToString() ?? string.Empty;
        var expectedStr = expected?.ToString() ?? string.Empty;

        // Check if actual is an array/list
        if (actual is JArray actualArray)
        {
            var containsValue = actualArray.Any(item =>
            {
                var (eq, _) = CompareEquals(item, expected);
                return eq;
            });

            return containsValue
                ? (true, null)
                : (false, $"Array does not contain {FormatValue(expected)}");
        }

        if (actual is IEnumerable<object> actualList)
        {
            var containsValue = actualList.Any(item =>
            {
                var (eq, _) = CompareEquals(item, expected);
                return eq;
            });

            return containsValue
                ? (true, null)
                : (false, $"List does not contain {FormatValue(expected)}");
        }

        // String contains
        return actualStr.Contains(expectedStr, StringComparison.Ordinal)
            ? (true, null)
            : (false, $"String \"{actualStr}\" does not contain \"{expectedStr}\"");
    }

    private (bool, string?) CompareNotContains(object? actual, object? expected)
    {
        var (contains, _) = CompareContains(actual, expected);
        return !contains
            ? (true, null)
            : (false, $"Value should not contain {FormatValue(expected)}");
    }

    private (bool, string?) CompareMatches(object? actual, object? expected)
    {
        if (actual == null)
            return (false, "Actual value is null");

        var actualStr = actual.ToString() ?? string.Empty;
        var pattern = expected?.ToString() ?? string.Empty;

        try
        {
            var regex = new Regex(pattern, RegexOptions.None, TimeSpan.FromSeconds(1));
            return regex.IsMatch(actualStr)
                ? (true, null)
                : (false, $"String \"{actualStr}\" does not match pattern \"{pattern}\"");
        }
        catch (RegexParseException ex)
        {
            return (false, $"Invalid regex pattern \"{pattern}\": {ex.Message}");
        }
    }

    private (bool, string?) CompareNumeric(object? actual, object? expected,
        Func<double, double, bool> comparison, string operatorSymbol)
    {
        if (actual == null)
            return (false, "Actual value is null");

        if (!double.TryParse(actual.ToString(), out var actualNum))
            return (false, $"Actual value \"{actual}\" is not a number");

        if (!double.TryParse(expected?.ToString(), out var expectedNum))
            return (false, $"Expected value \"{expected}\" is not a number");

        return comparison(actualNum, expectedNum)
            ? (true, null)
            : (false, $"Expected {actualNum} {operatorSymbol} {expectedNum}");
    }

    private (bool, string?) CompareType(object? actual, object? expected)
    {
        var expectedType = expected?.ToString()?.ToLowerInvariant() ?? "null";

        var actualType = actual switch
        {
            null => "null",
            JValue jv => jv.Type switch
            {
                JTokenType.Null => "null",
                JTokenType.String => "string",
                JTokenType.Integer => "integer",
                JTokenType.Float => "number",
                JTokenType.Boolean => "boolean",
                JTokenType.Array => "array",
                JTokenType.Object => "object",
                _ => jv.Type.ToString().ToLowerInvariant()
            },
            JArray => "array",
            JObject => "object",
            string => "string",
            int or long => "integer",
            float or double or decimal => "number",
            bool => "boolean",
            IEnumerable<object> => "array",
            _ => actual.GetType().Name.ToLowerInvariant()
        };

        // Handle type aliases
        var normalizedExpected = expectedType switch
        {
            "int" => "integer",
            "num" or "float" or "double" => "number",
            "bool" => "boolean",
            "str" => "string",
            "obj" => "object",
            "arr" or "list" => "array",
            _ => expectedType
        };

        return string.Equals(actualType, normalizedExpected, StringComparison.OrdinalIgnoreCase)
            ? (true, null)
            : (false, $"Expected type {normalizedExpected}, but got {actualType}");
    }

    private (bool, string?) CompareSchema(object? actual, object? expected)
    {
        var schemaName = expected?.ToString() ?? "unknown";

        if (_schemaValidator == null)
        {
            return (false, "Schema validation not available. AssertionEvaluator was created without a SchemaValidator.");
        }

        if (actual == null)
        {
            return (false, "Cannot validate null value against schema");
        }

        try
        {
            var json = actual is JToken jt
                ? jt.ToString(Formatting.None)
                : JsonConvert.SerializeObject(actual);

            var result = _schemaValidator.ValidateJson(json, schemaName);

            if (result.IsValid)
            {
                return (true, null);
            }

            var errorMessages = string.Join("; ", result.Errors.Select(e => e.Message));
            return (false, $"Schema validation failed: {errorMessages}");
        }
        catch (Exception ex)
        {
            return (false, $"Schema validation error: {ex.Message}");
        }
    }

    private object? ConvertJTokenToObject(JToken? token)
    {
        return token?.Type switch
        {
            null => null,
            JTokenType.Null => null,
            JTokenType.String => token.Value<string>(),
            JTokenType.Integer => token.Value<long>(),
            JTokenType.Float => token.Value<double>(),
            JTokenType.Boolean => token.Value<bool>(),
            JTokenType.Array => token,
            JTokenType.Object => token,
            _ => token.ToString()
        };
    }

    private string FormatValue(object? value)
    {
        return value switch
        {
            null => "null",
            string s => $"\"{s}\"",
            bool b => b ? "true" : "false",
            JToken jt => jt.ToString(Formatting.None),
            _ => value.ToString() ?? "null"
        };
    }

    /// <summary>
    /// Evaluates multiple assertions against a message.
    /// </summary>
    /// <param name="assertions">The assertions to evaluate.</param>
    /// <param name="message">The message to evaluate against.</param>
    /// <returns>List of assertion results.</returns>
    public List<AssertionResult> EvaluateAll(IEnumerable<Assertion> assertions, CapturedMessage message)
    {
        return assertions.Select(a => Evaluate(a, message)).ToList();
    }

    /// <summary>
    /// Evaluates multiple assertions and returns true only if all pass.
    /// </summary>
    /// <param name="assertions">The assertions to evaluate.</param>
    /// <param name="message">The message to evaluate against.</param>
    /// <param name="results">The individual assertion results.</param>
    /// <returns>True if all assertions pass.</returns>
    public bool EvaluateAllPass(IEnumerable<Assertion> assertions, CapturedMessage message,
        out List<AssertionResult> results)
    {
        results = EvaluateAll(assertions, message);
        return results.All(r => r.Passed);
    }
}
