namespace Mbrcpval.Testing;

/// <summary>
/// Represents an assertion to be evaluated against a message or value.
/// </summary>
public class Assertion
{
    /// <summary>
    /// Gets or sets the type of assertion to perform.
    /// </summary>
    public AssertionType Type { get; set; } = AssertionType.Equals;

    /// <summary>
    /// Gets or sets the JSONPath expression to extract the value to assert against.
    /// Examples: "$.data.playerstate", "$.context", "$.data.tracks[0].artist"
    /// </summary>
    public string Path { get; set; } = string.Empty;

    /// <summary>
    /// Gets or sets the expected value for comparison.
    /// The type depends on the assertion type (string, number, bool, regex pattern, etc.)
    /// </summary>
    public object? Expected { get; set; }

    /// <summary>
    /// Gets or sets a custom failure message to display when the assertion fails.
    /// If not set, a default message will be generated.
    /// </summary>
    public string? Message { get; set; }

    /// <summary>
    /// Creates an Equals assertion.
    /// </summary>
    public static Assertion Equals(string path, object expected, string? message = null) =>
        new() { Type = AssertionType.Equals, Path = path, Expected = expected, Message = message };

    /// <summary>
    /// Creates an Exists assertion.
    /// </summary>
    public static Assertion Exists(string path, string? message = null) =>
        new() { Type = AssertionType.Exists, Path = path, Message = message };

    /// <summary>
    /// Creates a NotExists assertion.
    /// </summary>
    public static Assertion NotExists(string path, string? message = null) =>
        new() { Type = AssertionType.NotExists, Path = path, Message = message };

    /// <summary>
    /// Creates a Contains assertion.
    /// </summary>
    public static Assertion Contains(string path, object expected, string? message = null) =>
        new() { Type = AssertionType.Contains, Path = path, Expected = expected, Message = message };

    /// <summary>
    /// Creates a Matches assertion for regex patterns.
    /// </summary>
    public static Assertion Matches(string path, string pattern, string? message = null) =>
        new() { Type = AssertionType.Matches, Path = path, Expected = pattern, Message = message };

    /// <summary>
    /// Creates a GreaterThan assertion.
    /// </summary>
    public static Assertion GreaterThan(string path, object expected, string? message = null) =>
        new() { Type = AssertionType.GreaterThan, Path = path, Expected = expected, Message = message };

    /// <summary>
    /// Creates a LessThan assertion.
    /// </summary>
    public static Assertion LessThan(string path, object expected, string? message = null) =>
        new() { Type = AssertionType.LessThan, Path = path, Expected = expected, Message = message };

    /// <summary>
    /// Creates a SchemaValid assertion.
    /// </summary>
    public static Assertion SchemaValid(string schemaName, string? message = null) =>
        new() { Type = AssertionType.SchemaValid, Expected = schemaName, Message = message };

    public override string ToString()
    {
        var expectedStr = Expected?.ToString() ?? "null";
        if (expectedStr.Length > 50)
            expectedStr = expectedStr[..47] + "...";

        return Type switch
        {
            AssertionType.Exists => $"Assert {Path} exists",
            AssertionType.NotExists => $"Assert {Path} does not exist",
            AssertionType.SchemaValid => $"Assert schema valid: {expectedStr}",
            _ => $"Assert {Path} {Type} {expectedStr}"
        };
    }
}
