namespace Mbrcpval.Reporting;

using System.Text;
using Mbrcpval.Testing;
using Spectre.Console;

/// <summary>
/// Generates human-readable text reports with optional ANSI colors.
/// </summary>
public class TextReporter : IReporter
{
    private readonly bool _useColors;

    public TextReporter(bool useColors = true)
    {
        _useColors = useColors;
    }

    public string Format => "text";
    public string FileExtension => ".txt";

    public string GenerateReport(IEnumerable<TestResult> results, TestRunMetadata metadata)
    {
        var sb = new StringBuilder();
        var resultList = results.ToList();

        // Header
        sb.AppendLine("MBRCPVAL Protocol Validation Report");
        sb.AppendLine("====================================");
        sb.AppendLine($"Target: {metadata.Host}:{metadata.Port}");
        sb.AppendLine($"Protocol: v{metadata.ProtocolVersion}");
        sb.AppendLine($"Date: {metadata.StartTime:yyyy-MM-dd HH:mm:ss}");
        sb.AppendLine();

        // Group by category
        var grouped = resultList.GroupBy(r => r.TestCase.Category).OrderBy(g => g.Key);

        foreach (var group in grouped)
        {
            sb.AppendLine($"Test Suite: {group.Key}");
            sb.AppendLine(new string('-', 40));

            foreach (var result in group)
            {
                var status = FormatStatus(result.Status);
                sb.AppendLine($"{status} {result.TestCase.Id}: {result.TestCase.Name}");

                if (result.Status == TestStatus.Fail && !string.IsNullOrEmpty(result.ErrorMessage))
                {
                    sb.AppendLine($"       {result.ErrorMessage}");
                    if (result.ExpectedValue != null)
                        sb.AppendLine($"       Expected: {result.ExpectedValue}");
                    if (result.ActualValue != null)
                        sb.AppendLine($"       Actual: {result.ActualValue}");
                }
                else if (result.Status == TestStatus.Error && !string.IsNullOrEmpty(result.ErrorMessage))
                {
                    sb.AppendLine($"       Error: {result.ErrorMessage}");
                }
                else if (result.Status == TestStatus.Skip && !string.IsNullOrEmpty(result.ErrorMessage))
                {
                    sb.AppendLine($"       Reason: {result.ErrorMessage}");
                }
            }
            sb.AppendLine();
        }

        // Summary
        var passed = resultList.Count(r => r.Status == TestStatus.Pass);
        var failed = resultList.Count(r => r.Status == TestStatus.Fail);
        var skipped = resultList.Count(r => r.Status == TestStatus.Skip);
        var errors = resultList.Count(r => r.Status == TestStatus.Error);

        sb.AppendLine("Summary");
        sb.AppendLine("-------");
        sb.AppendLine($"Passed:  {passed}");
        sb.AppendLine($"Failed:  {failed}");
        sb.AppendLine($"Skipped: {skipped}");
        sb.AppendLine($"Errors:  {errors}");
        sb.AppendLine($"Total:   {resultList.Count}");
        sb.AppendLine($"Duration: {metadata.Duration.TotalSeconds:F2} seconds");

        return sb.ToString();
    }

    public async Task WriteReportAsync(IEnumerable<TestResult> results, TestRunMetadata metadata, string outputPath)
    {
        var report = GenerateReport(results, metadata);
        await File.WriteAllTextAsync(outputPath, report);
    }

    /// <summary>
    /// Writes the report directly to console with colors using Spectre.Console.
    /// </summary>
    public void WriteToConsole(IEnumerable<TestResult> results, TestRunMetadata metadata)
    {
        var resultList = results.ToList();

        // Header
        AnsiConsole.Write(new Rule("[bold blue]MBRCPVAL Protocol Validation Report[/]").RuleStyle("blue"));
        AnsiConsole.MarkupLine($"[dim]Target:[/] {metadata.Host}:{metadata.Port}");
        AnsiConsole.MarkupLine($"[dim]Protocol:[/] v{metadata.ProtocolVersion}");
        AnsiConsole.MarkupLine($"[dim]Date:[/] {metadata.StartTime:yyyy-MM-dd HH:mm:ss}");
        AnsiConsole.WriteLine();

        // Results table
        var table = new Table()
            .Border(TableBorder.Rounded)
            .AddColumn("Status")
            .AddColumn("ID")
            .AddColumn("Name")
            .AddColumn("Duration");

        foreach (var result in resultList)
        {
            var statusMarkup = result.Status switch
            {
                TestStatus.Pass => "[green]PASS[/]",
                TestStatus.Fail => "[red]FAIL[/]",
                TestStatus.Skip => "[yellow]SKIP[/]",
                TestStatus.Error => "[red bold]ERROR[/]",
                _ => result.Status.ToString()
            };

            table.AddRow(
                statusMarkup,
                result.TestCase.Id,
                Markup.Escape(result.TestCase.Name),
                $"{result.Duration.TotalMilliseconds:F0}ms"
            );
        }

        AnsiConsole.Write(table);
        AnsiConsole.WriteLine();

        // Summary
        var passed = resultList.Count(r => r.Status == TestStatus.Pass);
        var failed = resultList.Count(r => r.Status == TestStatus.Fail);
        var skipped = resultList.Count(r => r.Status == TestStatus.Skip);
        var errors = resultList.Count(r => r.Status == TestStatus.Error);

        var summaryTable = new Table()
            .Border(TableBorder.None)
            .HideHeaders()
            .AddColumn("")
            .AddColumn("");

        summaryTable.AddRow("[green]Passed[/]", passed.ToString());
        summaryTable.AddRow("[red]Failed[/]", failed.ToString());
        summaryTable.AddRow("[yellow]Skipped[/]", skipped.ToString());
        summaryTable.AddRow("[red bold]Errors[/]", errors.ToString());
        summaryTable.AddRow("[bold]Total[/]", resultList.Count.ToString());
        summaryTable.AddRow("[dim]Duration[/]", $"{metadata.Duration.TotalSeconds:F2}s");

        AnsiConsole.Write(new Panel(summaryTable).Header("Summary"));
    }

    private string FormatStatus(TestStatus status) => status switch
    {
        TestStatus.Pass => _useColors ? "\u001b[32m[PASS]\u001b[0m" : "[PASS]",
        TestStatus.Fail => _useColors ? "\u001b[31m[FAIL]\u001b[0m" : "[FAIL]",
        TestStatus.Skip => _useColors ? "\u001b[33m[SKIP]\u001b[0m" : "[SKIP]",
        TestStatus.Error => _useColors ? "\u001b[31;1m[ERROR]\u001b[0m" : "[ERROR]",
        _ => $"[{status}]"
    };
}
