namespace Mbrcpval.Reporting;

/// <summary>
/// Factory for creating report generators.
/// </summary>
public static class ReporterFactory
{
    /// <summary>
    /// Creates a reporter for the specified format.
    /// </summary>
    /// <param name="format">The report format (text, json, html, junit).</param>
    /// <param name="useColors">Whether to use ANSI colors for text output.</param>
    /// <returns>A reporter instance.</returns>
    /// <exception cref="ArgumentException">If the format is not recognized.</exception>
    public static IReporter Create(string format, bool useColors = true)
    {
        return format.ToLowerInvariant() switch
        {
            "text" or "txt" => new TextReporter(useColors),
            "json" => new JsonReporter(),
            "html" => new HtmlReporter(),
            "junit" or "xml" => new JUnitReporter(),
            _ => throw new ArgumentException($"Unknown report format: {format}. Supported formats: text, json, html, junit", nameof(format))
        };
    }

    /// <summary>
    /// Gets the list of supported report formats.
    /// </summary>
    public static IReadOnlyList<string> SupportedFormats => ["text", "json", "html", "junit"];

    /// <summary>
    /// Gets the default file extension for a format.
    /// </summary>
    public static string GetDefaultExtension(string format)
    {
        return format.ToLowerInvariant() switch
        {
            "text" or "txt" => ".txt",
            "json" => ".json",
            "html" => ".html",
            "junit" or "xml" => ".xml",
            _ => ".txt"
        };
    }
}
