namespace Mbrcpval.Reporting;

using System.Text.Json;
using System.Text.Json.Serialization;
using Mbrcpval.Testing;

/// <summary>
/// Generates machine-readable JSON reports.
/// </summary>
public class JsonReporter : IReporter
{
    private static readonly JsonSerializerOptions JsonOptions = new()
    {
        WriteIndented = true,
        PropertyNamingPolicy = JsonNamingPolicy.CamelCase,
        DefaultIgnoreCondition = JsonIgnoreCondition.WhenWritingNull,
        Converters = { new JsonStringEnumConverter(JsonNamingPolicy.CamelCase) }
    };

    public string Format => "json";
    public string FileExtension => ".json";

    public string GenerateReport(IEnumerable<TestResult> results, TestRunMetadata metadata)
    {
        var resultList = results.ToList();

        var report = new JsonReport
        {
            Target = $"{metadata.Host}:{metadata.Port}",
            Protocol = metadata.ProtocolVersion,
            Timestamp = metadata.StartTime,
            ToolVersion = metadata.ToolVersion,
            Results = resultList.Select(r => new JsonTestResult
            {
                Id = r.TestCase.Id,
                Name = r.TestCase.Name,
                Category = r.TestCase.Category,
                Status = r.Status.ToString().ToLowerInvariant(),
                DurationMs = (int)r.Duration.TotalMilliseconds,
                Error = r.Status == TestStatus.Pass ? null : r.ErrorMessage,
                Expected = r.ExpectedValue?.ToString(),
                Actual = r.ActualValue?.ToString(),
                FailedStep = r.FailedStepIndex
            }).ToList(),
            Summary = new JsonSummary
            {
                Passed = resultList.Count(r => r.Status == TestStatus.Pass),
                Failed = resultList.Count(r => r.Status == TestStatus.Fail),
                Skipped = resultList.Count(r => r.Status == TestStatus.Skip),
                Errors = resultList.Count(r => r.Status == TestStatus.Error),
                Total = resultList.Count,
                DurationMs = (int)metadata.Duration.TotalMilliseconds
            }
        };

        return JsonSerializer.Serialize(report, JsonOptions);
    }

    public async Task WriteReportAsync(IEnumerable<TestResult> results, TestRunMetadata metadata, string outputPath)
    {
        var report = GenerateReport(results, metadata);
        await File.WriteAllTextAsync(outputPath, report);
    }

    private record JsonReport
    {
        public required string Target { get; init; }
        public required string Protocol { get; init; }
        public required DateTimeOffset Timestamp { get; init; }
        public required string ToolVersion { get; init; }
        public required List<JsonTestResult> Results { get; init; }
        public required JsonSummary Summary { get; init; }
    }

    private record JsonTestResult
    {
        public required string Id { get; init; }
        public required string Name { get; init; }
        public required string Category { get; init; }
        public required string Status { get; init; }
        public required int DurationMs { get; init; }
        public string? Error { get; init; }
        public string? Expected { get; init; }
        public string? Actual { get; init; }
        public int? FailedStep { get; init; }
    }

    private record JsonSummary
    {
        public required int Passed { get; init; }
        public required int Failed { get; init; }
        public required int Skipped { get; init; }
        public required int Errors { get; init; }
        public required int Total { get; init; }
        public required int DurationMs { get; init; }
    }
}
