namespace Mbrcpval.Reporting;

using System.Text;
using System.Xml.Linq;
using Mbrcpval.Testing;

/// <summary>
/// Generates JUnit XML reports for CI/CD integration.
/// </summary>
public class JUnitReporter : IReporter
{
    public string Format => "junit";
    public string FileExtension => ".xml";

    public string GenerateReport(IEnumerable<TestResult> results, TestRunMetadata metadata)
    {
        var resultList = results.ToList();
        var grouped = resultList.GroupBy(r => r.TestCase.Category);

        var testSuites = new XElement("testsuites",
            new XAttribute("name", "MBRCPVAL"),
            new XAttribute("tests", resultList.Count),
            new XAttribute("failures", resultList.Count(r => r.Status == TestStatus.Fail)),
            new XAttribute("errors", resultList.Count(r => r.Status == TestStatus.Error)),
            new XAttribute("skipped", resultList.Count(r => r.Status == TestStatus.Skip)),
            new XAttribute("time", metadata.Duration.TotalSeconds.ToString("F3")),
            new XAttribute("timestamp", metadata.StartTime.ToString("yyyy-MM-ddTHH:mm:ss"))
        );

        foreach (var group in grouped)
        {
            var suiteResults = group.ToList();
            var testSuite = new XElement("testsuite",
                new XAttribute("name", group.Key),
                new XAttribute("tests", suiteResults.Count),
                new XAttribute("failures", suiteResults.Count(r => r.Status == TestStatus.Fail)),
                new XAttribute("errors", suiteResults.Count(r => r.Status == TestStatus.Error)),
                new XAttribute("skipped", suiteResults.Count(r => r.Status == TestStatus.Skip)),
                new XAttribute("time", suiteResults.Sum(r => r.Duration.TotalSeconds).ToString("F3"))
            );

            foreach (var result in suiteResults)
            {
                var testCase = new XElement("testcase",
                    new XAttribute("name", $"{result.TestCase.Id}: {result.TestCase.Name}"),
                    new XAttribute("classname", $"MBRCPVAL.{group.Key}"),
                    new XAttribute("time", result.Duration.TotalSeconds.ToString("F3"))
                );

                switch (result.Status)
                {
                    case TestStatus.Fail:
                        testCase.Add(new XElement("failure",
                            new XAttribute("message", result.ErrorMessage ?? "Assertion failed"),
                            new XAttribute("type", "AssertionError"),
                            FormatFailureContent(result)
                        ));
                        break;

                    case TestStatus.Error:
                        testCase.Add(new XElement("error",
                            new XAttribute("message", result.ErrorMessage ?? "Test error"),
                            new XAttribute("type", "TestError"),
                            result.ErrorMessage ?? ""
                        ));
                        break;

                    case TestStatus.Skip:
                        testCase.Add(new XElement("skipped",
                            new XAttribute("message", result.ErrorMessage ?? "Test skipped")
                        ));
                        break;
                }

                testSuite.Add(testCase);
            }

            testSuites.Add(testSuite);
        }

        var doc = new XDocument(
            new XDeclaration("1.0", "UTF-8", null),
            testSuites
        );

        var sb = new StringBuilder();
        using (var writer = new StringWriter(sb))
        {
            doc.Save(writer);
        }
        return sb.ToString();
    }

    public async Task WriteReportAsync(IEnumerable<TestResult> results, TestRunMetadata metadata, string outputPath)
    {
        var report = GenerateReport(results, metadata);
        await File.WriteAllTextAsync(outputPath, report);
    }

    private static string FormatFailureContent(TestResult result)
    {
        var sb = new StringBuilder();
        if (result.ExpectedValue != null)
            sb.AppendLine($"Expected: {result.ExpectedValue}");
        if (result.ActualValue != null)
            sb.AppendLine($"Actual: {result.ActualValue}");
        if (result.FailedStepIndex.HasValue)
            sb.AppendLine($"Failed at step: {result.FailedStepIndex.Value}");
        return sb.ToString();
    }
}
