namespace Mbrcpval.Reporting;

using Mbrcpval.Testing;

/// <summary>
/// Interface for test result reporters.
/// </summary>
public interface IReporter
{
    /// <summary>
    /// Gets the name of this reporter format.
    /// </summary>
    string Format { get; }

    /// <summary>
    /// Gets the default file extension for this format.
    /// </summary>
    string FileExtension { get; }

    /// <summary>
    /// Generates a report from test results.
    /// </summary>
    /// <param name="results">The test results to report.</param>
    /// <param name="metadata">Additional metadata about the test run.</param>
    /// <returns>The formatted report as a string.</returns>
    string GenerateReport(IEnumerable<TestResult> results, TestRunMetadata metadata);

    /// <summary>
    /// Writes the report to a file.
    /// </summary>
    /// <param name="results">The test results to report.</param>
    /// <param name="metadata">Additional metadata about the test run.</param>
    /// <param name="outputPath">The path to write the report to.</param>
    Task WriteReportAsync(IEnumerable<TestResult> results, TestRunMetadata metadata, string outputPath);
}

/// <summary>
/// Metadata about a test run.
/// </summary>
public record TestRunMetadata
{
    /// <summary>Target host that was tested.</summary>
    public required string Host { get; init; }

    /// <summary>Target port that was tested.</summary>
    public required int Port { get; init; }

    /// <summary>Protocol version used for testing.</summary>
    public required string ProtocolVersion { get; init; }

    /// <summary>When the test run started.</summary>
    public required DateTimeOffset StartTime { get; init; }

    /// <summary>When the test run ended.</summary>
    public required DateTimeOffset EndTime { get; init; }

    /// <summary>Total duration of the test run.</summary>
    public TimeSpan Duration => EndTime - StartTime;

    /// <summary>MBRCPVAL version.</summary>
    public string ToolVersion { get; init; } = "1.5.26.3";

    /// <summary>Test suites that were run.</summary>
    public IReadOnlyList<string> Suites { get; init; } = [];
}
