namespace Mbrcpval.Reporting;

using System.Text;
using System.Web;
using Mbrcpval.Testing;

/// <summary>
/// Generates self-contained HTML reports with interactive features.
/// </summary>
public class HtmlReporter : IReporter
{
    public string Format => "html";
    public string FileExtension => ".html";

    public string GenerateReport(IEnumerable<TestResult> results, TestRunMetadata metadata)
    {
        var resultList = results.ToList();
        var passed = resultList.Count(r => r.Status == TestStatus.Pass);
        var failed = resultList.Count(r => r.Status == TestStatus.Fail);
        var skipped = resultList.Count(r => r.Status == TestStatus.Skip);
        var errors = resultList.Count(r => r.Status == TestStatus.Error);

        var sb = new StringBuilder();
        sb.AppendLine("<!DOCTYPE html>");
        sb.AppendLine("<html lang=\"en\">");
        sb.AppendLine("<head>");
        sb.AppendLine("  <meta charset=\"UTF-8\">");
        sb.AppendLine("  <meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">");
        sb.AppendLine("  <title>MBRCPVAL Test Report</title>");
        sb.AppendLine("  <style>");
        sb.AppendLine(GetStyles());
        sb.AppendLine("  </style>");
        sb.AppendLine("</head>");
        sb.AppendLine("<body>");

        // Header
        sb.AppendLine("<header>");
        sb.AppendLine("  <h1>MBRCPVAL Protocol Validation Report</h1>");
        sb.AppendLine($"  <p class=\"meta\">Target: <strong>{HttpUtility.HtmlEncode(metadata.Host)}:{metadata.Port}</strong> | ");
        sb.AppendLine($"Protocol: <strong>v{HttpUtility.HtmlEncode(metadata.ProtocolVersion)}</strong> | ");
        sb.AppendLine($"Date: <strong>{metadata.StartTime:yyyy-MM-dd HH:mm:ss}</strong></p>");
        sb.AppendLine("</header>");

        // Summary cards
        sb.AppendLine("<div class=\"summary\">");
        sb.AppendLine($"  <div class=\"card pass\"><span class=\"count\">{passed}</span><span class=\"label\">Passed</span></div>");
        sb.AppendLine($"  <div class=\"card fail\"><span class=\"count\">{failed}</span><span class=\"label\">Failed</span></div>");
        sb.AppendLine($"  <div class=\"card skip\"><span class=\"count\">{skipped}</span><span class=\"label\">Skipped</span></div>");
        sb.AppendLine($"  <div class=\"card error\"><span class=\"count\">{errors}</span><span class=\"label\">Errors</span></div>");
        sb.AppendLine($"  <div class=\"card total\"><span class=\"count\">{metadata.Duration.TotalSeconds:F1}s</span><span class=\"label\">Duration</span></div>");
        sb.AppendLine("</div>");

        // Results by category
        var grouped = resultList.GroupBy(r => r.TestCase.Category).OrderBy(g => g.Key);
        foreach (var group in grouped)
        {
            var categoryPassed = group.Count(r => r.Status == TestStatus.Pass);
            var categoryTotal = group.Count();
            var categoryClass = categoryPassed == categoryTotal ? "category-pass" : "category-mixed";

            sb.AppendLine($"<details class=\"category {categoryClass}\" open>");
            sb.AppendLine($"  <summary><span class=\"category-name\">{HttpUtility.HtmlEncode(group.Key)}</span>");
            sb.AppendLine($"    <span class=\"category-stats\">{categoryPassed}/{categoryTotal} passed</span></summary>");
            sb.AppendLine("  <table>");
            sb.AppendLine("    <thead><tr><th>Status</th><th>ID</th><th>Name</th><th>Duration</th></tr></thead>");
            sb.AppendLine("    <tbody>");

            foreach (var result in group)
            {
                var statusClass = result.Status.ToString().ToLowerInvariant();
                var statusText = result.Status switch
                {
                    TestStatus.Pass => "PASS",
                    TestStatus.Fail => "FAIL",
                    TestStatus.Skip => "SKIP",
                    TestStatus.Error => "ERROR",
                    _ => result.Status.ToString()
                };

                sb.AppendLine($"      <tr class=\"{statusClass}\">");
                sb.AppendLine($"        <td><span class=\"status {statusClass}\">{statusText}</span></td>");
                sb.AppendLine($"        <td class=\"test-id\">{HttpUtility.HtmlEncode(result.TestCase.Id)}</td>");
                sb.AppendLine($"        <td class=\"test-name\">{HttpUtility.HtmlEncode(result.TestCase.Name)}");

                if (result.Status != TestStatus.Pass && !string.IsNullOrEmpty(result.ErrorMessage))
                {
                    sb.AppendLine($"          <div class=\"error-details\">{HttpUtility.HtmlEncode(result.ErrorMessage)}");
                    if (result.ExpectedValue != null)
                        sb.AppendLine($"<br><strong>Expected:</strong> {HttpUtility.HtmlEncode(result.ExpectedValue.ToString())}");
                    if (result.ActualValue != null)
                        sb.AppendLine($"<br><strong>Actual:</strong> {HttpUtility.HtmlEncode(result.ActualValue.ToString())}");
                    sb.AppendLine("          </div>");
                }

                sb.AppendLine("        </td>");
                sb.AppendLine($"        <td class=\"duration\">{result.Duration.TotalMilliseconds:F0}ms</td>");
                sb.AppendLine("      </tr>");
            }

            sb.AppendLine("    </tbody>");
            sb.AppendLine("  </table>");
            sb.AppendLine("</details>");
        }

        // Footer
        sb.AppendLine("<footer>");
        sb.AppendLine($"  <p>Generated by MBRCPVAL v{HttpUtility.HtmlEncode(metadata.ToolVersion)}</p>");
        sb.AppendLine("</footer>");

        sb.AppendLine("</body>");
        sb.AppendLine("</html>");

        return sb.ToString();
    }

    public async Task WriteReportAsync(IEnumerable<TestResult> results, TestRunMetadata metadata, string outputPath)
    {
        var report = GenerateReport(results, metadata);
        await File.WriteAllTextAsync(outputPath, report);
    }

    private static string GetStyles() => """
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; background: #0f172a; color: #e2e8f0; line-height: 1.6; padding: 20px; }
        header { text-align: center; margin-bottom: 30px; }
        h1 { color: #60a5fa; margin-bottom: 10px; }
        .meta { color: #94a3b8; }
        .summary { display: flex; gap: 15px; justify-content: center; flex-wrap: wrap; margin-bottom: 30px; }
        .card { background: #1e293b; border-radius: 12px; padding: 20px 30px; text-align: center; min-width: 120px; }
        .card .count { display: block; font-size: 2.5em; font-weight: bold; }
        .card .label { color: #94a3b8; font-size: 0.9em; }
        .card.pass .count { color: #22c55e; }
        .card.fail .count { color: #ef4444; }
        .card.skip .count { color: #eab308; }
        .card.error .count { color: #f97316; }
        .card.total .count { color: #60a5fa; }
        .category { background: #1e293b; border-radius: 12px; margin-bottom: 15px; overflow: hidden; }
        .category summary { padding: 15px 20px; cursor: pointer; display: flex; justify-content: space-between; align-items: center; background: #334155; }
        .category summary:hover { background: #475569; }
        .category-name { font-weight: 600; font-size: 1.1em; }
        .category-stats { color: #94a3b8; }
        table { width: 100%; border-collapse: collapse; }
        th { text-align: left; padding: 12px 15px; background: #1e293b; color: #94a3b8; font-weight: 500; }
        td { padding: 12px 15px; border-top: 1px solid #334155; }
        tr:hover { background: #334155; }
        .status { display: inline-block; padding: 2px 8px; border-radius: 4px; font-size: 0.8em; font-weight: 600; }
        .status.pass { background: #166534; color: #bbf7d0; }
        .status.fail { background: #991b1b; color: #fecaca; }
        .status.skip { background: #854d0e; color: #fef08a; }
        .status.error { background: #9a3412; color: #fed7aa; }
        .test-id { font-family: monospace; color: #60a5fa; }
        .duration { font-family: monospace; color: #94a3b8; }
        .error-details { margin-top: 8px; padding: 10px; background: #450a0a; border-radius: 6px; font-size: 0.9em; color: #fca5a5; }
        footer { text-align: center; margin-top: 30px; color: #64748b; font-size: 0.9em; }
        @media (prefers-color-scheme: light) {
            body { background: #f1f5f9; color: #1e293b; }
            .card, .category { background: #fff; box-shadow: 0 1px 3px rgba(0,0,0,0.1); }
            .category summary { background: #f8fafc; }
            .category summary:hover { background: #f1f5f9; }
            th { background: #f8fafc; }
            td { border-color: #e2e8f0; }
            tr:hover { background: #f8fafc; }
            .error-details { background: #fef2f2; color: #991b1b; }
        }
        """;
}
